/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.comments;

import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;

import org.ametys.cms.repository.CommentableAmetysObject;
import org.ametys.cms.repository.comment.contributor.AbstractCommentNotifyMentionsObserver;
import org.ametys.core.observation.Event;
import org.ametys.core.right.RightManager;
import org.ametys.core.ui.mail.StandardMailBodyHelper;
import org.ametys.core.ui.mail.StandardMailBodyHelper.MailBodyBuilder;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.workspaces.WorkspacesHelper;
import org.ametys.plugins.workspaces.documents.DocumentWorkspaceModule;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModuleExtensionPoint;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.project.rights.ProjectRightHelper;
import org.ametys.plugins.workspaces.tasks.TasksWorkspaceModule;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.WebConstants;
import org.ametys.web.renderingcontext.RenderingContext;
import org.ametys.web.renderingcontext.RenderingContextHandler;

/**
 * Observer to send mails to mentioned users in contributor comments of workspaces objects
 * @param <T> type of the {@link CommentableAmetysObject}
 */
public abstract class AbstractNotifyWorkspacesCommentMentionsObserver<T extends CommentableAmetysObject> extends AbstractCommentNotifyMentionsObserver<T> implements Contextualizable
{
    /** The right manager */
    protected RightManager _rightManager;

    /** The project manager */
    protected ProjectManager _projectManager;

    /** The task module */
    protected TasksWorkspaceModule _taskModule;
    
    /** The document module */
    protected DocumentWorkspaceModule _documentModule;

    /** The Workspaces helper */
    protected WorkspacesHelper _workspaceHelper;
    
    /** The context */
    protected Context _context;
    
    /** The project rights helper */
    protected ProjectRightHelper _projectRightsHelper;

    /** The rendering context handler */
    protected RenderingContextHandler _renderingContextHandler;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _projectManager = (ProjectManager) manager.lookup(ProjectManager.ROLE);
        WorkspaceModuleExtensionPoint moduleManagerEP = (WorkspaceModuleExtensionPoint) manager.lookup(WorkspaceModuleExtensionPoint.ROLE);
        _taskModule = moduleManagerEP.getModule(TasksWorkspaceModule.TASK_MODULE_ID);
        _documentModule = moduleManagerEP.getModule(DocumentWorkspaceModule.DOCUMENT_MODULE_ID);
        _workspaceHelper = (WorkspacesHelper) manager.lookup(WorkspacesHelper.ROLE);
        _projectRightsHelper = (ProjectRightHelper) manager.lookup(ProjectRightHelper.ROLE);
        _renderingContextHandler = (RenderingContextHandler) manager.lookup(RenderingContextHandler.ROLE);
    }
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }

    @Override
    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        Map<String, Object> arguments = event.getArguments();
        
        AmetysObject ametysObject = _getAmetysObjectFromArguments(arguments);
        Project project = _projectManager.getParentProject(ametysObject);
        Request request = ContextHelper.getRequest(_context);
        request.setAttribute(WebConstants.REQUEST_ATTR_SITE_NAME, project.getName());
        super.observe(event, transientVars);
    }
    
    /**
     * Get the language of the project containing the given object
     * @param project the project
     * @return the language code
     */
    protected String _getLanguage(Project project)
    {
        return _workspaceHelper.getLang(project);
    }
    
    /**
     * Get the Ametys object from the event arguments
     * @param arguments the event arguments
     * @return the Ametys object
     */
    protected T _getAmetysObjectFromArguments(Map<String, Object> arguments)
    {
        String ametysObjectId = (String) arguments.get(org.ametys.plugins.explorer.ObservationConstants.ARGS_ID);
        T ametysObject = _resolver.resolveById(ametysObjectId);
        return ametysObject;
    }

    @Override
    protected MailBodyBuilder getStandardMailBodyHelper()
    {
        return StandardMailBodyHelper.newHTMLBody();
    }

    @Override
    protected I18nizableText _getMailSubject(MentionableObject mentionableObject)
    {
        Project project = _projectManager.getParentProject(mentionableObject.ametysObject());
        List<String> i18nParams = List.of(project.getTitle());
        return new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_WORKSPACES_MENTION_MAIL_TITLE", i18nParams);
    }
    
    @Override
    protected I18nizableText _getMailTitle(MentionableObject mentionableObject)
    {
        AmetysObject ametysObject = mentionableObject.ametysObject();
        Project project = _projectManager.getParentProject(ametysObject);

        List<String> i18nParams = List.of(project.getTitle());
        return new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_WORKSPACES_MENTION_MAIL_SUBJECT", i18nParams);
    }
    
    @Override
    protected boolean _canSendMailToMentionedUser(T ametysObject, User authorIdentity, UserIdentity mentionedUserIdentity)
    {
        Project project = _projectManager.getParentProject(ametysObject);
        
        return super._canSendMailToMentionedUser(ametysObject, authorIdentity, mentionedUserIdentity)
            && _projectRightsHelper.hasReadAccessOnModule(project, _getModuleId(), mentionedUserIdentity);
    }

    /**
     * Get the module id to check read access
     * @return the module id
     */
    protected abstract String _getModuleId();

    
    /**
     * Get the absolute URL of the given Ametys object in the context of the given project
     * @param ametysObject the Ametys object
     * @param project the project containing the Ametys object
     * @return the absolute URL
     */
    protected String _getAbsoluteUrl(T ametysObject, Project project)
    {
        Request request = ContextHelper.getRequest(_context);
        request.setAttribute("forceAbsoluteUrl", true);
        
        RenderingContext currentContext = _renderingContextHandler.getRenderingContext();
        _renderingContextHandler.setRenderingContext(RenderingContext.FRONT);
        
        try
        {
            return getUrl(ametysObject, project);
        }
        finally
        {
            _renderingContextHandler.setRenderingContext(currentContext);
        }
    }

    /**
     * Get the URL of the given Ametys object in the context of the given project
     * @param ametysObject the Ametys object
     * @param project the project containing the Ametys object
     * @return the URL (may be relative or absolute depending on the implementation)
     */
    protected abstract String getUrl(T ametysObject, Project project);
    
}
