/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.comments;

import java.util.List;
import java.util.Map;

import org.ametys.cms.repository.comment.Comment;
import org.ametys.cms.repository.mentions.MentionUtils;
import org.ametys.core.observation.Event;
import org.ametys.core.user.User;
import org.ametys.plugins.workspaces.ObservationConstants;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.tasks.Task;
import org.ametys.plugins.workspaces.tasks.TasksWorkspaceModule;
import org.ametys.runtime.i18n.I18nizableText;
/**
 * Observer to send mails to mentioned users in contributor comments of {@link Task}
 */
public class NotifyTaskCommentMentionsObserver extends AbstractNotifyWorkspacesCommentMentionsObserver<Task>
{
    @Override
    public boolean supports(Event event)
    {
        return event.getId().equals(ObservationConstants.EVENT_TASK_COMMENTED);
    }
    
    @Override
    protected String _getAmetysObjectTitle(Task task)
    {
        return task.getLabel();
    }

    @Override
    protected MentionableObject _getMentionableObjectFromArguments(Map<String, Object> arguments) throws Exception
    {
        Task task = _getAmetysObjectFromArguments(arguments);
        String commentId = (String) arguments.get(ObservationConstants.ARGS_TASK_COMMENT_ID);
        Comment comment = task.getComment(commentId);
        Project project = _projectManager.getParentProject(task);
        
        String url = _getAbsoluteUrl(task, project);

        return new MentionableObject(
            _userManager.getUser(comment.getAuthor()),
            comment.getContent(),
            MentionUtils.extractMentionedUsersFromSimpleText(comment.getContent()),
            comment.getCreationDate(),
            task,
            new LinkToAmetysObject(url, new I18nizableText("plugin.workspaces", "PROJECT_MAIL_NOTIFICATION_BODY_TASK_BUTTON_TEXT")),
            _getLanguage(project)
        );
    }

    @Override
    protected I18nizableText _getMailMessage(MentionableObject mentionableObject)
    {
        Task task = (Task) mentionableObject.ametysObject();
        User author = mentionableObject.author();
        Project project = _projectManager.getParentProject(task);
        List<String> i18nParams = List.of(author.getFullName(), _getAmetysObjectTitle(task), project.getTitle());
        return new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_TASK_MENTION_MAIL_MESSAGE", i18nParams);
    }

    @Override
    protected String _getModuleId()
    {
        return TasksWorkspaceModule.TASK_MODULE_ID;
    }

    @Override
    protected String getUrl(Task task, Project project)
    {
        return _taskModule.getTaskUri(project, task.getId());
    }
}
