/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.documents.jcr;

import java.time.ZonedDateTime;
import java.util.List;

import javax.jcr.Node;
import javax.jcr.RepositoryException;

import org.ametys.cms.repository.CommentableAmetysObject;
import org.ametys.cms.repository.comment.Comment;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.explorer.resources.jcr.JCRResource;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.data.holder.ModifiableModelLessDataHolder;
import org.ametys.plugins.repository.data.holder.impl.DefaultModifiableModelLessDataHolder;
import org.ametys.plugins.repository.data.repositorydata.ModifiableRepositoryData;
import org.ametys.plugins.repository.data.repositorydata.impl.JCRRepositoryData;

/**
 * Implementation of {@link Resource} used for document module of workspaces and can handle comments, backed by a JCR node.<br>
 */
public class File extends JCRResource<FileFactory> implements CommentableAmetysObject<Comment>
{

    /**
     * Creates an {@link File}.
     * @param node the node backing this {@link AmetysObject}
     * @param parentPath the parentPath in the Ametys hierarchy
     * @param factory the DefaultAmetysObjectFactory which created the AmetysObject
     */
    public File(Node node, String parentPath, FileFactory factory)
    {
        super(node, parentPath, factory);
    }

    /**
     * Get the file description
     * @return the description
     */
    public String getDescription() 
    {
        try
        {
            Node fileNode = getNode();
            return fileNode.hasProperty("ametys:description")
                ? fileNode.getProperty("ametys:description").getString()
                : null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Cannot get description for file " + this.getName() + " (" + this.getId() + ")", e);
        }
    }
    
    /**
     * Set the file description
     * @param description the description to set
     */
    public void setDescription (String description) 
    {
        try
        {
            Node fileNode = getNode();
            fileNode.setProperty("ametys:description", description);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Cannot set description for file " + this.getName() + " (" + this.getId() + ")", e);
        }
    } 
    /**
     * <code>true</code> if the description was automatically generated, <code>false</code> otherwise.
     * @return <code>true</code> if the description was automatically generated, <code>false</code> otherwise.
     */
    public boolean isAutomaticallyGeneratedDescription()
    {
        try
        {
            Node fileNode = getNode();
            if (fileNode.hasProperty("ametys:isAutoGeneratedDescription"))
            {
                return fileNode.getProperty("ametys:isAutoGeneratedDescription").getBoolean();
            }
            else
            {
                return false;
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Cannot get automatic description status for file " + this.getName() + " (" + this.getId() + ")", e);
        }
    }
    /**
     * Set to <code>true</code> if the description was automatically generated, <code>false</code> otherwise.
     * @param isAutoGeneratedDescription <code>true</code> if the description was automatically generated
     */
    public void setIsAutoGeneratedDescription(boolean isAutoGeneratedDescription)
    {
        try
        {
            Node fileNode = getNode();
            fileNode.setProperty("ametys:isAutoGeneratedDescription", isAutoGeneratedDescription);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Cannot set automatic description status for file " + this.getName() + " (" + this.getId() + ")", e);
        }
    }
    
    public Comment createComment()
    {
        return new Comment(_getCommentsDataHolder());
    }
    
    public Comment createComment(String commentId, ZonedDateTime creationDate)
    {
        return new Comment(_getCommentsDataHolder(), commentId, creationDate);
    }
    
    public Comment getComment(String commentId) throws AmetysRepositoryException
    {
        return Comment.getComment(_getCommentsDataHolder(), commentId);
    }
    
    public List<Comment> getComments(boolean includeNotValidatedComments, boolean includeValidatedComments) throws AmetysRepositoryException
    {
        return Comment.getComments(_getCommentsDataHolder(), includeNotValidatedComments, includeValidatedComments);
    }

    private ModifiableModelLessDataHolder _getCommentsDataHolder()
    {
        try
        {
            Node baseNode = getBaseNode();
            if (!baseNode.hasNode("ametys:comments"))
            {
                baseNode.addNode("ametys:comments", "ametys:compositeMetadata");
            }
            Node commentsNode = baseNode.getNode("ametys:comments");
            
            baseNode.getSession().save();
            
            ModifiableRepositoryData repositoryData = new JCRRepositoryData(commentsNode);
            return new DefaultModifiableModelLessDataHolder(_getFactory().getUnversionedDataTypeExtensionPoint(), repositoryData);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }
}
