/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.documents.observers;

import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.observation.AsyncObserver;
import org.ametys.core.observation.Event;
import org.ametys.plugins.explorer.ObservationConstants;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.workspaces.WorkspacesHelper;
import org.ametys.plugins.workspaces.WorkspacesHelper.FileType;
import org.ametys.plugins.workspaces.documents.WorkspaceFileAIHelper;
import org.ametys.plugins.workspaces.documents.jcr.File;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * This observer listens to document creation or update events and automatically
 * generates a summary using an AI provider, storing it in the resource's metadata.
 */
public class FileAISummaryObserver extends AbstractLogEnabled implements AsyncObserver, Serviceable
{
    
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
        
    private WorkspacesHelper _workspaceHelper;
    
    private WorkspaceFileAIHelper _workspaceFileAIHelper;

    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _workspaceHelper = (WorkspacesHelper) manager.lookup(WorkspacesHelper.ROLE);
        _workspaceFileAIHelper = (WorkspaceFileAIHelper) manager.lookup(WorkspaceFileAIHelper.ROLE);
    }
    
    public int getPriority()
    {
        return MAX_PRIORITY;
    }

    public boolean supports(Event event)
    {
        boolean isEnabled = _workspaceFileAIHelper.isAutomaticGenerationEnabled();
        return isEnabled
            && (
                event.getId().equals(ObservationConstants.EVENT_RESOURCE_CREATED)
             || event.getId().equals(ObservationConstants.EVENT_RESOURCE_UPDATED)
            );
    }

    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        Map<String, Object> args = event.getArguments();

        if (event.getId().equals(ObservationConstants.EVENT_RESOURCE_CREATED))
        {
            @SuppressWarnings("unchecked")
            Map<String, Resource> resources = (Map<String, Resource>) args.get(ObservationConstants.ARGS_RESOURCES);
            for (Resource resource : resources.values())
            {
                // Resolve again the resource because JCR session can be lost
                Resource addedResource = (Resource) _resolver.resolveById(resource.getId());
                _generateSummaryForResource(addedResource);
            }
        }
        else if (event.getId().equals(ObservationConstants.EVENT_RESOURCE_UPDATED))
        {
            String resourceId = (String) args.get(ObservationConstants.ARGS_ID);
            Resource resource = (Resource) _resolver.resolveById(resourceId);
            _generateSummaryForResource(resource);
        }
    }
    
    private void _generateSummaryForResource(Resource resource) throws Exception
    {
        FileType fileType =  _workspaceHelper.getFileType(resource);
        
        if ((fileType == FileType.TEXT || fileType == FileType.PDF)
                && resource instanceof File file
                && (file.isAutomaticallyGeneratedDescription() || StringUtils.isBlank(file.getDescription()))) // Only update files with automatic description or empty ones
        {
            _workspaceFileAIHelper.generateFileDescription(file);
        }
    }
}
