/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.forum;

import java.time.ZonedDateTime;
import java.util.Date;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.repository.comment.RichTextComment;
import org.ametys.core.ui.Callable;
import org.ametys.core.userpref.UserPreferencesException;
import org.ametys.core.userpref.UserPreferencesManager;
import org.ametys.core.util.DateUtils;

/**
 * DAO for thread's user prefrerences
 */
public class WorkspaceThreadUserPreferencesDAO extends AbstractWorkspaceThreadDAO
{
    /** Avalon Role */
    public static final String ROLE = WorkspaceThreadUserPreferencesDAO.class.getName();

    /** the user preferences context for activity stream */
    public static final String LAST_THREAD_READ_USER_PREF_CONTEXT = "/workspaces/forum";

    /** The user preferences */
    protected UserPreferencesManager _userPrefsManager;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _userPrefsManager = (UserPreferencesManager) manager.lookup(UserPreferencesManager.ROLE);
    }

    /**
     * Update the user prefs of current user, so we can know that he has opened the thread
     * @param threadId the thread id
     */
    @Callable (rights = Callable.NO_CHECK_REQUIRED)
    public void clearUnopenedThreadNotification(String threadId)
    {
        _setDate(threadId, DateUtils.dateToString(new Date()));
    }
    
    /**
     * Update the user prefs of current user 
     * @param threadId the thread id
     * @param commentId last seen comment id
     */
    @Callable (rights = Callable.NO_CHECK_REQUIRED)
    public void clearUnreadCommentsNotification(String threadId, String commentId)
    {
        Thread thread = _resolver.resolveById(threadId);
        RichTextComment  comment = thread.getComment(commentId);
        _setDate(threadId, comment.getCreationDate().toString());
    }

    private void _setDate(String threadId, String dateAsString)
    {
        try
        {
            _userPrefsManager.addUserPreference(_currentUserProvider.getUser(), LAST_THREAD_READ_USER_PREF_CONTEXT, Map.of(), threadId, dateAsString);
        }
        catch (UserPreferencesException e)
        {
            getLogger().error("An error occured while setting the user preferences.", e);
        }
    }

    /**
     * Check if the thread has been currently read by the current user since the last contribution (creation or new comment)
     * @param thread the thread
     * @return true if the thread has been currently read by the current user
     */
    public boolean hasNotification(Thread thread)
    {
        ZonedDateTime lastReadDate = getLastReadDate(thread);
        return lastReadDate == null || lastReadDate.isBefore(thread.getLastContribution());
    }
    
    /**
     * Check if the thread has been opened by the current user
     * @param thread the thread
     * @return true if the thread has been opened by the current user
     */
    public boolean hasOpenedThread(Thread thread)
    {
        ZonedDateTime lastReadDate = getLastReadDate(thread);
        return lastReadDate != null && lastReadDate.isAfter(thread.getCreationDate());
    }

    /**
     * Get how many unread comments a user who has already opened a thread has
     * @param thread the thread
     * @return the number of unread comments
     */
    public long getUnreadCommentNumber(Thread thread)
    {
        ZonedDateTime lastReadDate = getLastReadDate(thread);

        if (lastReadDate != null)
        {
            return thread.getFlattenComments(true, true).stream()
            .filter(comment -> comment.getCreationDate().isAfter(lastReadDate))
            .count();
        }
        else
        {
            return 0;
        }
    }

    /**
     * Get the last read date of a thread for the current user
     * @param thread the thread
     * @return the last read date of a thread for the current user
     */
    public ZonedDateTime getLastReadDate(Thread thread)
    {
        ZonedDateTime lastReadDate = null;
        try
        {
            lastReadDate = _userPrefsManager.getUserPreferenceAsDate(_currentUserProvider.getUser(), LAST_THREAD_READ_USER_PREF_CONTEXT, Map.of(), thread.getId());
        }
        catch (UserPreferencesException e)
        {
            getLogger().error("An error occured while getting the user preferences.", e);
        }
        
        return lastReadDate;
    }
}
