/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.forum.datapolicy;

import javax.jcr.Node;
import javax.jcr.RepositoryException;

import org.apache.commons.lang3.tuple.Pair;

import org.ametys.cms.repository.comment.RichTextComment;
import org.ametys.core.user.population.UserPopulationDAO;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.workspaces.datapolicy.AbstractCommentAndReactionDataPolicy;
import org.ametys.plugins.workspaces.forum.jcr.JCRThread;
import org.ametys.plugins.workspaces.forum.jcr.JCRThreadFactory;

/**
 * Data policy that anonymize thread's comments and delete reactions made by an unknown user
 */
public class ThreadCommentAndReactionDataPolicy extends AbstractCommentAndReactionDataPolicy
{
    @Override
    protected String getLogCategory()
    {
        return "thread";
    }
    
    @Override
    protected boolean handleComment(Node commentNode)
    {
        try
        {
            Pair<Node, String> holderAndCommentId = getObjectNodeAndCommentId(commentNode);
            if (holderAndCommentId != null)
            {
                JCRThread thread = _resolver.resolve(holderAndCommentId.getLeft(), false);
                RichTextComment comment = thread.getComment(holderAndCommentId.getRight());
                
                comment.setAuthor(UserPopulationDAO.UNKNOWN_USER_IDENTITY);
                comment.setAuthorEmail(null);
                comment.setAuthorName(null);
                thread.saveChanges();
                
                // No specific observation event  at this point
                
                return true;
            }
        }
        catch (RepositoryException | AmetysRepositoryException e)
        {
            getLogger().error("Failed to retrieve thread and comment from comment node '{}'", commentNode,  e);
        }
        
        return false;
    }
    
    @Override
    protected String getObjectPrimaryType()
    {
        return JCRThreadFactory.THREAD_NODETYPE;
    }
}
