/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.forum.jcr;

import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.function.Predicate;

import javax.jcr.Node;
import javax.jcr.RepositoryException;

import org.apache.commons.lang3.EnumUtils;

import org.ametys.cms.data.Binary;
import org.ametys.cms.data.RichText;
import org.ametys.cms.data.holder.ModifiableIndexableDataHolder;
import org.ametys.cms.data.holder.impl.DefaultModifiableModelAwareDataHolder;
import org.ametys.cms.repository.comment.RichTextComment;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.data.holder.ModifiableModelLessDataHolder;
import org.ametys.plugins.repository.data.holder.impl.DefaultModifiableModelLessDataHolder;
import org.ametys.plugins.repository.data.repositorydata.ModifiableRepositoryData;
import org.ametys.plugins.repository.data.repositorydata.impl.JCRRepositoryData;
import org.ametys.plugins.repository.jcr.DefaultAmetysObject;
import org.ametys.plugins.repository.tag.TaggableAmetysObjectHelper;
import org.ametys.plugins.workspaces.forum.Thread;
import org.ametys.plugins.workspaces.forum.ThreadCategoryEnum;

/**
 * JCR implementation of a Thread
 */
public class JCRThread extends DefaultAmetysObject<JCRThreadFactory> implements Thread
{
    /**
     * Default constructor for the JCRThread
     * @param node The thread node
     * @param parentPath The parent path
     * @param factory The factory
     */
    public JCRThread(Node node, String parentPath, JCRThreadFactory factory)
    {
        super(node, parentPath, factory);
    }

    public String getTitle()
    {
        return getValue(ATTRIBUTE_TITLE);
    }

    public void setTitle(String title)
    {
        setValue(ATTRIBUTE_TITLE, title);
    }
    
    public RichText getContent()
    {
        return getValue(ATTRIBUTE_CONTENT);
    }

    public void setContent(RichText richText)
    {
        setValue(ATTRIBUTE_CONTENT, richText);
    }
    
    public UserIdentity getAuthor()
    {
        return getValue(ATTRIBUTE_AUTHOR);
    }
    
    public void setAuthor(UserIdentity author)
    {
        setValue(ATTRIBUTE_AUTHOR, author);
    }
    
    public ZonedDateTime getCreationDate()
    {
        return getValue(ATTRIBUTE_CREATIONDATE);
    }
    
    public void setCreationDate(ZonedDateTime creationDate)
    {
        setValue(ATTRIBUTE_CREATIONDATE, creationDate);
    }

    public ZonedDateTime getLastModified()
    {
        return getValue(ATTRIBUTE_LASTMODIFIED);
    }

    public void setLastModified(ZonedDateTime date)
    {
        setValue(ATTRIBUTE_LASTMODIFIED, date);
    }

    public ZonedDateTime getLastContribution()
    {
        return getValue(ATTRIBUTE_LASTCONTRIBUTION);
    }

    public void setLastContribution(ZonedDateTime date)
    {
        setValue(ATTRIBUTE_LASTCONTRIBUTION, date);
    }
    
    public void tag(String tag) throws AmetysRepositoryException
    {
        TaggableAmetysObjectHelper.tag(this, tag);
    }

    public void untag(String tag) throws AmetysRepositoryException
    {
        TaggableAmetysObjectHelper.untag(this, tag);
    }

    public Set<String> getTags() throws AmetysRepositoryException
    {
        return TaggableAmetysObjectHelper.getTags(this);
    }
    
    public ModifiableIndexableDataHolder getDataHolder()
    {
        ModifiableRepositoryData repositoryData = new JCRRepositoryData(getNode());
        return new DefaultModifiableModelAwareDataHolder(repositoryData, _getFactory().getThreadModel());
    }

    public ThreadCategoryEnum getCategory()
    {
        String category = getValue(ATTRIBUTE_CATEGORY);
        if (EnumUtils.isValidEnum(ThreadCategoryEnum.class, category))
        {
            return ThreadCategoryEnum.valueOf(category);
        }
        else
        {
            return ThreadCategoryEnum.DISCUSSION;
        }
    }

    public void setCategory(ThreadCategoryEnum category)
    {
        setValue(ATTRIBUTE_CATEGORY, category.name());
    }
    
    public UserIdentity getCloseAuthor()
    {
        return hasValue(ATTRIBUTE_CLOSEAUTHOR) ? getValue(ATTRIBUTE_CLOSEAUTHOR) : null;
    }
    
    public void setCloseAuthor(UserIdentity author)
    {
        setValue(ATTRIBUTE_CLOSEAUTHOR, author);
    }
    
    public ZonedDateTime getCloseDate()
    {
        return hasValue(ATTRIBUTE_CLOSEDATE) ? getValue(ATTRIBUTE_CLOSEDATE) : null;
    }

    public void setCloseDate(ZonedDateTime date)
    {
        setValue(ATTRIBUTE_CLOSEDATE, date);
    }
    
    public List<Binary> getAttachments()
    {
        Binary[] attachements = getValue(ATTRIBUTE_ATTACHMENTS, false, new Binary[0]);
        return Arrays.asList(attachements);
    }
    
    public void setAttachments(List<Binary> attachments)
    {
        setValue(ATTRIBUTE_ATTACHMENTS, attachments.toArray(new Binary[attachments.size()]));
    }

    private ModifiableModelLessDataHolder _getCommentDataHolder()
    {
        try
        {
            Node baseNode = getBaseNode();
            if (!baseNode.hasNode("ametys:comments"))
            {
                baseNode.addNode("ametys:comments", "ametys:compositeMetadata");
            }
            baseNode.getSession().save();
            ModifiableRepositoryData repositoryData = new JCRRepositoryData(baseNode.getNode("ametys:comments"));
            return new DefaultModifiableModelLessDataHolder(_getFactory().getUnversionedDataTypeExtensionPoint(), repositoryData);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }
    
    public RichTextComment createComment()
    {
        return new RichTextComment(_getCommentDataHolder());
    }
    
    public RichTextComment createComment(String commentId, ZonedDateTime creationDate)
    {
        return new RichTextComment(_getCommentDataHolder(), commentId, creationDate);
    }
    
    public RichTextComment getComment(String commentId) throws AmetysRepositoryException
    {
        return RichTextComment.getComment(_getCommentDataHolder(), commentId);
    }

    public List<RichTextComment> getComments(boolean includeNotValidatedComments, boolean includeValidatedComments) throws AmetysRepositoryException
    {
        return RichTextComment.getComments(_getCommentDataHolder(), includeNotValidatedComments, includeValidatedComments);
    }
    
    public RichTextComment getAcceptedAnswer()
    {
        // Discussion can't have accepted answer, return null immediately;
        if (getCategory() ==  ThreadCategoryEnum.DISCUSSION)
        {
            return null;
        }
        
        return getFlattenComments(true, true).stream()
            .filter(RichTextComment::isAccepted)
            .filter(Predicate.not(RichTextComment::isDeleted))
            .findFirst()
            .orElse(null);
    }
    
    public List<RichTextComment> getFlattenComments(boolean includeNotValidatedComments, boolean includeValidatedComments) throws AmetysRepositoryException
    {
        List<RichTextComment> flattenedComments = _flattenComment(getComments(includeNotValidatedComments, includeValidatedComments)).stream()
                .filter(subComment -> !subComment.isDeleted() || subComment.hasSubComments())
                .sorted((c1, c2) -> 
                {
                    return c1.getCreationDate().compareTo(c2.getCreationDate());
                })
                .toList();
        return flattenedComments;
    }

    
    private List<RichTextComment> _flattenComment(List<RichTextComment> comments)
    {
        List<RichTextComment> flattenComments = new ArrayList<>();
        flattenComments.addAll(comments);
        for (RichTextComment comment : comments)
        {
            List<RichTextComment> subComments = comment.getSubComment(true, true);
            flattenComments.addAll(_flattenComment(subComments));
        }
        return flattenComments;
    }

}
