/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.minisite;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.web.ObservationConstants;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.Site;

/**
 * This {@link Observer} converts Page events to minisite page events if needed
 */
public class ConvertPageEvent2MinisitePageEventObserver extends AbstractConvertMinisiteEventObserver
{
    /** Map for events id conversion */
    protected static final Map<String, String> _EVENTS_ID_CONVERSION = Map.of(
            org.ametys.web.ObservationConstants.EVENT_PAGE_ADDED, org.ametys.plugins.workspaces.ObservationConstants.EVENT_MINISITE_PAGE_CREATED,
            org.ametys.web.ObservationConstants.EVENT_PAGE_UPDATED, org.ametys.plugins.workspaces.ObservationConstants.EVENT_MINISITE_PAGE_UPDATED,
            org.ametys.web.ObservationConstants.EVENT_PAGE_RENAMED, org.ametys.plugins.workspaces.ObservationConstants.EVENT_MINISITE_PAGE_RENAMED,
            org.ametys.web.ObservationConstants.EVENT_PAGE_DELETING, org.ametys.plugins.workspaces.ObservationConstants.EVENT_MINISITE_PAGE_DELETED);
    
    private AmetysObjectResolver _resolver;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
    }
    
    public boolean supports(Event event)
    {
        String eventId = event.getId();
        return _EVENTS_ID_CONVERSION.containsKey(eventId);
    }

    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        Page page = (Page) event.getArguments().get(org.ametys.web.ObservationConstants.ARGS_PAGE);
        if (page == null)
        {
            // in case of page deletion
            String pageId = (String) event.getArguments().get(org.ametys.web.ObservationConstants.ARGS_PAGE_ID);
            page = _resolver.resolveById(pageId);
        }
        
        Project project = _getProject(page);
        
        if (project != null && isMinisitePage(project, page))
        {
            Map<String, Object> eventParams = new HashMap<>(event.getArguments());
            // add project to event parameters
            eventParams.put(org.ametys.plugins.workspaces.ObservationConstants.ARGS_PROJECT, project);
            
            if (event.getId().equals(org.ametys.web.ObservationConstants.EVENT_PAGE_DELETING))
            {
                eventParams.remove(ObservationConstants.ARGS_PAGE);
                eventParams.put(ObservationConstants.ARGS_PAGE_ID, page.getId());
                eventParams.put(org.ametys.plugins.workspaces.ObservationConstants.ARGS_PAGE_TITLE, page.getTitle());
            }
            
            _observationManager.notify(new Event(_EVENTS_ID_CONVERSION.get(event.getId()), event.getIssuer(), eventParams));
        }
    }
    
    /**
     * Get the project the page belongs
     * @param page the page
     * @return the project or null if the page does not belong to a project
     */
    protected Project _getProject(Page page)
    {
        Site site = page.getSite();
        List<Project> projectsForSite = _projectManager.getProjectsForSite(site);
        if (!projectsForSite.isEmpty())
        {
            return projectsForSite.get(0);
        }
        
        return null;
    }
}
