/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.report;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.SAXException;

import org.ametys.plugins.workspaces.categories.Category;
import org.ametys.plugins.workspaces.categories.CategoryHelper;
import org.ametys.plugins.workspaces.project.objects.Project;

/**
 * Generator for the CSV report service.
 * Sax data required to select which report to use, and its context
 */
public class ReportServiceGenerator extends ServiceableGenerator
{
    private ReportHelper _reportHelper;
    private CategoryHelper _categoryHelper;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _reportHelper = (ReportHelper) smanager.lookup(ReportHelper.ROLE);
        _categoryHelper = (CategoryHelper) smanager.lookup(CategoryHelper.ROLE);
    }
    
    public void generate() throws IOException, SAXException, ProcessingException
    {
        contentHandler.startDocument();
        XMLUtils.startElement(contentHandler, "reports");
        
        List<Project> availableProjects = _reportHelper.getAvailableProjects();
        // Get the root categories from the available projects
        Set<Category> rootCategories = _reportHelper.getAvailableCategories(availableProjects);
        
        saxCategories(rootCategories);
        saxProjects(availableProjects);
        
        XMLUtils.endElement(contentHandler, "reports");
        contentHandler.endDocument();
    }
    
    /**
     * SAX the available projects
     * @param projects the projects to sax
     * @throws SAXException if an error occurred while saxing
     */
    protected void saxProjects(List<Project> projects) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "projects");
        for (Project project : projects)
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("id", project.getId());
            attrs.addCDATAAttribute("name", project.getName());
            XMLUtils.createElement(contentHandler, "project", attrs, project.getTitle());
        }
        XMLUtils.endElement(contentHandler, "projects");
    }
    
    /**
     * SAX the categories
     * @param rootCategories the root categories to sax
     * @throws SAXException if an error occurred while saxing
     */
    protected void saxCategories(Set<Category> rootCategories) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "categories");
        for (Category category : rootCategories)
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("id", category.getId());
            attrs.addCDATAAttribute("name", category.getName());
            XMLUtils.startElement(contentHandler, "category", attrs);
            
            Map<String, String> colors = _categoryHelper.getCategoryColor(category);
            XMLUtils.createElement(contentHandler, "color", colors.get("main"));
            category.getTitle().toSAX(contentHandler, "title");
            XMLUtils.endElement(contentHandler, "category");
        }
        XMLUtils.endElement(contentHandler, "categories");
    }
    
}
