/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.search.module;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;

import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.content.indexing.solr.SolrFieldNames;
import org.ametys.cms.repository.Content;
import org.ametys.cms.search.SortOrder;
import org.ametys.cms.search.content.ContentSearcherFactory.SimpleContentSearcher;
import org.ametys.cms.search.query.AndQuery;
import org.ametys.cms.search.query.ContentLanguageQuery;
import org.ametys.cms.search.query.ContentTypeQuery;
import org.ametys.cms.search.query.FullTextQuery;
import org.ametys.cms.search.query.MatchNoneQuery;
import org.ametys.cms.search.query.OrQuery;
import org.ametys.cms.search.query.Query;
import org.ametys.cms.search.query.Query.Operator;
import org.ametys.cms.search.query.StringQuery;
import org.ametys.plugins.workspaces.WorkspacesConstants;
import org.ametys.plugins.workspaces.minisite.MiniSiteWorkspaceModule;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.web.repository.content.WebContent;

/**
 * Generator for minisite search module
 *
 */
public class MiniSiteSearchModuleGenerator extends AbstractContentSolrSearchModuleGenerator
{
    @Override
    protected SimpleContentSearcher getSearcher()
    {
        return _contentSearcherFactory.create(WorkspacesConstants.PROJECT_ARTICLE_CONTENT_TYPE);
    }
    
    @Override
    protected String getSortFieldName()
    {
        return "lastModified";
    }
    
    @Override
    protected SortOrder getSortOrder()
    {
        return SortOrder.DESC;
    }
    
    @Override
    protected Query getQuery(String siteName, String lang, String textfield, Request request)
    {
        Collection<Query> queries = new HashSet<>();
        
        if (StringUtils.isNotBlank(textfield))
        {
            Query titleQuery = new StringQuery(SolrFieldNames.TITLE, Operator.SEARCH, textfield.trim(), lang);
            Query fullTextQuery = new FullTextQuery(textfield.trim(), SolrFieldNames.FULL, lang, Operator.SEARCH);
            queries.add(new OrQuery(titleQuery, fullTextQuery));
        }
        
        List<Project> projects = getProjects(request, true);
        projects = filterProjectsForModule(projects, MiniSiteWorkspaceModule.MINISITE_MODULE_ID);
        if (projects.isEmpty())
        {
            // No projet available for this module
            return new MatchNoneQuery();
        }
        
        queries.add(getProjectArticleQuery(request, lang, projects));
        
        return new AndQuery(queries);
    }
    
    /**
     * Get the query for projects' articles
     * @param request the request
     * @param lang the language
     * @param projects the targeted projects
     * @return the query
     */
    protected Query getProjectArticleQuery(Request request, String lang, List<Project> projects)
    {
        Query projectArticleTypeQuery = new ContentTypeQuery(Operator.EQ,  WorkspacesConstants.PROJECT_ARTICLE_CONTENT_TYPE);
        Query projectSiteQuery = getSiteQuery(projects);
        Query langQuery = new ContentLanguageQuery(lang);
        
        return new AndQuery(projectArticleTypeQuery, projectSiteQuery, langQuery);
    }
    
    @Override
    protected void saxAdditionalInformation(Content content) throws SAXException
    {
        super.saxAdditionalInformation(content);
        
        // SAX project
        String siteName = ((WebContent) content).getSiteName();
        Project project = _projectManager.getProject(siteName);
        if (project != null)
        {
            saxProject(project);
        }
    }
}
