/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.statistics;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModule;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModuleExtensionPoint;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.PluginAware;
import org.ametys.runtime.plugins.admin.statistics.Statistics;
import org.ametys.runtime.plugins.admin.statistics.StatisticsNode;
import org.ametys.runtime.plugins.admin.statistics.StatisticsProvider;
import org.ametys.runtime.plugins.admin.statistics.StatisticsValue;

/**
 * Send workspaces statistics
 */
public class WorkspacesStatisticsProvider implements StatisticsProvider, Serviceable, PluginAware
{
    private ProjectManager _projectManager;
    private WorkspaceModuleExtensionPoint _workspaceModuleEP;
    private String _id;

    public void service(ServiceManager manager) throws ServiceException
    {
        _projectManager = (ProjectManager) manager.lookup(ProjectManager.ROLE);
        _workspaceModuleEP = (WorkspaceModuleExtensionPoint) manager.lookup(WorkspaceModuleExtensionPoint.ROLE);
    }
    
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _id = id;
    }
    
    public Statistics getStatistics()
    {
        return new StatisticsNode(
            _id,
            new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_STATISTICS_LABEL"),
            "ametysicon-abecedary4",
            null,
            List.of(
                _configStats(),
                _projectsStats(),
                _modulesStats()
            ),
            true
        );
    }

    private StatisticsNode _configStats()
    {
        boolean configOOPreview = Config.getInstance().getValue("workspaces.onlyoffice.enabled", false, false);
        boolean configOOEdit = configOOPreview && Config.getInstance().getValue("workspaces.onlyoffice.edition.enabled", false, false);
        boolean configWebdav = Config.getInstance().getValue("workspaces.msoffice.enabled", false, false);
        
        return new StatisticsNode(
            "config",
            new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_STATISTICS_CONFIG_LABEL"),
            "ametysicon-gear39",
            (configOOPreview ? 1 : 0) + (configOOEdit ? 1 : 0) + (configWebdav ? 1 : 0),
            List.of(
                new StatisticsValue(
                    "oopreview",
                    new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_STATISTICS_CONFIG_OOPREVIEW_LABEL"),
                    "ametysicon-code-html-picture62",
                    configOOPreview
                ),
                new StatisticsValue(
                    "ooedit",
                    new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_STATISTICS_CONFIG_OOEDIT_LABEL"),
                    "ametysicon-editor-indent-more",
                    configOOEdit
                ),
                new StatisticsValue(
                    "webdav",
                    new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_STATISTICS_CONFIG_WEBDAV_LABEL"),
                    "ametysicon-file-extension-doc",
                    configWebdav
                )
            ),
            false
        );
    }
    
    private StatisticsNode _modulesStats()
    {
        List<Map<String, Object>> projectsStatistics = _projectManager.getProjectsStatisticsForClientSide();
        
        List<Statistics> modulesStats = new ArrayList<>();
        for (WorkspaceModule module : _workspaceModuleEP.getModules())
        {
            modulesStats.add(_moduleStat(projectsStatistics, module));
        }
        
        return new StatisticsNode(
            "modules",
            new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_STATISTICS_MODULES_LABEL"),
            "ametysicon-puzzle-piece1",
            null,
            modulesStats,
            true
        );
    }
    
    private Statistics _moduleStat(List<Map<String, Object>> projectsStatistics, WorkspaceModule module)
    {
        long totalActivated = 0;
        
        long totalSize = 0;
        long maxSize = 0;
        List<Long> sizes = new ArrayList<>();
        
        long totalCount = 0;
        long maxCount = 0;
        List<Long> counts = new ArrayList<>();
        
        for (Map<String, Object> oneProjectStatistics : projectsStatistics)
        {
            if ((Boolean) oneProjectStatistics.get(module.getModuleName() + "$activated"))
            {
                totalActivated++;
                
                long size = (Long) oneProjectStatistics.get(module.getModuleName() + "$size");
                if (size > 0) // negative size are error messages
                {
                    totalSize += size;
                    maxSize = Math.max(maxSize, size);
                    sizes.add(size);
                }
                
                String numberKey = oneProjectStatistics.keySet().stream().filter(k -> k.startsWith(module.getModuleName() + "$") && k.endsWith("_number")).findFirst().orElse(null);
                if (numberKey != null)
                {
                    long count = ((Number) oneProjectStatistics.get(numberKey)).longValue();
                    totalCount += count;
                    maxCount = Math.max(maxCount, count);
                    counts.add(count);
                }
            }
        }
        
        counts.sort(null);
        sizes.sort(null);
        
        return new StatisticsNode(
            module.getModuleName(),
            module.getModuleTitle(),
            "ametysicon-puzzle33",
            totalActivated,
            List.of(
                new StatisticsNode(
                    "count",
                    new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_STATISTICS_MODULES_COUNT_LABEL"),
                    "ametysicon-maths-abacus",
                    totalCount,
                    List.of(
                        new StatisticsValue(
                            "max",
                            new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_STATISTICS_MODULES_COUNT_MAX_LABEL"),
                            "ametysicon-sort51",
                            maxCount
                        ),
                        new StatisticsValue(
                            "median",
                            new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_STATISTICS_MODULES_COUNT_MEDIAN_LABEL"),
                            "ametysicon-maths-window-symbol-x",
                            counts.size() > 0 ? counts.get(counts.size() / 2) : 0
                        )
                    ),
                    false
                ),
                new StatisticsNode(
                    "size",
                    new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_STATISTICS_MODULES_SIZE_LABEL"),
                    "ametysicon-code-css-letter-spacing",
                    totalSize,
                    List.of(
                        new StatisticsValue(
                            "max",
                            new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_STATISTICS_MODULES_SIZE_MAX_LABEL"),
                            "ametysicon-sort51",
                            maxSize
                        ),
                        new StatisticsValue(
                            "median",
                            new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_STATISTICS_MODULES_SIZE_MEDIAN_LABEL"),
                            "ametysicon-maths-window-symbol-x",
                            sizes.size() > 0 ? sizes.get(sizes.size() / 2) : 0
                        )
                    ),
                    false
                )
            ),
            false
        );
    }
    
    private StatisticsNode _projectsStats()
    {
        long nbPrivate = 0;
        long nbModerated = 0;
        long nbPublic = 0;
           
        try (AmetysObjectIterable<Project> projects = _projectManager.getProjects())
        {
            for (Project project : projects)
            {
                switch (project.getInscriptionStatus())
                {
                    case OPEN:
                        nbPublic++;
                        break;
                    case MODERATED:
                        nbModerated++;
                        break;
                    case PRIVATE:
                        nbPrivate++;
                        break;
                    default:
                        throw new IllegalArgumentException("Unknown inscriptionStatus " + project.getInscriptionStatus().toString() + " for project " + project.getId());
                }
            }
        }
        
        return new StatisticsNode(
            "count",
            new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_STATISTICS_PROJECTS_LABEL"),
            "ametysicon-file98",
            nbPrivate + nbModerated + nbPublic,
            List.of(
                new StatisticsValue(
                    "open",
                    new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_STATISTICS_PROJECTS_OPEN_LABEL"),
                    "ametysicon-body-group",
                    nbPublic
                ),
                new StatisticsValue(
                    "moderated",
                    new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_STATISTICS_PROJECTS_MODERATED_LABEL"),
                    "ametysicon-body-people-tie",
                    nbModerated
                ),
                new StatisticsValue(
                    "private",
                    new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_STATISTICS_PROJECTS_PRIVATE_LABEL"),
                    "ametysicon-body-people",
                    nbPrivate
                )
            ),
            false
        );
    }
}
