/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.tasks;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.data.Binary;
import org.ametys.plugins.explorer.resources.ModifiableResourceCollection;
import org.ametys.plugins.explorer.resources.jcr.JCRResourcesCollectionFactory;
import org.ametys.plugins.repository.data.holder.ModifiableModelAwareDataHolder;
import org.ametys.plugins.repository.jcr.DefaultTraversableAmetysObject;
import org.ametys.plugins.workspaces.AbstractWorkspaceModule;
import org.ametys.plugins.workspaces.ObservationConstants;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.util.StatisticColumn;
import org.ametys.plugins.workspaces.util.StatisticsColumnType;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.repository.page.ModifiablePage;
import org.ametys.web.repository.page.ModifiableZone;
import org.ametys.web.repository.page.ModifiableZoneItem;
import org.ametys.web.repository.page.ZoneItem.ZoneType;

import com.google.common.collect.ImmutableSet;

/**
 * Tasks manager for workspaces
 */
public class TasksWorkspaceModule extends AbstractWorkspaceModule
{
    /** The id of task module */
    public static final String TASK_MODULE_ID = TasksWorkspaceModule.class.getName();
    
    /** Workspaces tasks list node name */
    private static final String __WORKSPACES_TASKS_NODE_NAME = "tasks";
    
    /** Workspaces root tasks node name */
    private static final String __WORKSPACES_TASKS_ROOT_NODE_NAME = "tasks-root";

    /** Workspaces root tasks lists node name */
    private static final String __WORKSPACES_TASKS_LIST_ROOT_NODE_NAME = "tasks-list-root";

    private static final String __TASK_NUMBER_HEADER_ID = __WORKSPACES_TASKS_NODE_NAME + "$task_number";
    
    /** The tasks module DAO */
    private WorkspaceTaskDAO _workspaceTaskDAO;

    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _workspaceTaskDAO = (WorkspaceTaskDAO) serviceManager.lookup(WorkspaceTaskDAO.ROLE);
    }
    
    @Override
    public String getId()
    {
        return TASK_MODULE_ID;
    }
    
    @Override
    public String getModuleName()
    {
        return __WORKSPACES_TASKS_NODE_NAME;
    }
    
    public int getOrder()
    {
        return ORDER_TASKS;
    }
    
    @Override
    protected String getModulePageName()
    {
        return "tasks";
    }
    
    public I18nizableText getModuleTitle()
    {
        return new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_TASK_LABEL");
    }
    public I18nizableText getModuleDescription()
    {
        return new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_TASK_DESCRIPTION");
    }
    @Override
    protected I18nizableText getModulePageTitle()
    {
        return new I18nizableText("plugin." + _pluginName, "PLUGINS_WORKSPACES_PROJECT_WORKSPACE_PAGE_TASKS_TITLE");
    }
    
    @Override
    protected void initializeModulePage(ModifiablePage taskPage)
    {
        ModifiableZone defaultZone = taskPage.createZone("default");
        
        String serviceId = "org.ametys.plugins.workspaces.module.Tasks";
        ModifiableZoneItem defaultZoneItem = defaultZone.addZoneItem();
        defaultZoneItem.setType(ZoneType.SERVICE);
        defaultZoneItem.setServiceId(serviceId);
        
        ModifiableModelAwareDataHolder serviceDataHolder = defaultZoneItem.getServiceParameters();
        serviceDataHolder.setValue("xslt", _getDefaultXslt(serviceId));
    }
    
    /**
     * Get the URI of a task in project'site
     * @param project The project
     * @param taskId The id of the task
     * @return The task uri
     */
    public String getTaskUri(Project project, String taskId)
    {
        String moduleUrl = getModuleUrl(project);
        if (moduleUrl != null)
        {
            StringBuilder sb = new StringBuilder();
            sb.append(moduleUrl);
            sb.append("#task-").append(taskId);
            
            return sb.toString();
        }
        
        return null;
    }
    
    /**
     * Get the root for tasks
     * @param project The project
     * @param create true to create root if not exists
     * @return The root for tasks
     */
    public DefaultTraversableAmetysObject getTasksRoot(Project project, boolean create)
    {
        ModifiableResourceCollection moduleRoot = getModuleRoot(project, create);
        return _getAmetysObject(moduleRoot, __WORKSPACES_TASKS_ROOT_NODE_NAME, JCRResourcesCollectionFactory.RESOURCESCOLLECTION_NODETYPE, create);
    }
    
    /**
     * Get the root for tasks lists
     * @param project The project
     * @param create true to create root if not exists
     * @return The root for tasks lists
     */
    public DefaultTraversableAmetysObject getTasksListsRoot(Project project, boolean create)
    {
        ModifiableResourceCollection moduleRoot = getModuleRoot(project, create);
        return _getAmetysObject(moduleRoot, __WORKSPACES_TASKS_LIST_ROOT_NODE_NAME, JCRResourcesCollectionFactory.RESOURCESCOLLECTION_NODETYPE, create);
    }


    @Override
    public Set<String> getAllowedEventTypes()
    {
        return ImmutableSet.of(ObservationConstants.EVENT_TASK_CREATED, ObservationConstants.EVENT_TASK_ASSIGNED, ObservationConstants.EVENT_TASK_CLOSED_STATUS_CHANGED, ObservationConstants.EVENT_TASK_DELETING);
    }
    
    @Override
    public Map<String, Object> _getInternalStatistics(Project project, boolean isActive)
    { 
        if (isActive)
        {
            return Map.of(__TASK_NUMBER_HEADER_ID, _workspaceTaskDAO.getTasksCount(project));
        }
        else
        {
            return Map.of(__TASK_NUMBER_HEADER_ID, __SIZE_INACTIVE);
        }
    }
    
    @Override
    protected long _getModuleSize(Project project)
    {
        return _workspaceTaskDAO.getProjectTasks(project)
                         .stream()
                         .map(Task::getAttachments)
                         .flatMap(Collection::stream)
                         .mapToLong(Binary::getLength)
                         .sum();
    }
    
    @Override
    public List<StatisticColumn> _getInternalStatisticModel()
    {
        return List.of(new StatisticColumn(__TASK_NUMBER_HEADER_ID, new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_PROJECT_STATISTICS_TOOL_COLUMN_TASK_NUMBER"))
                .withRenderer("Ametys.plugins.workspaces.project.tool.ProjectsGridHelper.renderElements")
                .withType(StatisticsColumnType.LONG)
                .withGroup(GROUP_HEADER_ELEMENTS_ID));
    }

    @Override
    protected boolean _showModuleSize()
    {
        return true;
    }
}
