/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
import { defineStore } from 'pinia'
import { callMethod } from '@common/helper/ServerCommHelper'

/**
 * Pinia store for managing user preferences concerning all modules.
 */
export const useGlobalUserPreferencesStore = defineStore('globalUserPreferences', {
    state: () => (
    {
        userPreferences: {} // User preferences, with moduleId as key and JSON of preferences as value
    }),
    actions:
    {
        /**
         * Returns the user preferences for a given module.
         * @param {String} moduleId the module identifier
         */
        getUserPreferences(moduleId)
        {
            // The || {} is needed, as before the load, we need an empty JSON
            return this.userPreferences[moduleId] || {};
        },

        /**
         * Sets the user preference for a given module.
         * @param {String} moduleId the module identifier
         * @param {Object} preferences the preferences of the module to set
         */
        setModulePreferences(moduleId, preferences)
        {
            this.userPreferences[moduleId] = preferences;
            callMethod(
            {
                role: 'org.ametys.plugins.workspaces.preferences.WorkspaceViewUserPreferencesDAO',
                methodName: 'setUserPreference',
                parameters: [moduleId, JSON.stringify(preferences)]
            })
        },

        /**
         * Loads the user preference for a given module.
         * @param {String} moduleId the module identifier
         * @param {Object} initialPreferences the initial preferences of the module
         */
        async loadModulePreferences(moduleId, initialPreferences)
        {
            let valueAsJSON = await callMethod(
            {
                role: 'org.ametys.plugins.workspaces.preferences.WorkspaceViewUserPreferencesDAO',
                methodName: 'getUserPreference',
                parameters: [moduleId]
            })
            this.userPreferences[moduleId] = valueAsJSON ? JSON.parse(valueAsJSON) : initialPreferences;

        },

    }
})
