/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
import { defineStore } from 'pinia'
import { callMethod } from '@common/helper/ServerCommHelper'
import { openWebdavUri } from '@/helpers/WebdavHelper'
import { initOnlyOffice, createOnlyOfficeModal, initOnlyOfficeModal, closeEditor } from '@/helpers/OnlyOfficeHelper'
import WorkspacesUtils from 'WorkspacesUtils'
import AmetysFront from 'AmetysFront'
import i18n from 'i18n'
import DocsAPI from 'DocsAPI'
import { useFiltersStore } from '@/stores/filters'
import { useTagsStore } from '@common/store/project-tags/tags'
import { useFoldersStore } from '@/stores/folders'
import { usePreviewItemStore } from '@/stores/previewItem'
import { getRelativeDateWithPrefix } from '@common/helper/dateComposable';

/**
 * Pinia store for managing files and related actions.
 */
export const useFilesStore = defineStore('files', {
    state: () => (
    {
        files: [], // The list of files of the current folder
    }),
    getters:
    {
        /**
         * Returns the filtered and sorted list of files.
         */
        getFiles: state =>
        {
            let files = state.files;
            const filtersStore = useFiltersStore();
            const tagId = filtersStore.filterByTag?.text;
            function byTag(file, tagId)
            {
                if (tagId)
                {
                    return file.tags.find(tag => tag.text === tagId)
                }
                else
                {
                    return true
                }
            }
            files = files.filter(file => byTag(file, tagId))
            return files.sort((f1, f2) =>
                f1.name.toLowerCase() < f2.name.toLowerCase() ? -1 :
                (f1.name.toLowerCase() > f2.name.toLowerCase() ? 1 : 0)
            )
        },

        /**
         * Returns all unique tags from files.
         */
        getTags: state =>
        {
            let tags = []
            function exists(tag)
            {
                return tags.find(t => t.text === tag.text)
            }
            state.files.forEach(file =>
            {
                file.tags.forEach(tag =>
                {
                    if (!exists(tag))
                    {
                        tags.push(tag)
                    }
                })
            })

            tags.sort(function(c1, c2)
            {
                if (AmetysFront.Utils.deemphasize(c1.title.toLowerCase()) == AmetysFront.Utils.deemphasize(c2.title.toLowerCase())) return 0;
                else if (AmetysFront.Utils.deemphasize(c1.title.toLowerCase()) < AmetysFront.Utils.deemphasize(c2.title.toLowerCase())) return -1;
                else return 1;
            });

            return tags
        }
    },
    actions:
    {
        /**
         * Internal: transform file object for display.
         * @param {Object} file the file to transform.
         * @param {String} file.lastModified the last modified date of the file.
         * @param {String} file.creationDate the creation date of the file.
         * @param {Number} file.length the size of the file in bytes.
         * @param {String} file.fileType the type of the file (e.g. 'image', 'pdf', 'video'...).
         * @param {String} file.name the name of the file.
         */
        _transform(file)
        {
            let prefixMap = {
                defaultPrefix: i18n.PLUGINS_WORKSPACES_RELATIVE_DATE_PREFIX_RELATIVE,
                hoursPrefix: i18n.PLUGINS_WORKSPACES_RELATIVE_DATE_PREFIX_AT,
                dayPrefix: i18n.PLUGINS_WORKSPACES_RELATIVE_DATE_PREFIX_WEEK,
                datePrefix: i18n.PLUGINS_WORKSPACES_RELATIVE_DATE_PREFIX
            }
            file.readableLastModified = getRelativeDateWithPrefix(file.lastModified, prefixMap);
            file.readableCreationDate = getRelativeDateWithPrefix(file.creationDate, prefixMap);
            file.readableLength = AmetysFront.Utils.getReadableSize(file.length)
            file.fileTypeIcon = WorkspacesUtils.getFileTypeIcon(file.fileType)
            file.name = file.name.substring(0, file.name.lastIndexOf('.')) || file.name
        },

        /**
         * Set the files list.
         * @param {Array} files the list of files to set.
         */
        setFiles(files)
        {
            files.forEach(file => { this._transform(file) })
            this.files = files
        },

        /**
         * Update files after move.
         * @param {Array} files the list of files to update.
         */
        updateFilesAfterMove(files)
        {
            this.files = this.files.filter(value => !files.includes(value.id))
        },

        /**
         * Add a file to the list. Internal method.
         * @param {Object} file the file to add.
         */
        _addFile(file)
        {
            this._transform(file)
            this.files.push(file)
        },

        /**
         * Update a file in the list. Internal method.
         * @param {Object} payload the payload containing the file id and data to update.
         */
        _updateFile(payload)
        {
            this._transform(payload.data)

            const updateFileProps = (target, data) => {
                target.lastModified = data.lastModified;
                target.readableLastModified = data.readableLastModified;
                target.length = data.length;
                target.readableLength = data.readableLength;
                target.name = data.name;
                target.description = data.description;
                target.isAIDescription = data.isAIDescription;
                target.tags = data.tags;
                target.token = data.token;
            };

            let idx = this.files.findIndex(file => file.id === payload.id);
            if (idx !== -1) {
                updateFileProps(this.files[idx], payload.data);
            }
            const previewItemStore = usePreviewItemStore();
            if (previewItemStore.item?.id === payload.id) {
                updateFileProps(previewItemStore.item, payload.data);
            }
        },

        /**
         * Set tags for a file. Internal method.
         * @param {Object} payload the payload containing the file id and tags to set.
         * @param {String} payload.id the id of the file to set tags for.
         * @param {String[]} payload.tags the tags to set for the file.
         */
        _setTags(payload)
        {
            let idx = this.files.findIndex(file => file.id === payload.id)
            if (idx !== -1)
            {
                this.files[idx].tags = payload.tags
            }
        },

        /**
         * Delete a file from the list. Internal method.
         */
        _deleteFile(payload)
        {
            this.files.splice(payload.file, 1)
        },
        /**
         * Rename a file. Internal method.
         */
        _renameFile(payload)
        {
            payload.file.name = payload.name
            payload.file.path = payload.path
            payload.file.encodedPath = payload.encodedPath
        },
        /**
         * Tag a file.
         * @param {Object} payload the payload containing the file id and tags to set.
         * @param {String} payload.id the id of the file to set tags for.
         * @param {String[]} payload.tags the tags to set for the file.
         */
        async setTags(payload)
        {
            try
            {
                let data = await callMethod(
                {
                    role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
                    methodName: 'setTags',
                    parameters: [payload.id, payload.tags]
                })

                const tagsStore = useTagsStore()
                tagsStore.addTags(data.newTags);
                this._setTags({ id: payload.id, tags: data.fileTags })
            }
            catch (e)
            {
                AmetysFront.Event.fire('loaderFail',
                {
                    title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FILE_ERROR_TAG,
                    text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
                    details: null
                })
            }
        },

        /**
         * Check if a file exists.
         * @param {Object} payload the payload containing the folder id and file name to check.
         * @param {String} payload.folderId the id of the folder to check in.
         * @param {String} payload.fileName the name of the file to check.
         */
        async checkFileExists(payload)
        {
            try
            {
                return await callMethod(
                {
                    role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
                    methodName: 'resourceExists',
                    parameters: [payload.folderId, payload.fileName]
                })
            }
            catch (e)
            {
                AmetysFront.Event.fire('loaderFail',
                {
                    title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FILE_ERROR_ADD,
                    text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_CHECK_NAME_ERROR,
                    details: "Unable to check if file exists !" + e
                })
            }
        },

        /**
         * Upload a file.
         * @param {Object} payload the payload containing the file, folder id, unzip option, rename option, and update option.
         * @param {File} payload.file the file to upload.
         * @param {String} payload.folderId the id of the folder to upload the file to.
         * @param {boolean} payload.unzip whether to unzip the file after upload.
         * @param {boolean} payload.rename whether to rename the file if it already exists.
         * @param {boolean} payload.update whether to update the file if it already exists.
         */
        async uploadFile(payload)
        {
            AmetysFront.Event.fire('loaderStart',
            {
                text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FILE_UPLOAD_STARTED
            })
            try
            {
                let response = await callMethod(
                {
                    role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
                    methodName: 'addFile',
                    parameters: [payload.file, payload.folderId, payload.unzip, payload.rename, payload.update]
                })
                if (response.unzip)
                {
                    const foldersStore = useFoldersStore()
                    foldersStore.reload()
                    AmetysFront.Event.fire('loaderEnd',
                    {
                        text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FILES_UPLOAD_ENDED
                    })
                }
                else if (response.resources)
                {
                    if (!payload.update)
                    {
                        response.resources.forEach(res => { this._addFile(res) })
                        AmetysFront.Event.fire('loaderEnd',
                        {
                            text: response.resources.length > 1
                                ? i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FILES_UPLOAD_ENDED
                                : i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FILE_UPLOAD_ENDED
                        })
                    }
                    else
                    {
                        response.resources.forEach(res =>
                        {
                            this._updateFile({ id: res.id, data: res })
                            let idx = this.files.findIndex(file => file.id == res.id)

                            const previewItemStore = usePreviewItemStore();
                            previewItemStore.setItem({ item: this.files[idx], notOpen: true })
                        })
                        AmetysFront.Event.fire('loaderEnd',
                        {
                            text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FILE_UPDATED
                        })
                    }
                }
                else
                {
                    let errorMsg = response.message
                    let text = i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR
                    if (errorMsg == 'rejected')
                    {
                        text = i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FILE_ERROR_MAX_UPLOAD_SIZE_ERROR;
                    }
                    else if (errorMsg == 'infected')
                    {
                        text = i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FILE_ERROR_INFECTED;
                    }
                    else if (errorMsg == 'locked')
                    {
                        text = i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_LOCKED;
                    }
                    else if (errorMsg == 'already-exist')
                    {
                        text = i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FILE_ERROR_ALREADY_EXIST;
                    }
                    else if (errorMsg == 'unzip-errort')
                    {
                        text = i18n.PLUGINS_WORKSPACES_FILE_MANAGER_UNZIP_ERROR;
                    }

                    AmetysFront.Event.fire('loaderFail',
                    {
                        title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FILE_ERROR_ADD,
                        text: text,
                        details: null
                    })
                }
            }
            catch (e)
            {
                AmetysFront.Event.fire('loaderFail',
                {
                    title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FILE_ERROR_ADD,
                    text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
                    details: "Unable to add file !" + e
                })
            }
        },

        /**
         * Add a new file (after upload).
         */
        async addFile(payload)
        {
            try
            {
                let fileId = payload.resultsForFileSuccess[0].id
                let newFile = await callMethod(
                {
                    role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
                    methodName: 'getFile',
                    parameters: [fileId]
                })
                this._addFile(newFile)
            }
            catch (e)
            {
                AmetysFront.Event.fire('loaderFail',
                {
                    title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FILE_ERROR_GET,
                    text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
                    details: null
                })
            }
        },

        /**
         * Create a new file of a given type.
         * @param {String} folderId the id of the folder where to create the file.
         * @param {String} type the type of the file to create.
         */
        async createNewFile(folderId, type)
        {
            AmetysFront.Event.fire('loaderStart',
            {
                text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FILE_CREATED_STARTED
            })
            try
            {
                let response = await callMethod(
                {
                    role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
                    methodName: 'createFile',
                    parameters: [folderId, type, AmetysFront.getAppParameter('sitemapLanguage')]
                })
                if (response.resources)
                {
                    let file = response.resources[0]
                    this._addFile(file)
                    AmetysFront.Event.fire('loaderEnd',
                    {
                        text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FILE_CREATED_ENDED
                    })
                    return file
                }
                else
                {
                    let errorMsg = response.message
                    let text = i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR
                    if (errorMsg == 'locked')
                        text = i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_LOCKED
                    AmetysFront.Event.fire('loaderFail',
                    {
                        title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FILE_ERROR_CREATED,
                        text: text,
                        details: null
                    })
                }
            }
            catch (e)
            {
                AmetysFront.Event.fire('loaderFail',
                {
                    title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FILE_ERROR_CREATED,
                    text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
                    details: null
                })
            }
            return null
        },

        /**
         * Delete a file.
         * @param {String} id the id of the file to delete.
         */
        async deleteFile(id)
        {
            AmetysFront.Event.fire('loaderStart',
            {
                text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_DELETING
            })
            try
            {
                let result = await callMethod(
                {
                    role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
                    methodName: 'deleteObject',
                    parameters: [[id]]
                })
                if (result.success)
                {
                    let idx = this.files.findIndex(file => file.id == id)
                    this._deleteFile({ file: idx })
                    const previewItemStore = usePreviewItemStore();
                    const foldersStore = useFoldersStore()
                    previewItemStore.item = foldersStore.currentFolder;
                    setTimeout(() =>
                    {
                        AmetysFront.Event.fire('loaderEnd',
                        {
                            text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_DELETE_FILE_SUCCESS
                        })
                    }, 500)
                }
                else
                {
                    let errorMsg = result.message
                    AmetysFront.Event.fire('loaderFail',
                    {
                        title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FILE_ERROR_DELETE,
                        text: errorMsg == 'locked'
                            ? i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FILE_ERROR_LOCKED
                            : i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
                        details: null
                    })
                }
            }
            catch (e)
            {
                AmetysFront.Event.fire('loaderFail',
                {
                    title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FILE_ERROR_DELETE,
                    text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
                    details: null
                })
            }
        },

        /**
         * Rename a file.
         * @param {Object} payload the payload containing the file id and new name.
         * @param {String} payload.id the id of the file to rename.
         * @param {String} payload.name the new name for the file (without extension).
         * @param {String} payload.fileExtension the file extension to append to the new name.
         */
        async renameFile(payload)
        {
            let file = this.files.find(file => file.id === payload.id)
            let fullName = payload.name + "." + file.fileExtension
            let success = false
            try
            {
                let result = await callMethod(
                {
                    role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
                    methodName: 'renameFile',
                    parameters: [file.id, fullName]
                })
                if (!result.success)
                {
                    let text = i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR
                    if (result.message == 'locked')
                        text = i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FILE_ERROR_LOCKED
                    else if (result.message == 'already-exist')
                        text = i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FILE_ERROR_ALREADY_EXISTS
                    AmetysFront.Event.fire('loaderFail',
                    {
                        title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FILE_ERROR_RENAME,
                        text: text,
                        details: null
                    })
                }
                else
                {
                    success = true
                    this._renameFile(
                    {
                        file: file,
                        name: payload.name,
                        path: result.path,
                        encodedPath: result.encodedPath
                    })
                }
            }
            catch (e)
            {
                AmetysFront.Event.fire('loaderFail',
                {
                    title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FILE_ERROR_RENAME,
                    text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
                    details: null
                })
            }
            return success
        },

        /**
         * Open file for edition with MS Office.
         * @param {Object} file the file to edit.
         */
        async msOfficeEditFile(file)
        {
            try
            {
                let webdavUri = await callMethod(
                {
                    role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
                    methodName: 'generateWebdavUri',
                    parameters: [file.id]
                })
                openWebdavUri(file, webdavUri)
                AmetysFront.Event.fire('loaderEnd',
                {
                    text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_MSOFFICE_EDIT_OPEN_END
                })
            }
            catch (e)
            {
                AmetysFront.Event.fire('loaderFail',
                {
                    title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_MSOFFICE_EDIT_ERROR_RESULT_TITLE,
                    text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_MSOFFICE_EDIT_ERROR_RESULT_MESSAGE,
                    details: 'Unable to open the document in edition ' + file.id
                })
            }
        },

        /**
         * Open file for edition with Only Office.
         * @param {Object} payload the payload containing the file id, div id, edition flag, mobile flag, and callback.
         * @param {String} payload.file.id the id of the file to edit.
         * @param {String} payload.divId the id of the div where to open the editor.
         * @param {boolean} payload.isEdition whether the file is being edited.
         * @param {boolean} payload.isMobile whether the file is being opened on a mobile device.
         * @param {Function} payload.setDocEditor a callback to set the document editor instance.
         * @param {Function} payload.callback a callback to execute after the editor is initialized.
         * @param {Function} payload.onFailureCb a callback to execute on failure.
         */
        async onlyOfficeEditFile(payload)
        {
            let triesCount = 0
            async function onFailureCb(docEditor)
            {
                closeEditor(docEditor)
                if (triesCount < 3)
                {
                    window.console.log("Failed to open OnlyOffice edition (outdated document key), try again...")
                    triesCount++
                    await openOnlyOfficeDocument()
                }
                else
                {
                    AmetysFront.Event.fire('loaderFail',
                    {
                        title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ONLYOFFICE_EDIT_ERROR_TITLE,
                        text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ONLYOFFICE_EDIT_ERROR_MESSAGE
                    })
                }
            }
            async function openOnlyOfficeDocument()
            {
                let docEditor = null
                try
                {
                    let ooInfos = await callMethod(
                    {
                        role: 'org.ametys.plugins.workspaces.documents.onlyoffice.OnlyOfficeManager',
                        methodName: 'getOnlyOfficeInfo',
                        parameters: [payload.file.id]
                    })
                    let ooConfig = initOnlyOffice(ooInfos, onFailureCb, payload.isEdition, payload.isMobile)
                    let sign = await callMethod(
                    {
                        role: 'org.ametys.plugins.workspaces.documents.onlyoffice.OnlyOfficeManager',
                        methodName: 'signConfiguration',
                        parameters: [JSON.stringify(ooConfig)]
                    })
                    if (sign && sign.success)
                    {
                        if (sign.signature)
                        {
                            ooConfig.token = sign.signature
                        }
                        if (payload.isEdition)
                        {
                            createOnlyOfficeModal(
                            {
                                closeTxt: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ONLYOFFICE_EDIT_CLOSE_TXT,
                                openNewTxt: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ONLYOFFICE_EDIT_OPENNEW_TXT
                            })
                        }
                        docEditor = new DocsAPI.DocEditor(payload.divId, ooConfig)
                        if (payload.setDocEditor)
                        {
                            payload.setDocEditor(docEditor)
                        }
                        ooConfig.docEditor = docEditor
                        if (payload.isEdition)
                        {
                            initOnlyOfficeModal(ooConfig, payload.callback)
                        }
                    }
                    else
                    {
                        closeEditor(docEditor)
                        AmetysFront.Event.fire('loaderFail',
                        {
                            title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ONLYOFFICE_EDIT_ERROR_TITLE,
                            text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ONLYOFFICE_SIGNATURE_KEY,
                            details: 'Unable to open the document in edition ' + payload.file.id
                        })
                    }
                }
                catch (e)
                {
                    closeEditor(docEditor)
                    AmetysFront.Event.fire('loaderFail',
                    {
                        title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ONLYOFFICE_EDIT_ERROR_TITLE,
                        text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ONLYOFFICE_EDIT_ERROR_MESSAGE,
                        details: 'Unable to open the document in edition ' + payload.file.id + ': ' + e
                    })
                }
            }
            openOnlyOfficeDocument()
        },

        /**
         * Generate a description for a file using AI.
         * @param {String} fileId the id of the file to generate a description for.
         */
        generateDescriptionByIA(fileId)
        {
            try
                {

                let file = this.files.find(file => file.id === fileId)

                if (!file) return;

                const previewItemStore = usePreviewItemStore();
                previewItemStore.isGenerating = true;

                callMethod(
                {
                    role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
                    methodName: 'generateFileDescription',
                    parameters: [fileId]
                })
                .then(result => {
                    if (result.error)
                    {
                        AmetysFront.Event.fire('loaderFail',
                        {
                            title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_AI_DOCUMENT_SUMMARY_ERROR_TITLE,
                            text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_AI_DOCUMENT_SUMMARY_ERROR_MESSAGE,
                        })
                    }
                    this._updateFile({id: fileId, data: result})
                    previewItemStore.isGenerating = false;
                })
                .catch(_ => {
                    AmetysFront.Event.fire('loaderFail',
                    {
                        title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_AI_DOCUMENT_SUMMARY_ERROR_TITLE,
                        text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_AI_DOCUMENT_SUMMARY_ERROR_MESSAGE,
                    })
                    previewItemStore.isGenerating = false;
                });
            }
            catch (e)
            {
                AmetysFront.Event.fire('loaderFail',
                {
                    title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_AI_DOCUMENT_SUMMARY_ERROR_TITLE,
                    text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_AI_DOCUMENT_SUMMARY_ERROR_MESSAGE,
                    details: null
                })
            }
        },

        /**
         * Edit a file
         * @param {Object} payload the payload object
         * @param {Object} payload.file the file object to edit
         * @param {string} payload.name the new name of the file
         * @param {string} payload.description the new description of the file
         * @returns {boolean} true if the file was updated successfully, false otherwise
         */
        async updateFile(payload){
            let fileObj = payload.file
            let success = false

            try
            {

                let result = await callMethod({
                    role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
                    methodName: 'setDescription',
                    parameters: [fileObj.id, payload.description]
                })

                if (result.message)
                {
                    let text = result.message == 'locked'
                        ? i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FILE_ERROR_LOCKED
                        : i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR

                    AmetysFront.Event.fire('loaderFail', {
                        title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FILE_ERROR_UPDATE,
                        text: text,
                        details: null
                    })
                }
                else
                {
                    success = true
                    this._updateFile({id: fileObj.id, data: result})
                }
            }
            catch (e)
            {
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FILE_ERROR_UPDATE,
                    text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
                    details: null
                })
            }
            return success
        },

        /**
         * Create a token for a file.
         * @param {String} fileId the id of the file to create a token for.
         */
        async createTokenForFile(fileId)
        {
            try
            {
                let result = await callMethod(
                {
                    role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
                    methodName: 'createPublicLinkTokenForFile',
                    parameters: [fileId]
                })
                this._updateFile({id: fileId, data: result})
            }
            catch (e)
            {
                AmetysFront.Event.fire('loaderFail',
                {
                    title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FILE_TOKEN_GENERATION_ERROR,
                    text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
                    details: null
                })
                throw e
            }
        },

        /**
         * Delete a token of a file.
         * @param {String} fileId the id of the file to delete the token.
         */
        async deleteTokenByFileId(fileId)
        {
            try
            {
                let result = await callMethod(
                {
                    role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
                    methodName: 'deleteTokenByFileId',
                    parameters: [fileId]
                })
                console.log(result);

                this._updateFile({id: fileId, data: result})

            }
            catch (e)
            {
                AmetysFront.Event.fire('loaderFail',
                {
                    title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FILE_TOKEN_REVOCATION_ERROR,
                    text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
                    details: null
                })
                throw e
            }
        }
    }
})
