/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
import { defineStore } from 'pinia'
import { useGlobalUserPreferencesStore } from '@common/store/user-preferences/globalUserPreferences'

/**
 * Pinia store for managing user preferences concerning file-manager.
 */
export const useUserPreferencesStore = defineStore('userPreferences', {
    getters:
    {
        /**
         * Returns the file manager user preferences.
         */
        fileManagerPreferences: () =>
        {
            const globalUserPreferencesStore = useGlobalUserPreferencesStore();
            return globalUserPreferencesStore.getUserPreferences('file-manager');
        },
        /**
         * Returns true if the user prefers grid view, false otherwise.
         */
        displayGrid: (state) =>
        {
            return state.fileManagerPreferences.filesAndFolderDisplayPreference == "grid";
        },
        /**
         * Returns true if the user prefers list view, false otherwise.
         */
        displayList: (state) =>
        {
            return state.fileManagerPreferences.filesAndFolderDisplayPreference == "list";
        },
    },
    actions:
    {
        /**
         * Sets the user preference for displaying files in grid or list view.
         * @param {boolean} displayGrid - True for grid view, false for list view.
         */
        setDisplayGridPreferences(filesAndFolderDisplayPreference)
        {
            const globalUserPreferencesStore = useGlobalUserPreferencesStore()
            let preferences = this.fileManagerPreferences;
            preferences.filesAndFolderDisplayPreference = filesAndFolderDisplayPreference
            globalUserPreferencesStore.setModulePreferences('file-manager', preferences)
        },

        /**
         * Loads the user preference for displaying files in grid or list view from the server.
         */
        async loadFileManagerPreferences()
        {
            const globalUserPreferencesStore = useGlobalUserPreferencesStore()
            await globalUserPreferencesStore.loadModulePreferences('file-manager', {filesAndFolderDisplayPreference: "grid"})
        },

    }
})
