/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.activities.activitystream;

import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.right.RightManager;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.userpref.UserPreferencesException;
import org.ametys.core.userpref.UserPreferencesManager;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.explorer.resources.ModifiableResourceCollection;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.activities.Activity;
import org.ametys.plugins.repository.activities.ActivityHelper;
import org.ametys.plugins.repository.activities.ActivityTypeExpression;
import org.ametys.plugins.repository.activities.ActivityTypeExtensionPoint;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.DateExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.ExpressionContext;
import org.ametys.plugins.repository.query.expression.OrExpression;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.plugins.workspaces.activities.AbstractWorkspacesActivityType;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModule;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Component gathering methods for the activity stream service
 */
public class ActivityStreamClientInteraction extends AbstractLogEnabled implements Component, Serviceable
{
    /** The Avalon role */
    public static final String ROLE = ActivityStreamClientInteraction.class.getName();
    
    /** the user preferences context for activity stream */
    public static final String ACTIVITY_STREAM_USER_PREF_CONTEXT = "/workspaces/activity-stream";
    
    /** the id of user preferences for the last update of activity stream*/
    public static final String ACTIVITY_STREAM_USER_PREF_LAST_UPDATE = "lastUpdate";

    private ProjectManager _projectManager;
    private ActivityTypeExtensionPoint _activityTypeExtensionPoint;

    private CurrentUserProvider _currentUserProvider;

    private RightManager _rightManager;

    private UserPreferencesManager _userPrefManager;

    private AmetysObjectResolver _resolver;

    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _projectManager = (ProjectManager) serviceManager.lookup(ProjectManager.ROLE);
        _rightManager = (RightManager) serviceManager.lookup(RightManager.ROLE);
        _activityTypeExtensionPoint = (ActivityTypeExtensionPoint) serviceManager.lookup(ActivityTypeExtensionPoint.ROLE);
        _currentUserProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
        _userPrefManager = (UserPreferencesManager) serviceManager.lookup(UserPreferencesManager.ROLE);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
    }
    
    /**
     * Get the activities of the given projects and of the given event types
     * @param projectNames the names of the projects. Can not be null.
     * @param filterEventTypes the type of events to retain. Can be empty to get all activities.
     * @param limit The max number of activities
     * @return the retained activities
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public List<Map<String, Object>> getActivities(List<String> projectNames, List<String> filterEventTypes, int limit)
    {
        return getActivities(projectNames, filterEventTypes, null, null, null, limit);
    }
    
    /**
     * Get the activities of the given projects and of the given event types
     * @param projectNames the names of the projects. Can not be null.
     * @param filterEventTypes the type of events to retain. Can be empty to get all activities.
     * @param fromDate To get activities after the given date. Can be null.
     * @param untilDate To get activities before the given date. Can be null.
     * @param pattern A filter pattern. Can be null or empty
     * @param limit The max number of activities
     * @return the retained activities
     */
    public List<Map<String, Object>> getActivities(List<String> projectNames, List<String> filterEventTypes, ZonedDateTime fromDate, ZonedDateTime untilDate, String pattern, int limit)
    {
        List<Map<String, Object>> mergedActivities = new ArrayList<>();
        
        if (projectNames.isEmpty())
        {
            return mergedActivities;
        }
        
        try (AmetysObjectIterable<Activity> activitiesIterable = _getActivities(projectNames, filterEventTypes, fromDate, untilDate, pattern))
        {
            if (activitiesIterable != null)
            {
                // FIXME Add 20% to requested limit to take into account merge of activities
                List<Activity> activities = activitiesIterable.stream().limit(Math.round(limit + limit * 0.5)).toList();
                
                // FIXME After merge, the number of activities could be lower than limit
                mergedActivities.addAll(_activityTypeExtensionPoint.mergeActivities(activities));
            }
        }
        
        return mergedActivities.stream().limit(limit).toList();
    }
    
    private AmetysObjectIterable<Activity> _getActivities(List<String> projectNames, List<String> filterEventTypes, ZonedDateTime fromDate, ZonedDateTime untilDate, String pattern)
    {
        OrExpression exprs = new OrExpression();
        
        Set<String> allAllowedEventTypes = new HashSet<>();
        
        for (String projectName : projectNames)
        {
            Project project = _projectManager.getProject(projectName);
            
            Set<String> allowedEventTypes = _getAllowedEventTypesByProject(project);
            
            if (filterEventTypes != null && filterEventTypes.size() > 0)
            {
                allowedEventTypes.retainAll(filterEventTypes);
            }
            
            allAllowedEventTypes.addAll(allowedEventTypes);
            
            if (allowedEventTypes.size() > 0)
            {
                Expression activityTypeExpr = new ActivityTypeExpression(Operator.EQ, allowedEventTypes.toArray(new String[allowedEventTypes.size()]));
                Expression projectExpr = new StringExpression(AbstractWorkspacesActivityType.PROJECT_NAME, Operator.EQ, projectName);
                
                exprs.add(new AndExpression(activityTypeExpr, projectExpr));
            }
        }
        
        if (!exprs.isEmpty())
        {
            AndExpression finalExpr = new AndExpression();
            
            finalExpr.add(exprs);
            
            if (untilDate != null)
            {
                finalExpr.add(new DateExpression("date", Operator.LT, untilDate));
            }
            
            if (fromDate != null)
            {
                finalExpr.add(new DateExpression("date", Operator.GT, fromDate));
            }
            
            if (StringUtils.isNotEmpty(pattern))
            {
                OrExpression patternExprs = new OrExpression();
                
                patternExprs.add(new StringExpression(AbstractWorkspacesActivityType.PROJECT_TITLE, Operator.WD, pattern, ExpressionContext.newInstance().withCaseInsensitive(true)));
                
                for (String allowedEventType : allAllowedEventTypes)
                {
                    _activityTypeExtensionPoint.getActivityTypes(allowedEventType)
                            .stream()
                            .filter(AbstractWorkspacesActivityType.class::isInstance)
                            .map(AbstractWorkspacesActivityType.class::cast)
                            .forEach(wsActivityType -> {
                                Expression patternExpr = wsActivityType.getFilterPatternExpression(pattern);
                                if (patternExpr != null)
                                {
                                    patternExprs.add(patternExpr);
                                }
                            });
                }
                
                finalExpr.add(patternExprs);
            }
            
            String xpathQuery = ActivityHelper.getActivityXPathQuery(finalExpr);
            return _resolver.query(xpathQuery);
        }
        
        
        return null;
    }
    
    /**
     * Get the date of last activity regardless the current user's rights
     * @param projectName The project's name
     * @param excludeActivityTypes the types of activity to ignore from this search
     * @return the date of last activity or null if no activity found or an error occurred
     */
    public ZonedDateTime getDateOfLastActivity(String projectName, List<String> excludeActivityTypes)
    {
        AndExpression expressions = new AndExpression();
        
        for (String eventType : excludeActivityTypes)
        {
            expressions.add(new ActivityTypeExpression(Operator.NE, eventType));
        }
        
        return _getDateOfLastActivity(projectName, expressions);
    }

    /**
     * Get the date of last activity regardless the current user's rights
     * @param projectName The project's name
     * @param includeActivityTypes the types of activity to ignore from this search
     * @return the date of last activity or null if no activity found or an error occurred
     */
    public ZonedDateTime getDateOfLastActivityByActivityType(String projectName, Collection<String> includeActivityTypes)
    {
        OrExpression expressions = new OrExpression();
        
        for (String eventType : includeActivityTypes)
        {
            expressions.add(new ActivityTypeExpression(Operator.EQ, eventType));
        }
        
        return _getDateOfLastActivity(projectName, expressions);
    }
    
    private ZonedDateTime _getDateOfLastActivity(String projectName, Expression eventTypesExpression)
    {
        Expression projectNameExpression = new StringExpression("projectName", Operator.EQ, projectName);
        
        Expression eventExpr = new AndExpression(projectNameExpression, eventTypesExpression);
        
        String xpathQuery = ActivityHelper.getActivityXPathQuery(eventExpr);
        AmetysObjectIterable<Activity> activities = _resolver.query(xpathQuery);
        
        for (Activity activity: activities)
        {
            return activity.getDate();
        }
        
        return null;
    }
    
    /**
     * Get the list of allowed event types for the given projects
     * @param projects The projects
     * @return The allowed event types
     */
    public Set<String> getAllowedEventTypes (Set<Project> projects)
    {
        Set<String> allowedTypes = new HashSet<>();
        
        for (Project project : projects)
        {
            allowedTypes.addAll(_getAllowedEventTypesByProject(project));
        }
        
        return allowedTypes;
    }
    
    // FIXME temporary method
    // The allowed types are hard coded according the user access on root modules.
    private Set<String> _getAllowedEventTypesByProject (Project project)
    {
        Set<String> allowedTypes = new HashSet<>();
        
        for (WorkspaceModule moduleManager : _projectManager.getModules(project))
        {
            ModifiableResourceCollection moduleRoot = moduleManager.getModuleRoot(project, false);
            if (moduleRoot != null && _rightManager.currentUserHasReadAccess(moduleRoot))
            {
                allowedTypes.addAll(moduleManager.getAllowedEventTypes());
            }
        }
        
        return allowedTypes;
    }
    
    /**
     * Get the number of unread events for the current user
     * @return the number of unread events or -1 if user never read events
     */
    @Callable (rights = Callable.NO_CHECK_REQUIRED)
    public long getNumberOfUnreadActivitiesForCurrentUser()
    {
        ZonedDateTime lastUpdate = _getLastReadDate();
        if (lastUpdate != null)
        {
            Set<Project> userProjects = _getProjectsForCurrentUser(null);
            List<String> projectNames = transformProjectsToName(userProjects);
            Set<String> allowedEventTypes = getAllowedEventTypes(userProjects);
            
            AmetysObjectIterable<Activity> activities = _getActivities(projectNames, new ArrayList<>(allowedEventTypes), lastUpdate, null, null);
            return activities != null ? activities.getSize() : -1;
        }
        return -1;
    }
    
    /**
     * Get the activities for the current user with the allowed event types get from the user projects.
     * @param limit The max number of results
     * @return The activities for the user projects
     */
    public List<Map<String, Object>> getActivitiesForCurrentUser(int limit)
    {
        return getActivitiesForCurrentUser((String) null, null, null, limit);
    }
    
    /**
     * Get the activities for the current user with the allowed event types get from the user projects.
     * @param pattern Pattern to search on activity. Can null or empty to not filter on pattern.
     * @param activityTypes the type of activities to retrieve. Can null or empty to not filter on activity types.
     * @param categories the categories of projects to retrieve. Can null or empty to not filter on themes.
     * @param limit The max number of results
     * @return The activities for the user projects
     */
    public List<Map<String, Object>> getActivitiesForCurrentUser(String pattern, Set<String> categories, Set<String> activityTypes, int limit)
    {
        return getActivitiesForCurrentUser(pattern,  categories, activityTypes, null, null, limit);
    }
    
    /**
     * Get the activities for the current user with the allowed event types get from the user projects.
     * @param pattern Pattern to search on activity. Can null or empty to not filter on pattern.
     * @param activityTypes the type of activities to retrieve. Can null or empty to not filter on activity types.
     * @param categories the categories of projects to retrieve. Can null or empty to not filter on themes.
     * @param fromDate To get activities after the given date. Can be null.
     * @param untilDate To get activities before the given date. Can be null.
     * @param limit The max number of results
     * @return The activities for the user projects
     */
    public List<Map<String, Object>> getActivitiesForCurrentUser(String pattern, Set<String> categories, Set<String> activityTypes, ZonedDateTime fromDate, ZonedDateTime untilDate, int limit)
    {
        Set<Project> userProjects = _getProjectsForCurrentUser(categories);
        return getActivitiesForCurrentUser(userProjects, activityTypes, fromDate, untilDate, pattern, limit);
    }
    
    /**
     * Get the activities for the current user with the allowed event types get from the given projects.
     * @param projects the projects
     * @param activityTypes the type of activities to retrieve. Can null or empty to not filter on activity types.
     * @param fromDate To get activities after the given date. Can be null.
     * @param untilDate To get activities before the given date. Can be null.
     * @param pattern Pattern to search on activity. Can null or empty to not filter on pattern.
     * @param limit The max number of results
     * @return The activities for the user projects
     */
    public List<Map<String, Object>> getActivitiesForCurrentUser(Set<Project> projects, Set<String> activityTypes, ZonedDateTime fromDate, ZonedDateTime untilDate, String pattern, int limit)
    {
        List<String> projectNames = transformProjectsToName(projects);
        Set<String> allowedActivityTypes = getAllowedEventTypes(projects);
        
        if (activityTypes != null && activityTypes.size() > 0)
        {
            allowedActivityTypes.retainAll(activityTypes);
        }
        
        List<Map<String, Object>> activities = getActivities(projectNames, new ArrayList<>(allowedActivityTypes), fromDate, untilDate, pattern, limit);
        
        // Add a parameter representing the date in the ISO 8601 format
        activities.stream().forEach(activity ->
        {
            String eventDate = (String) activity.get("date");
            
            ZonedDateTime lastReadDate = _getLastReadDate();
            if (lastReadDate != null)
            {
                activity.put("unread", DateUtils.parseZonedDateTime(eventDate).compareTo(lastReadDate) > 0);
            }
            // start date
            activity.put("date-iso", eventDate);
            
            // optional end date
            String endDate = (String) activity.get("endDate");
            if (endDate != null)
            {
                activity.put("end-date-iso", endDate);
            }
        });
        
        return activities;
    }
    
    private ZonedDateTime _getLastReadDate()
    {
        UserIdentity user = _currentUserProvider.getUser();
        try
        {
            return _userPrefManager.getUserPreferenceAsDate(user, ACTIVITY_STREAM_USER_PREF_CONTEXT, Map.of(), ACTIVITY_STREAM_USER_PREF_LAST_UPDATE);
        }
        catch (UserPreferencesException e)
        {
            getLogger().warn("Unable to get last unread events date from user preferences", e);
            return null;
        }
    }
    
    private Set<Project> _getProjectsForCurrentUser(Set<String> filteredCategories)
    {
        UserIdentity user = _currentUserProvider.getUser();
        
        Predicate<Project> matchCategories = p -> filteredCategories == null || filteredCategories.isEmpty() || !Collections.disjoint(p.getCategories(), filteredCategories);
        
        return _projectManager.getUserProjects(user).keySet()
                   .stream()
                   .filter(matchCategories)
                   .collect(Collectors.toSet());
    }
    
    private List<String> transformProjectsToName(Set<Project> userProjects)
    {
        return userProjects.stream()
            .map(p -> p.getName())
            .collect(Collectors.toList());
    }
}
