/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.activities.documents;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.jcr.RepositoryException;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.core.observation.Event;
import org.ametys.plugins.explorer.ObservationConstants;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.explorer.resources.ResourceCollection;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.activities.Activity;
import org.ametys.plugins.repository.activities.ActivityType;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.ExpressionContext;
import org.ametys.plugins.repository.query.expression.OrExpression;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.plugins.workspaces.activities.AbstractWorkspacesActivityType;
import org.ametys.plugins.workspaces.documents.DocumentWorkspaceModule;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModuleExtensionPoint;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * {@link ActivityType} implementation for documents-related activities
 */
public abstract class AbstractDocumentsActivityType extends AbstractWorkspacesActivityType
{
    /** data name for the parent folder path in explorer */
    public static final String PARENT_FOLDER_PATH = "parentPath";

    /** data name for the document parent folder id */
    public static final String PARENT_FOLDER_ID = "parentFolderId";

    /** data name for the document parent folder name */
    public static final String PARENT_FOLDER = "parentFolder";

    /** data name for the activity file */
    public static final String FILE_DATA_NAME = "file";
    
    /** data name for the activity files */
    public static final String FILES_DATA_NAME = "files";

    /** Constant for event's category */
    public static final String EVENT_CATEGORY_DOCUMENTS = "documents";
    
    /** Resolver for Ametys objects */
    protected AmetysObjectResolver _ametysObjectResolver;
    
    /** Document Workspace Module */
    protected DocumentWorkspaceModule _documentModule;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _ametysObjectResolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        WorkspaceModuleExtensionPoint moduleManagerEP = (WorkspaceModuleExtensionPoint) serviceManager.lookup(WorkspaceModuleExtensionPoint.ROLE);
        _documentModule = moduleManagerEP.getModule(DocumentWorkspaceModule.DOCUMENT_MODULE_ID);
    }
    
    @Override
    public void setAdditionalActivityData(Activity activity, Map<String, Object> parameters) throws RepositoryException
    {
        super.setAdditionalActivityData(activity, parameters);
        
        activity.setValue(PROJECT_CATEGORY, EVENT_CATEGORY_DOCUMENTS);
        
        String folderId = (String) parameters.get(ObservationConstants.ARGS_PARENT_ID);
        ResourceCollection parentFolder = null;
        if (folderId == null)
        {
            String resourceId = (String) parameters.get(ObservationConstants.ARGS_ID);
            Resource resource = _ametysObjectResolver.resolveById(resourceId);
            parentFolder = resource.getParent();
            folderId = parentFolder.getId();
        }
        else
        {
            parentFolder = _ametysObjectResolver.resolveById(folderId);
        }
        activity.setValue(PARENT_FOLDER_ID, folderId);
        
        // Parent name
        String parentFolderName = parentFolder.getName();
        if ("ametys-internal:resources".equals(parentFolderName))
        {
            parentFolderName = _i18nUtils.translate(new I18nizableText("plugin.explorer", "PLUGINS_EXPLORER_ROOT_NODE"));
        }
        activity.setValue(PARENT_FOLDER, parentFolderName);
        
        activity.setValue(PARENT_FOLDER_PATH, parentFolder.getExplorerPath());
    }
    
    @Override
    public Map<String, Object> additionnalDataToJSONForClient(Activity activity)
    {
        Map<String, Object> json = super.additionnalDataToJSONForClient(activity);
        String parentId = activity.getValue(PARENT_FOLDER_ID);
        
        String projectName = activity.getValue(PROJECT_NAME);
        Project project = _projectManager.getProject(projectName);
        if (project != null)
        {
            json.put("parentFolderUrl", _documentModule.getFolderUri(project, parentId));
        }
        
        return json;
    }
    
    @Override
    public boolean isMergeable(Activity activity1, Activity activity2)
    {
        if (!super.isMergeable(activity1, activity2))
        {
            return false;
        }
        String parentFolder1 = activity1.getValue(PARENT_FOLDER_ID);
        String parentFolder2 = activity2.getValue(PARENT_FOLDER_ID);
        
        return parentFolder1 != null && parentFolder2 != null && parentFolder1.equals(parentFolder2);
    }
    
    @Override
    public Expression getFilterPatternExpression(String pattern)
    {
        Expression folderExpr = new StringExpression(PARENT_FOLDER, Operator.WD, pattern, ExpressionContext.newInstance().withCaseInsensitive(true));
        Expression filesExpr = new StringExpression(FILES_DATA_NAME + "/*/name", Operator.WD, pattern, ExpressionContext.newInstance().withCaseInsensitive(true));
        Expression fileExpr = new StringExpression(FILE_DATA_NAME + "/name", Operator.WD, pattern, ExpressionContext.newInstance().withCaseInsensitive(true));
        
        return new OrExpression(folderExpr, filesExpr, fileExpr);
    }
    
    @SuppressWarnings("unchecked")
    @Override
    public Map<String, Object> mergeActivities(List<Activity> activities)
    {
        Map<String, Object> mergedActivities = super.mergeActivities(activities);
        
        List<Map<String, Object>> files = new ArrayList<>();
        
        List<String> knownFiles = new ArrayList<>();
        
        for (Activity activity : activities)
        {
            Map<String, Object> json = activity.toJSONForClient();
            if (json.containsKey(FILE_DATA_NAME))
            {
                Map<String, Object> file = (Map<String, Object>) json.get(FILE_DATA_NAME);
                if (!knownFiles.contains(file.get("id")))
                {
                    files.add((Map<String, Object>) json.get(FILE_DATA_NAME));
                    knownFiles.add((String) file.get("id"));
                }
            }
            else if (json.containsKey(FILES_DATA_NAME))
            {
                files.addAll((List<Map<String, Object>>) json.get("files"));
            }
        }
        
        if (files.size() > 0)
        {
            mergedActivities.put("files", files);
        }
        
        mergedActivities.put("amount", files.size());
        
        return mergedActivities;
    }
    
    @Override
    public Project getProjectFromEvent(Event event)
    {
        Map<String, Object> args = event.getArguments();
        String parentId = (String) args.get(ObservationConstants.ARGS_PARENT_ID);
        if (parentId != null)
        {
            return getParentProject(_resolver.resolveById(parentId));
        }
        else
        {
            String resourceId = (String) args.get(ObservationConstants.ARGS_ID);
            Resource resource = _resolver.resolveById(resourceId);
            return getParentProject(resource.getParent());
        }
    }
}
