/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.activities.documents;

import java.util.List;
import java.util.stream.Stream;

import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.activities.Activity;
import org.ametys.plugins.repository.activities.ActivityType;
import org.ametys.plugins.workspaces.activities.AbstractWorkspacesActivityNotifier;
import org.ametys.plugins.workspaces.activities.documents.ResourceReferenceElementType.ResourceReference;

/**
 * Class representing a activity notifier for documents workspaces
 */
public class DocumentsActivityNotifier extends AbstractWorkspacesActivityNotifier
{
    public boolean supports(ActivityType activityType)
    {
        return activityType instanceof DocumentCommentedActivityType
                || activityType instanceof ResourceCreatedOrUpdatedActivityType
                || activityType instanceof ResourceRenamedActivityType;
    }
    
    @Override
    public String getMailBodyURI(Activity activity)
    {
        return "cocoon://_plugins/workspaces/notification-mail-resource";
    }
    
    @Override
    protected String _getSubjectI18nKey(Activity activity)
    {
        String i18nKey = super._getSubjectI18nKey(activity);
        if (activity.getActivityType() instanceof AbstractDocumentsActivityType && activity.hasValue(AbstractDocumentsActivityType.FILES_DATA_NAME))
        {
            ResourceReference[] refs = activity.getValue(AbstractDocumentsActivityType.FILES_DATA_NAME);
            if (refs.length > 1)
            {
                i18nKey += "_MULTIPLE";
            }
        }
        
        return i18nKey;
    }
    
    @Override
    public List<String> getSubjectI18nParams(Activity activity)
    {
        List<String> i18nParams = super.getSubjectI18nParams(activity);
        
        if (activity.hasValue(AbstractDocumentsActivityType.FILE_DATA_NAME))
        {
            ResourceReference ref = activity.getValue(AbstractDocumentsActivityType.FILE_DATA_NAME);
            i18nParams.add(ref.oldName() != null ? ref.oldName() : ref.name()); // {1}
        }
        else if (activity.hasValue(AbstractDocumentsActivityType.FILES_DATA_NAME))
        {
            ResourceReference[] refs = activity.getValue(AbstractDocumentsActivityType.FILES_DATA_NAME);
            if (refs.length > 1)
            {
                i18nParams.add(String.valueOf(refs.length)); // {1}
            }
            else
            {
                i18nParams.add(Stream.of(refs).findFirst().map(ResourceReference::name).orElse("")); // {1}
            }
        }
        return i18nParams;
    }
    
    @Override
    public AmetysObject getTargetAmetysObject(Activity activity)
    {
        return _resolver.resolveById(activity.getValue(AbstractDocumentsActivityType.PARENT_FOLDER_ID));
    }
}
