/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.activities.documents;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.version.Version;
import javax.jcr.version.VersionManager;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.core.observation.Event;
import org.ametys.plugins.explorer.ObservationConstants;
import org.ametys.plugins.explorer.resources.jcr.JCRResource;
import org.ametys.plugins.repository.activities.Activity;
import org.ametys.plugins.repository.activities.ActivityType;
import org.ametys.plugins.workspaces.activities.documents.ResourceReferenceElementType.ResourceReference;
import org.ametys.plugins.workspaces.project.notification.ResourceNotifierHelper;
import org.ametys.plugins.workspaces.project.objects.Project;

/**
 * {@link ActivityType} implementation for the renaming of a resource
 */
public class ResourceCreatedOrUpdatedActivityType extends AbstractDocumentsActivityType
{

    private ResourceNotifierHelper _resourceNotifierHelper;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _resourceNotifierHelper = (ResourceNotifierHelper) serviceManager.lookup(ResourceNotifierHelper.ROLE);
    }
    
    @Override
    public boolean supports(Event event)
    {
        return !event.getId().equals(ObservationConstants.EVENT_RESOURCE_UPDATED)
            || _isResourceRecentlyModified(event);
    }

    private boolean _isResourceRecentlyModified(Event event)
    {
        String resourceId = (String) event.getArguments().getOrDefault(ObservationConstants.ARGS_ID, null);
        return !_resourceNotifierHelper.resourceRecentlyModified(resourceId);
    }
    
    @SuppressWarnings("unchecked")
    @Override
    public void setAdditionalActivityData(Activity activity, Map<String, Object> parameters) throws RepositoryException
    {
        super.setAdditionalActivityData(activity, parameters);
        
        // File data
        Set<String> resourceIds = new HashSet<>();
        if (parameters.containsKey(ObservationConstants.ARGS_RESOURCES))
        {
            resourceIds.addAll(((Map<String, Object>) parameters.get(ObservationConstants.ARGS_RESOURCES)).keySet());
        }
        else
        {
            resourceIds.add((String) parameters.get(ObservationConstants.ARGS_ID));
        }
        
        ResourceReference[] references = new ResourceReference[resourceIds.size()];
        int i = 0;
        for (String resourceId : resourceIds)
        {
            JCRResource resource = (JCRResource) _ametysObjectResolver.resolveById(resourceId);
            Node resourceNode = resource.getNode();
            
            VersionManager versionManager = resourceNode.getSession().getWorkspace().getVersionManager();
            Version baseVersion = versionManager.getBaseVersion(resourceNode.getPath());
            
            // Store a reference to the version of the node instead of on the node itself
            references[i] = new ResourceReference(resourceId, resource.getName(), null, resource.getMimeType(), baseVersion.getName());
            i++;
        }
        activity.setValue(AbstractDocumentsActivityType.FILES_DATA_NAME, references);
        
    }
    
    @Override
    public Map<String, Object> additionnalDataToJSONForClient(Activity activity)
    {
        Map<String, Object> json = super.additionnalDataToJSONForClient(activity);
        
        String projectName = activity.getValue(PROJECT_NAME);
        Project project = _projectManager.getProject(projectName);

        ResourceReference[] refs = activity.getValue(AbstractDocumentsActivityType.FILES_DATA_NAME);
        if (refs.length == 1)
        {
            String id = refs[0].id();
            json.put("fileUrl", _documentModule.getFileUri(project, id));
        }
        else
        {
            json.put("fileUrl", _documentModule.getModuleUrl(project));
        }
        return json;
    }
}
