/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.activities.tasks;

import java.util.Map;

import javax.jcr.RepositoryException;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.repository.comment.Comment;
import org.ametys.cms.repository.mentions.MentionUtils;
import org.ametys.plugins.repository.activities.Activity;
import org.ametys.plugins.repository.activities.ActivityType;
import org.ametys.plugins.workspaces.ObservationConstants;
import org.ametys.plugins.workspaces.tasks.Task;

/**
 * {@link ActivityType} implementation for the commented task activities
 */
public class TasksCommentedActivityType extends TasksActivityType
{

    /** data name for storing the task creation date */
    public static final String CREATION_FATE = "creationDate";
    /** data name for storing the author of the task */
    public static final String TASK_AUTHOR = "taskAuthor";
    /** data name for storing the comment */
    public static final String COMMENT_SUMMARY = "commentSummary";
    /** data name for the comment creation date */
    public static final String COMMENT_CREATIONDATE = "commentCreationDate";

    /** The mentions helper */
    protected MentionUtils _mentionUtils;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _mentionUtils = (MentionUtils) manager.lookup(MentionUtils.ROLE);
    }
    
    @Override
    public void setAdditionalActivityData(Activity activity, Map<String, Object> parameters) throws RepositoryException
    {
        super.setAdditionalActivityData(activity, parameters);

        String commentId = (String) parameters.get(ObservationConstants.ARGS_TASK_COMMENT_ID);
        
        Task task = (Task) parameters.get(ObservationConstants.ARGS_TASK);
        
        Comment comment = task.getComment(commentId);
        
        activity.setValue(COMMENT_CREATIONDATE, comment.getCreationDate());
        activity.setValue(CREATION_FATE, task.getCreationDate(), "datetime");
        activity.setValue(TASK_AUTHOR, task.getAuthor(), "user");
        
        String commentValue = (String) parameters.get(ObservationConstants.ARGS_TASK_COMMENT);
        activity.setValue(COMMENT_SUMMARY, StringUtils.abbreviate(commentValue, "…", 150));
    }
    
    
    @Override
    public boolean isMergeable(Activity activity1, Activity activity2)
    {
        // We don't want to merge for the moment as it would require to merge between different author
        // to recreate a conversation otherwise it would just be useless.
        return false;
    }
}
