/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.archive;

import java.io.IOException;
import java.nio.file.Path;
import java.util.zip.ZipOutputStream;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.RepositoryException;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.xml.sax.ContentHandler;

import org.ametys.plugins.contentio.archive.DefaultPluginArchiver;
import org.ametys.plugins.contentio.archive.ImportReport;
import org.ametys.plugins.contentio.archive.Merger;
import org.ametys.plugins.contentio.archive.SystemViewHandler;
import org.ametys.plugins.contentio.archive.ZipEntryHelper;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.workspaces.project.objects.Project;

/**
 * Archiver handling the data stored by the workspaces plugin.
 * Mainly this archiver will separate each projects in its own directory.
 * Each project folder consisting of different directory for each module.
 * And the data in the document module will be stored as file tree allowing easy access to each document
 */
public class WorkspacePluginArchiver extends DefaultPluginArchiver
{
    private static final String __PROJECTS_NODE_NAME = "projects";
    /** the ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** the project archiver helper */
    protected ProjectArchiverHelper _projectArchiverHelper;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _projectArchiverHelper = (ProjectArchiverHelper) manager.lookup(ProjectArchiverHelper.ROLE);
    }
    
    @Override
    protected ContentHandler getSystemViewHandler(ContentHandler initialHandler)
    {
        return new SystemViewHandler(initialHandler, name -> __PROJECTS_NODE_NAME.equals(name), __ -> false);
    }
    
    @Override
    public void export(String pluginName, Node pluginNode, ZipOutputStream zos, String prefix) throws IOException
    {
        // export every thing except the projets node
        super.export(pluginName, pluginNode, zos, prefix);
        
        try
        {
            // then the contents if any, one by one
            if (pluginNode.hasNode(__PROJECTS_NODE_NAME))
            {
                Node projectsNode = pluginNode.getNode(__PROJECTS_NODE_NAME);
                NodeIterator it = projectsNode.getNodes();
                while (it.hasNext())
                {
                    Node projectNode = it.nextNode();
                    Project project = _resolver.resolve(projectNode, false);
                    
                    _projectArchiverHelper.exportProject(project, zos, prefix + "/" + __PROJECTS_NODE_NAME + "/" + project.getName() + "/");
                }
            }
        }
        catch (Exception e)
        {
            throw new RuntimeException("Unable to archive plugin " + pluginName, e);
        }
    }
    
    @Override
    public ImportReport partialImport(String pluginName, Node allPluginsNode, Path zipPath, String zipPluginEntryPath, Merger merger) throws IOException
    {
        ImportReport report = super.partialImport(pluginName, allPluginsNode, zipPath, zipPluginEntryPath, merger);
        
        String baseImportProjectPath = zipPluginEntryPath + "/" + __PROJECTS_NODE_NAME;
        if (ZipEntryHelper.zipEntryFolderExists(zipPath, baseImportProjectPath))
        {
            try
            {
                // The plugin node is created by the call to super
                Node pluginNode = allPluginsNode.getNode(pluginName);
                report.addFrom(_projectArchiverHelper.importProjects(pluginNode, zipPath, baseImportProjectPath, merger));
            }
            catch (RepositoryException e)
            {
                throw new IOException("Could not get plugin node for " + pluginName, e);
            }
        }
        else
        {
            getLogger().info("No project node to import. The path '{}!{}' does not exist", zipPath, baseImportProjectPath);
        }
        return report;
    }
}
