/*
 *  Copyright 2014 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.calendars.jcr;

import java.time.ZonedDateTime;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.jcr.Node;

import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.data.ametysobject.ModifiableModelAwareDataAwareAmetysObject;
import org.ametys.cms.data.holder.ModifiableIndexableDataHolder;
import org.ametys.cms.data.holder.impl.DefaultModifiableModelAwareDataHolder;
import org.ametys.plugins.explorer.ExplorerNode;
import org.ametys.plugins.explorer.ModifiableExplorerNode;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.data.repositorydata.ModifiableRepositoryData;
import org.ametys.plugins.repository.data.repositorydata.impl.JCRRepositoryData;
import org.ametys.plugins.repository.jcr.DefaultTraversableAmetysObject;
import org.ametys.plugins.workspaces.calendars.Calendar;
import org.ametys.plugins.workspaces.calendars.WorkflowAwareCalendar;
import org.ametys.plugins.workspaces.calendars.events.CalendarEvent;
import org.ametys.plugins.workspaces.calendars.events.CalendarEventOccurrence;
import org.ametys.plugins.workspaces.project.objects.Project;

/**
 * Default implementation of an {@link Calendar}, backed by a JCR node.<br>
 */
public class JCRCalendar extends DefaultTraversableAmetysObject<JCRCalendarFactory> implements WorkflowAwareCalendar, ModifiableExplorerNode, ModifiableModelAwareDataAwareAmetysObject
{
    /** application id for resources collections. */
    public static final String APPLICATION_ID = "Ametys.plugins.explorer.applications.Calendar";
    
    /** Constants for title Calendar* */
    public static final String CALENDAR_TITLE = "title";
    
    /** Constants for description Calendar* */
    public static final String CALENDAR_DESC = "description";
    
    /** Constants for color Calendar* */
    public static final String CALENDAR_COLOR = "color";
    
    /** Constants for color Visibility* */
    public static final String CALENDAR_VISIBILITY = "visibility";
    
    /** Constants for workflowName Calendar* */
    public static final String CALENDAR_WORKFLOW_NAME = "workflowName";
    
    /** Constants for template description Calendar* */
    public static final String CALENDAR_TEMPLATE_DESC = "template-description";
    
    /** Constants for Calendar ICS token */
    public static final String CALENDAR_ICS_TOKEN = "ics-token";
    
    /**
     * Creates an {@link JCRCalendar}.
     * @param node the node backing this {@link AmetysObject}
     * @param parentPath the parentPath in the Ametys hierarchy
     * @param factory the DefaultAmetysObjectFactory which created the AmetysObject
     */
    public JCRCalendar(Node node, String parentPath, JCRCalendarFactory factory)
    {
        super(node, parentPath, factory);
    }

    public String getIconCls()
    {
        return "calendar";
    }

    public String getApplicationId()
    {
        return APPLICATION_ID;
    }

    public Project getProject()
    {
        return _getFactory().getProjectManager().getParentProject(this);
    }
    
    public String getExplorerPath()
    {
        AmetysObject parent = getParent();
        
        if (parent instanceof ExplorerNode)
        {
            return ((ExplorerNode) parent).getExplorerPath() + "/" + getName();
        }
        else
        {
            return "";
        }
    }
    
    public String getDescription()
    {
        return getValue(CALENDAR_DESC);
    }

    public String getColor()
    {
        return getValue(CALENDAR_COLOR);
    }
    
    public CalendarVisibility getVisibility()
    {
        if (hasValue(CALENDAR_VISIBILITY))
        {
            return CalendarVisibility.valueOf(getValue(CALENDAR_VISIBILITY));
        }
        else
        {
            return CalendarVisibility.PRIVATE;
        }
    }
    
    public String getWorkflowName()
    {
        return getValue(CALENDAR_WORKFLOW_NAME);
    }
    
    public String getTemplateDescription()
    {
        return getValue(CALENDAR_TEMPLATE_DESC, false, StringUtils.EMPTY);
    }
    
    public List<Calendar> getChildCalendars()
    {
        return this.getChildren().stream()
                .filter(Calendar.class::isInstance)
                .map(Calendar.class::cast)
                .toList();
    }
    
    public List<CalendarEvent> getAllEvents()
    {
        return this.getChildren().stream()
                    .filter(CalendarEvent.class::isInstance)
                    .map(CalendarEvent.class::cast)
                    .toList();
    }
    
    public Map<CalendarEvent, List<CalendarEventOccurrence>> getEvents(ZonedDateTime startDate, ZonedDateTime endDate)
    {
        Map<CalendarEvent, List<CalendarEventOccurrence>> events = new HashMap<>();
        
        for (CalendarEvent event : getAllEvents())
        {
            List<CalendarEventOccurrence> occurences = event.getOccurrences(startDate, endDate);
            if (!occurences.isEmpty())
            {
                events.put(event, occurences);
            }
        }
        
        return events;
    }

    public void setDescription(String desc)
    {
        setValue(CALENDAR_DESC, desc);
    }

    public void setColor(String color)
    {
        setValue(CALENDAR_COLOR, color); 
    }
    
    public void setVisibility(CalendarVisibility visibility)
    {
        setValue(CALENDAR_VISIBILITY, visibility.name());
    }

    public void setWorkflowName(String workflowName)
    {
        setValue(CALENDAR_WORKFLOW_NAME, workflowName); 
    }
    
    public void setTemplateDescription(String templateDesc)
    {
        setValue(CALENDAR_TEMPLATE_DESC, templateDesc);
    }
    
    public boolean hasChildExplorerNodes() throws AmetysRepositoryException
    {
        try (AmetysObjectIterable<AmetysObject> children = getChildren())
        {
            for (AmetysObject child : children)
            {
                if (child instanceof ExplorerNode)
                {
                    return true;
                }
            }
    
            return false;
        }
    }
    
    public String getIcsUrlToken()
    {
        return getValue(CALENDAR_ICS_TOKEN);
    }
    
    public void setIcsUrlToken(String token)
    {
        if (token != null)
        {
            setValue(CALENDAR_ICS_TOKEN, token);
        }
        else
        {
            removeValue(CALENDAR_ICS_TOKEN);
        }
    }

    public ModifiableIndexableDataHolder getDataHolder()
    {
        ModifiableRepositoryData repositoryData = new JCRRepositoryData(getNode());
        return new DefaultModifiableModelAwareDataHolder(repositoryData, _getFactory().getCalendarEventModel());
    }
}
