/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.comments;

import java.util.List;
import java.util.Map;

import org.ametys.cms.repository.comment.Comment;
import org.ametys.cms.repository.mentions.MentionUtils;
import org.ametys.core.observation.Event;
import org.ametys.core.user.User;
import org.ametys.plugins.workspaces.ObservationConstants;
import org.ametys.plugins.workspaces.documents.DocumentWorkspaceModule;
import org.ametys.plugins.workspaces.documents.jcr.File;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.runtime.i18n.I18nizableText;
/**
 * Observer to send mails to mentioned users in contributor comments of {@link File}
 */
public class NotifyFileCommentMentionsObserver extends AbstractNotifyWorkspacesCommentMentionsObserver<File>
{
    @Override
    public boolean supports(Event event)
    {
        return event.getId().equals(ObservationConstants.EVENT_RESOURCE_COMMENTED);
    }
    
    @Override
    protected MentionableObject _getMentionableObjectFromArguments(Map<String, Object> arguments) throws Exception
    {
        File file = _getAmetysObjectFromArguments(arguments);
        String commentId = (String) arguments.get(ObservationConstants.ARGS_FILE_COMMENT_ID);
        Comment comment = file.getComment(commentId);
        Project project = _projectManager.getParentProject(file);
        String url = _getAbsoluteUrl(file, project);
        
        return new MentionableObject(
            _userManager.getUser(comment.getAuthor()),
            comment.getContent(),
            MentionUtils.extractMentionedUsersFromSimpleText(comment.getContent()),
            comment.getCreationDate(),
            file,
            new LinkToAmetysObject(url, new I18nizableText("plugin.workspaces", "PROJECT_MAIL_NOTIFICATION_BODY_RESOURCE_BUTTON_TEXT")),
            _getLanguage(project)
        );
    }

    @Override
    protected String _getAmetysObjectTitle(File file)
    {
        return file.getName();
    }
    
    @Override
    protected I18nizableText _getMailMessage(MentionableObject mentionableObject)
    {
        File file = (File) mentionableObject.ametysObject();
        Project project = _projectManager.getParentProject(file);
        User author = mentionableObject.author();
        List<String> i18nParams = List.of(author.getFullName(), _getAmetysObjectTitle(file), project.getTitle());
        return new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_RESOURCE_MENTION_MAIL_MESSAGE", i18nParams);
    }

    @Override
    protected String _getModuleId()
    {
        return DocumentWorkspaceModule.DOCUMENT_MODULE_ID;
    }

    @Override
    protected String getUrl(File ametysObject, Project project)
    {
        return _documentModule.getFileUri(project, ametysObject.getId());
    }
}
