/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.documents;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;

import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.explorer.ExplorerNode;
import org.ametys.plugins.explorer.ObservationConstants;
import org.ametys.plugins.explorer.resources.ModifiableResourceCollection;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.explorer.resources.ResourceCollection;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.data.holder.ModifiableModelAwareDataHolder;
import org.ametys.plugins.workspaces.AbstractWorkspaceModule;
import org.ametys.plugins.workspaces.documents.jcr.FolderFactory;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.util.StatisticColumn;
import org.ametys.plugins.workspaces.util.StatisticsColumnType;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.repository.page.ModifiablePage;
import org.ametys.web.repository.page.ModifiableZone;
import org.ametys.web.repository.page.ModifiableZoneItem;
import org.ametys.web.repository.page.ZoneItem.ZoneType;

import com.google.common.collect.ImmutableSet;

/**
 * Helper component for managing documents
 */
public class DocumentWorkspaceModule extends AbstractWorkspaceModule
{
    /** The id of document module */
    public static final String DOCUMENT_MODULE_ID = DocumentWorkspaceModule.class.getName();
    
    /** Workspaces documents node name */
    public static final String WORKSPACES_DOCUMENTS_NODE_NAME = "documents";

    private static final String __DOCUMENT_NUMBER_HEADER_ID = WORKSPACES_DOCUMENTS_NODE_NAME + "$document_number";
    
    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public String getId()
    {
        return DOCUMENT_MODULE_ID;
    }
    
    @Override
    public String getModuleName()
    {
        return WORKSPACES_DOCUMENTS_NODE_NAME;
    }
    
    public int getOrder()
    {
        return ORDER_DOCUMENTS;
    }
    
    @Override
    protected String getModulePageName()
    {
        return "documents";
    }
    
    public I18nizableText getModuleTitle()
    {
        return new I18nizableText("plugin." + _pluginName, "PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_DOCUMENT_LABEL");
    }
    public I18nizableText getModuleDescription()
    {
        return new I18nizableText("plugin." + _pluginName, "PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_DOCUMENT_DESCRIPTION");
    }
    @Override
    protected I18nizableText getModulePageTitle()
    {
        return new I18nizableText("plugin." + _pluginName, "PLUGINS_WORKSPACES_PROJECT_WORKSPACE_PAGE_DOCUMENTS_TITLE");
    }
    
    @Override
    protected void initializeModulePage(ModifiablePage documentPage)
    {
        ModifiableZone defaultZone = documentPage.createZone("default");
        
        String serviceId = "org.ametys.plugins.workspaces.module.Document";
        ModifiableZoneItem defaultZoneItem = defaultZone.addZoneItem();
        defaultZoneItem.setType(ZoneType.SERVICE);
        defaultZoneItem.setServiceId(serviceId);
        
        ModifiableModelAwareDataHolder serviceDataHolder = defaultZoneItem.getServiceParameters();
        serviceDataHolder.setValue("xslt", _getDefaultXslt(serviceId));
    }

    /**
     * Get the URI of a folder in project'site
     * @param project The project
     * @param collectionId The id of collection
     * @return The thread uri
     */
    public String getFolderUri(Project project, String collectionId)
    {
        String moduleUrl = getModuleUrl(project);
        if (moduleUrl != null)
        {
            StringBuilder sb = new StringBuilder();
            sb.append(moduleUrl);
            sb.append("#").append(collectionId);
            
            return sb.toString();
        }
        
        return null;
    }
    
    /**
     * Retrieves the set of general rights used in the document module for the current user
     * @return The map of right data. Keys are the rights id, and values indicates whether the current user has the right or not.
     */
    @Callable (rights = Callable.NO_CHECK_REQUIRED)
    public Map<String, Object> getModuleBaseRights()
    {
        Request request = ContextHelper.getRequest(_context);
        
        String projectName = (String) request.getAttribute("projectName");
        Project project = _projectManager.getProject(projectName);
        
        ModifiableResourceCollection documentRoot = getModuleRoot(project, false);
        
        Map<String, Object> rightsData = new HashMap<>();
        UserIdentity user = _currentUserProvider.getUser();
        
        // Add file / folder
        rightsData.put("add-file", _rightManager.hasRight(user, "Plugin_Explorer_File_Add", documentRoot) == RightResult.RIGHT_ALLOW);
        rightsData.put("add-folder", _rightManager.hasRight(user, "Plugin_Explorer_Folder_Add", documentRoot) == RightResult.RIGHT_ALLOW);
        rightsData.put("add-cmis-folder", _rightManager.hasRight(user, "Plugin_Explorer_CMIS_Add", documentRoot) == RightResult.RIGHT_ALLOW);
        
        // Tags
        rightsData.put("add-tag", _projectRightHelper.canAddTag(project));
        rightsData.put("remove-tag", _projectRightHelper.canRemoveTag(project));
        
        return rightsData;
    }

    @Override
    public Set<String> getAllowedEventTypes()
    {
        return ImmutableSet.of(ObservationConstants.EVENT_RESOURCE_CREATED,
                org.ametys.plugins.workspaces.ObservationConstants.EVENT_RESOURCE_COMMENTED,
                ObservationConstants.EVENT_RESOURCE_RENAMED,
                ObservationConstants.EVENT_RESOURCE_UPDATED,
                ObservationConstants.EVENT_RESOURCE_DELETED);
    }

    @Override
    public Map<String, Object> _getInternalStatistics(Project project, boolean isActive)
    {
        if (isActive)
        {
            ModifiableResourceCollection documentRoot = getModuleRoot(project, false);
            long fileNumber = documentRoot.getChildren()
                    .stream()
                    .mapToLong(child -> _getFileNumber(child))
                    .sum();
            return Map.of(__DOCUMENT_NUMBER_HEADER_ID, fileNumber);
        }
        else
        {
            return Map.of(__DOCUMENT_NUMBER_HEADER_ID, __SIZE_INACTIVE);
        }
    }
    
    private long _getRessourceSize(AmetysObject document)
    {
        if (document instanceof Resource)
        {
            Resource file = (Resource) document;
            return file.getLength();
        }
        else if (document instanceof ResourceCollection)
        {
            ResourceCollection folder = (ResourceCollection) document;
            return folder.getChildren()
                .stream()
                .mapToLong(child -> _getRessourceSize(child))
                .sum();
        }
        return 0;
    }
    
    private long _getFileNumber(AmetysObject document)
    {
        if (document instanceof Resource)
        {
            return 1;
        }
        else if (document instanceof ResourceCollection)
        {
            ResourceCollection folder = (ResourceCollection) document;
            return folder.getChildren()
                .stream()
                .mapToLong(child -> _getFileNumber(child))
                .sum();
        }
        return 0;
    }

    @Override
    public List<StatisticColumn> _getInternalStatisticModel()
    {

        return List.of(new StatisticColumn(__DOCUMENT_NUMBER_HEADER_ID, new I18nizableText("plugin." + _pluginName, "PLUGINS_WORKSPACES_PROJECT_STATISTICS_TOOL_COLUMN_FILE_NUMBER"))
                .withRenderer("Ametys.plugins.workspaces.project.tool.ProjectsGridHelper.renderElements")
                .withType(StatisticsColumnType.LONG)
                .withGroup(GROUP_HEADER_ELEMENTS_ID));
    }
    
    @Override
    protected long _getModuleSize(Project project)
    {
        return getModuleRoot(project, false).getChildren()
            .stream()
            .mapToLong(child -> _getRessourceSize(child))
            .sum();
    }

    @Override
    protected boolean _showModuleSize()
    {
        return true;
    }

    @Override
    public Set<String> getAllEventTypes()
    {
        return Set.of(ObservationConstants.EVENT_RESOURCE_CREATED,
                      org.ametys.plugins.workspaces.ObservationConstants.EVENT_RESOURCE_COMMENTED,
                      ObservationConstants.EVENT_RESOURCE_DELETED,
                      ObservationConstants.EVENT_RESOURCE_MOVED,
                      ObservationConstants.EVENT_RESOURCE_RENAMED,
                      ObservationConstants.EVENT_RESOURCE_UPDATED);
    }

    @Override
    public ModifiableResourceCollection getModuleRoot(Project project, boolean create)
    {
        try
        {
            ExplorerNode projectRootNode = project.getExplorerRootNode();
            
            if (projectRootNode instanceof ModifiableResourceCollection projectRootNodeRc)
            {
                return _getAmetysObject(projectRootNodeRc, getModuleName(), FolderFactory.FOLDER_NODETYPE, create);
            }
            else
            {
                throw new IllegalArgumentException("Error getting the " + getModuleName() + " root node : it should be a ModifiableResourceCollection and not " + projectRootNode.getClass());
            }
        }
        catch (AmetysRepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the " + getModuleName() + " root node.", e);
        }
    }
    
    /**
     * Get the URI of a fileId in project'site
     * @param project The project
     * @param fileId The id of the file
     * @return The fileId uri
     */
    public String getFileUri(Project project, String fileId)
    {
        String moduleUrl = getModuleUrl(project);
        if (moduleUrl != null)
        {
            StringBuilder sb = new StringBuilder();
            sb.append(moduleUrl);
            sb.append("#file-").append(fileId);
            
            return sb.toString();
        }
        
        return null;
    }
}
