/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.documents;

import java.util.HashSet;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.transformation.URIResolver;
import org.ametys.core.authentication.token.AuthenticationTokenManager;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.explorer.resources.ResourceCollection;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.workspaces.project.objects.Project;

/**
 * {@link URIResolver} for webdav "project-resource". <br>
 * These links point to a file from the resources of a project through webdav
 */
public class WebdavProjectResourceURIResolver extends ProjectResourceURIResolver
{
    /** The authentication token manager */
    protected AuthenticationTokenManager _authenticationTokenManager;
    
    /** Current User Provider */
    protected CurrentUserProvider _currentUserProvider;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _authenticationTokenManager = (AuthenticationTokenManager) manager.lookup(AuthenticationTokenManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
    }
    
    @Override
    public String getType()
    {
        return "webdav-project-resource";
    }
    
    @Override
    protected String getUriPrefix(AmetysObject object, boolean download, boolean absolute, boolean internal)
    {
        Project project = null;
        String projectName = null;
        String siteName = null;

        UserIdentity user = _currentUserProvider.getUser();
        long duration = 12 * 60 * 60; // 12h
        boolean autoRenew = true; // Auto Renew the token to be able to continue to edit it
        
        Set<String> contexts = new HashSet<>();
        String path = getPath(object); // Token only usable on this resource
        path = StringUtils.removeStart(path, "/"); // / at the beginning is removed because it is used in a uri path
        contexts.add(path);
        
        if (object instanceof Resource)
        {
            String parentPath = getPath(object.getParent());
            parentPath = StringUtils.removeStart(parentPath, "/").concat("/"); // / at the beginning is removed because it is used in a uri path, / at the end to avoid identical folders
            contexts.add(parentPath);
        }
        String type = "Workspaces-Webdav";
        String comment = "Token created to modify content : '" + object.getId() + "'";
        String token = _authenticationTokenManager.generateToken(user, duration, autoRenew, null, contexts, type, comment);
        
        project = _getProject(object);
        projectName = project.getName();

        siteName = project.getSite().getName();
        
        if (internal)
        {
            return "cocoon://_workspaces/dav/" + projectName + "/" + token;
        }
        else if (absolute)
        {
            return _webPrefixHandler.getAbsoluteUriPrefix(siteName) + "/_workspaces/dav/" + projectName + "/" + token;
        }
        else
        {
            return _webPrefixHandler.getUriPrefix(siteName) + "/_workspaces/dav/" + projectName + "/" + token;
        }
    }
    
    /**
     * Get the path of a {@link Resource} or a {@link ResourceCollection}
     * @param resource the resource to work with
     * @return the path
     */
    protected String getPath(AmetysObject resource)
    {
        String path = null;
        if (resource instanceof Resource)
        {
            path = getResourcePath((Resource) resource);
        }
        else if (resource instanceof ResourceCollection)
        {
            path = ((ResourceCollection) resource).getExplorerPath();
        }
        else
        {
            getLogger().error("Trying to access to something that is not a Resource or a ResourceCollection : " + resource.getId());
        }
        return path;
    }
    
    @Override
    protected String getRealPrefix(Resource resource, String prefix)
    {
        return null;
    }
}
