/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.documents;

import java.util.Date;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.user.CurrentUserProvider;
import org.ametys.plugins.ai.AIHelper;
import org.ametys.plugins.explorer.resources.ModifiableResourceCollection;
import org.ametys.plugins.workspaces.documents.jcr.File;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Helper class for AI-related functionalities in workspace.
 */
public class WorkspaceFileAIHelper extends AbstractLogEnabled implements Component, Serviceable
{
    /** Avalon role */
    public static final String ROLE = WorkspaceFileAIHelper.class.getName();
    
    /** The summary max lentgh */
    public static final Integer SUMMARY_MAX_LENGTH = 200;
    
    private static final String __CONFIG_IS_AUTOMATIC_GENERATION_ENABLED = "workspaces.ai.docsummary.automatic.enabled";
    
    private AIHelper _aiHelper;
    private CurrentUserProvider _currentUserProvider;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _aiHelper = (AIHelper) manager.lookup(AIHelper.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
    }
    
    /**
     * Get the value of the configuration param
     * @return true if it's true, false if it's false
     */
    public boolean isAutomaticGenerationEnabled()
    {
        return _aiHelper.isEnabled()
                && Config.getInstance().getValue(__CONFIG_IS_AUTOMATIC_GENERATION_ENABLED, false, false);
    }
    
    /**
     * Generate the description and set it to the file
     * @param file the file to summarize
     * @return String the generated summary
     * @throws Exception if an error occurred
     */
    public String generateFileDescription(File file) throws Exception
    {
        String summary = _aiHelper.resourceToSummary(file, SUMMARY_MAX_LENGTH);
        file.setDescription(summary);
        file.setIsAutoGeneratedDescription(true);
        
        file.setLastContributor(_currentUserProvider.getUser());
        file.setLastModified(new Date());

        ModifiableResourceCollection parent = file.getParent();
        parent.saveChanges();
        file.checkpoint();
        
        return summary;
    }
}
