/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.documents.jcr;

import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.RepositoryException;

import org.ametys.cms.repository.CommentableAmetysObject;
import org.ametys.cms.repository.comment.Comment;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.explorer.resources.jcr.JCRResource;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.data.holder.ModifiableModelLessDataHolder;
import org.ametys.plugins.repository.data.holder.impl.DefaultModifiableModelLessDataHolder;
import org.ametys.plugins.repository.data.repositorydata.ModifiableRepositoryData;
import org.ametys.plugins.repository.data.repositorydata.impl.JCRRepositoryData;

/**
 * Implementation of {@link Resource} used for document module of workspaces and can handle comments, backed by a JCR node.<br>
 */
public class File extends JCRResource<FileFactory> implements CommentableAmetysObject<Comment>
{

    /** Tokens node */
    public static final String TOKENS_NODE = RepositoryConstants.NAMESPACE_PREFIX + ":tokens";
    /** Token node */
    public static final String TOKEN_NODE = RepositoryConstants.NAMESPACE_PREFIX + ":token";
    /** Token id */
    public static final String TOKEN_ID = "tokenId";
    /** Token id property */
    public static final String TOKEN_ID_PROPERTY = RepositoryConstants.NAMESPACE_PREFIX + ":" + TOKEN_ID;
    /** Creation date property for a token */
    public static final String TOKEN_CREATION_DATE_PROPERTY = RepositoryConstants.NAMESPACE_PREFIX + ":creationDate";
    /** File id property for a token */
    public static final String TOKEN_FILE_ID_PROPERTY = RepositoryConstants.NAMESPACE_PREFIX + ":fileId";

    /**
     * Creates an {@link File}.
     * @param node the node backing this {@link AmetysObject}
     * @param parentPath the parentPath in the Ametys hierarchy
     * @param factory the DefaultAmetysObjectFactory which created the AmetysObject
     */
    public File(Node node, String parentPath, FileFactory factory)
    {
        super(node, parentPath, factory);
    }

    /**
     * Get the file description
     * @return the description
     */
    public String getDescription()
    {
        try
        {
            Node fileNode = getNode();
            return fileNode.hasProperty("ametys:description")
                ? fileNode.getProperty("ametys:description").getString()
                : null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Cannot get description for file " + this.getName() + " (" + this.getId() + ")", e);
        }
    }
    
    /**
     * Set the file description
     * @param description the description to set
     */
    public void setDescription (String description)
    {
        try
        {
            Node fileNode = getNode();
            fileNode.setProperty("ametys:description", description);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Cannot set description for file " + this.getName() + " (" + this.getId() + ")", e);
        }
    }
    /**
     * <code>true</code> if the description was automatically generated, <code>false</code> otherwise.
     * @return <code>true</code> if the description was automatically generated, <code>false</code> otherwise.
     */
    public boolean isAutomaticallyGeneratedDescription()
    {
        try
        {
            Node fileNode = getNode();
            if (fileNode.hasProperty("ametys:isAutoGeneratedDescription"))
            {
                return fileNode.getProperty("ametys:isAutoGeneratedDescription").getBoolean();
            }
            else
            {
                return false;
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Cannot get automatic description status for file " + this.getName() + " (" + this.getId() + ")", e);
        }
    }
    /**
     * Set to <code>true</code> if the description was automatically generated, <code>false</code> otherwise.
     * @param isAutoGeneratedDescription <code>true</code> if the description was automatically generated
     */
    public void setIsAutoGeneratedDescription(boolean isAutoGeneratedDescription)
    {
        try
        {
            Node fileNode = getNode();
            fileNode.setProperty("ametys:isAutoGeneratedDescription", isAutoGeneratedDescription);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Cannot set automatic description status for file " + this.getName() + " (" + this.getId() + ")", e);
        }
    }
    
    public Comment createComment()
    {
        return new Comment(_getCommentsDataHolder());
    }
    
    public Comment createComment(String commentId, ZonedDateTime creationDate)
    {
        return new Comment(_getCommentsDataHolder(), commentId, creationDate);
    }
    
    public Comment getComment(String commentId) throws AmetysRepositoryException
    {
        return Comment.getComment(_getCommentsDataHolder(), commentId);
    }
    
    public List<Comment> getComments(boolean includeNotValidatedComments, boolean includeValidatedComments) throws AmetysRepositoryException
    {
        return Comment.getComments(_getCommentsDataHolder(), includeNotValidatedComments, includeValidatedComments);
    }

    private ModifiableModelLessDataHolder _getCommentsDataHolder()
    {
        try
        {
            Node baseNode = getBaseNode();
            if (!baseNode.hasNode("ametys:comments"))
            {
                baseNode.addNode("ametys:comments", "ametys:compositeMetadata");
            }
            Node commentsNode = baseNode.getNode("ametys:comments");
            
            baseNode.getSession().save();
            
            ModifiableRepositoryData repositoryData = new JCRRepositoryData(commentsNode);
            return new DefaultModifiableModelLessDataHolder(_getFactory().getUnversionedDataTypeExtensionPoint(), repositoryData);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }
    
    /**
     * Token class representing a composite object with id, creation date, and file id.
     * @param id the token id
     * @param creationDate the creation date
     * @param fileId  the file id
     */
    public record PublicLinkToken(String id, ZonedDateTime creationDate, String fileId) { /* empty */ }

    // FIXME We only handle single token for now, so we just return the first one
    /**
     * Retrieves the file token. The JCR can handle multiple tokens, but only the first one is returned for now.
     * @return the file token.
     */
    public PublicLinkToken getToken()
    {
        List<PublicLinkToken> tokens = new ArrayList<>();
        try
        {
            Node fileNode = getNode();
            if (fileNode.hasNode(TOKENS_NODE))
            {
                Node tokensNode = fileNode.getNode(TOKENS_NODE);
                NodeIterator tokenNodes = tokensNode.getNodes();
                while (tokenNodes.hasNext())
                {
                    Node tokenNode = tokenNodes.nextNode();
                    String id = tokenNode.getProperty(TOKEN_ID_PROPERTY).getString();
                    ZonedDateTime creationDate = DateUtils.asZonedDateTime(tokenNode.getProperty(TOKEN_CREATION_DATE_PROPERTY).getDate());
                    String fileId = tokenNode.getProperty(TOKEN_FILE_ID_PROPERTY).getString();
                    tokens.add(new PublicLinkToken(id, creationDate, fileId));
                }
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Cannot get tokens for file " + this.getName() + " (" + this.getId() + ")", e);
        }
        
        if (tokens.isEmpty())
        {
            return null;
        }
        PublicLinkToken token = tokens.get(0);
        
        if (!this.getId().equals(token.fileId()))
        {
            return null;
        }
       
        return token;
    }

    /**
     * Stores the list of tokens.
     * @param tokens the list of tokens.
     */
    public void setTokens(List<PublicLinkToken> tokens)
    {
        try
        {
            Node fileNode = getNode();
            Node tokensNode;
            if (fileNode.hasNode(TOKENS_NODE))
            {
                tokensNode = fileNode.getNode(TOKENS_NODE);
                tokensNode.remove();
            }
            tokensNode = fileNode.addNode(TOKENS_NODE, "ametys:compositeMetadata");

            for (PublicLinkToken token : tokens)
            {
                Node tokenNode = tokensNode.addNode(TOKEN_NODE, "ametys:compositeMetadata");
                tokenNode.setProperty(TOKEN_ID_PROPERTY, token.id());
                tokenNode.setProperty(TOKEN_CREATION_DATE_PROPERTY, DateUtils.asCalendar(token.creationDate()));
                tokenNode.setProperty(TOKEN_FILE_ID_PROPERTY, token.fileId());
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Cannot set tokens for file " + this.getName() + " (" + this.getId() + ")", e);
        }
    }
    
    /**
     * Creates a new token with the current date as the creation date.
     * @return the created token.
     */
    public PublicLinkToken createToken()
    {
        String tokenId = UUID.randomUUID().toString();
        ZonedDateTime creationDate = ZonedDateTime.now();
        PublicLinkToken newToken = new PublicLinkToken(tokenId, creationDate, this.getId());

        // Create a new list of tokens for idempotence, this may change in the future if we want to handle multiple tokens
        List<PublicLinkToken> tokens = new ArrayList<>();
        tokens.add(newToken);
        setTokens(tokens);

        return newToken;
    }
}
