/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.forum;

import java.io.IOException;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.servlet.multipart.Part;
import org.apache.commons.lang3.EnumUtils;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.data.RichText;
import org.ametys.cms.tag.Tag;
import org.ametys.core.observation.Event;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.userpref.UserPreferencesManager;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.explorer.resources.ModifiableResourceCollection;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.workspaces.ObservationConstants;
import org.ametys.plugins.workspaces.forum.filters.AcceptedFilter;
import org.ametys.plugins.workspaces.forum.filters.CategoryFilter;
import org.ametys.plugins.workspaces.forum.filters.CloseFilter;
import org.ametys.plugins.workspaces.forum.filters.NotificationFilter;
import org.ametys.plugins.workspaces.forum.filters.TagFilter;
import org.ametys.plugins.workspaces.forum.filters.TextSearchFilter;
import org.ametys.plugins.workspaces.forum.filters.ThreadFilter;
import org.ametys.plugins.workspaces.forum.jcr.JCRThread;
import org.ametys.plugins.workspaces.forum.jcr.JCRThreadFactory;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.tags.ProjectTagProviderExtensionPoint;
import org.ametys.runtime.authentication.AccessDeniedException;

/**
 * DAO for manipulating thread of a project
 *
 */
public class WorkspaceThreadDAO extends AbstractWorkspaceThreadDAO
{
    /** Avalon Role */
    public static final String ROLE = WorkspaceThreadDAO.class.getName();

    /** Rights to handle all threads */
    public static final String RIGHTS_HANDLE_ALL_THREAD = "Plugin_Workspace_Handle_All_Thread";

    /** Rights to handle reports */
    public static final String RIGHTS_REPORT_NOTIFICATION = "Plugins_Workspaces_Handle_Reported_Comment_Thread";
    
    /** Rights to create thread and edit his own thread */
    public static final String RIGHTS_CREATE_THREAD = "Plugin_Workspace_Create_Thread";
    
    /** Rights to contribute to threads */
    public static final String RIGHTS_CONTRIBUTE_THREAD = "Plugin_Workspace_Contribute_To_Thread";
    
    /** Rights to delete his own thread */
    public static final String RIGHTS_DELETE_OWN_THREAD = "Plugin_Workspace_Delete_Own_Thread";
    
    /** Rights to delete answers in his own thread */
    public static final String RIGHTS_DELETE_OWN_THREAD_ANSWERS = "Plugin_Workspace_Delete_Own_Thread_Answers";

    /** Rights to handle reports in his own thread */
    public static final String RIGHTS_REPORT_NOTIFICATION_OWN_THREAD = "Plugins_Workspaces_Handle_Reported_Comment_Own_Thread";
    
    /** The tag provider extension point */
    protected ProjectTagProviderExtensionPoint _tagProviderExtPt;

    /** The user preferences */
    protected UserPreferencesManager _userPrefsManager;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _tagProviderExtPt = (ProjectTagProviderExtensionPoint) manager.lookup(ProjectTagProviderExtensionPoint.ROLE);
        _userPrefsManager = (UserPreferencesManager) manager.lookup(UserPreferencesManager.ROLE);
    }

    /**
     * Add a new thread to the threads list
     * @param parameters The thread parameters
     * @param search filter by search query
     * @param category filter by category id
     * @param tag filter by tag id
     * @param closed filter by closeInfo field
     * @param newFiles the new file to add
     * @param newFileNames the file names to add
     * @param accepted filter by accepted answer
     * @param hasNotification filter by whether subjects have notification
     * @return The thread data
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> addThread(Map<String, Object> parameters, String search, String category, String tag, Boolean closed, List<Part> newFiles, List<String> newFileNames, Boolean accepted, Boolean hasNotification)
    {
        Project project = _workspaceHelper.getProjectFromRequest();

        ModifiableTraversableAmetysObject threadsRoot = _getThreadRoot(project);

        // Either RIGHTS_HANDLE_ALL_THREAD or RIGHTS_CREATE_THREAD is enough to allow thread creation, only throw exception when both fails
        if (_rightManager.hasRight(_currentUserProvider.getUser(), RIGHTS_HANDLE_ALL_THREAD, threadsRoot) != RightResult.RIGHT_ALLOW
            && _rightManager.hasRight(_currentUserProvider.getUser(), RIGHTS_CREATE_THREAD, threadsRoot) != RightResult.RIGHT_ALLOW)
        {
            throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to perform operation without convenient right");
        }
        
        int index = 1;
        String name = "thread-1";
        while (threadsRoot.hasChild(name))
        {
            index++;
            name = "thread-" + index;
        }
        JCRThread thread = (JCRThread) threadsRoot.createChild(name, JCRThreadFactory.THREAD_NODETYPE);

        ZonedDateTime now = ZonedDateTime.now();
        thread.setCreationDate(now);
        thread.setLastContribution(now);
        thread.setAuthor(_currentUserProvider.getUser());

        List<Map<String, Object>> newTags = _editThreadByParameters(thread, parameters, newFiles, newFileNames, new ArrayList<>());
        thread.saveChanges();

        _workspaceThreadUserPrefDAO.clearUnopenedThreadNotification(thread.getId());

        Map<String, Object> threadAsJSON = _threadJSONHelper.threadAsJSON(thread, getSitemapLanguage(), getSiteName(), false);

        threadAsJSON.put("passFilter", _passFilters(thread, search, category, tag, closed, accepted, hasNotification));

        _notifyEvent(thread, ObservationConstants.EVENT_THREAD_CREATED);

        Map<String, Object> results = new HashMap<>();
        results.put("thread", threadAsJSON);
        results.put("newTags", newTags);
        return results;
    }

    /**
     * Edit a thread
     * @param parameters The thread parameters
     * @param search filter by search query
     * @param category filter by category id
     * @param tag filter by tag id
     * @param closed filter by closeInfo field
     * @param newFiles the new file to add
     * @param newFileNames the file names to add
     * @param deleteFiles the file to delete
     * @param accepted filter by accepted answer
     * @param hasNotification filter by whether subjects have notification
     * @return The thread data
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> editThread(Map<String, Object> parameters, String search, String category, String tag, Boolean closed, List<Part> newFiles, List<String> newFileNames, List<String> deleteFiles, Boolean accepted, Boolean hasNotification)
    {
        Project project = _workspaceHelper.getProjectFromRequest();
        ModifiableTraversableAmetysObject threadsRoot = _getThreadRoot(project);

        String threadId = (String) parameters.get("id");
        JCRThread thread = _resolver.resolveById(threadId);

        _checkUserRights(threadsRoot, RIGHTS_HANDLE_ALL_THREAD, thread, RIGHTS_CREATE_THREAD);

        boolean wasOpen = thread.getCloseAuthor() == null;

        List<Map<String, Object>> newTags = _editThreadByParameters(thread, parameters, newFiles, newFileNames, deleteFiles);

        thread.saveChanges();
        Map<String, Object> threadAsJSON = _threadJSONHelper.threadAsJSON(thread, getSitemapLanguage(), getSiteName(), true);

        threadAsJSON.put("passFilter", _passFilters(thread, search, category, tag, closed, accepted, hasNotification));


        if (wasOpen && thread.getCloseAuthor() != null)
        {
            _notifyEvent(thread, ObservationConstants.EVENT_THREAD_CLOSED);
        }
        else
        {
            _notifyEvent(thread, ObservationConstants.EVENT_THREAD_MODIFIED);
        }


        Map<String, Object> results = new HashMap<>();
        results.put("thread", threadAsJSON);
        results.put("newTags", newTags);
        return results;
    }

    private List<Map<String, Object>> _editThreadByParameters(JCRThread thread, Map<String, Object> parameters, List<Part> newFiles, List<String> newFileNames, List<String> deleteFiles)
    {
        _checkValidParameters(parameters);

        _setAttachments(thread, newFiles, newFileNames, deleteFiles);

        String label = (String) parameters.get(Thread.ATTRIBUTE_TITLE);
        thread.setTitle(label);

        String description = (String) parameters.getOrDefault(Thread.ATTRIBUTE_CONTENT_FOR_EDITING, StringUtils.EMPTY);
        try
        {
            RichText richText = thread.getContent() != null ? thread.getContent() : new RichText();
            _richTextTransformer.transform(description, richText);
            thread.setContent(richText);
        }
        catch (AmetysRepositoryException | IOException e)
        {
            throw new AmetysRepositoryException("Unable to transform the text " + description + " into a rich text for thread " + thread.getId(), e);
        }

        String category = (String) parameters.get("category");
        thread.setCategory(ThreadCategoryEnum.valueOf(category));

        ZonedDateTime now = ZonedDateTime.now();
        thread.setLastModified(now);

        @SuppressWarnings("unchecked")
        List<Object> tags = (List<Object>) parameters.getOrDefault(Thread.ATTRIBUTE_TAGS, new ArrayList<>());

        String projectName = getProjectName();

        ModifiableResourceCollection moduleRoot = _getModuleRoot(projectName);
        
        List<Map<String, Object>> createdTagsJson = _handleTags(thread, tags, moduleRoot);

        String closeDateString = (String) parameters.get(Thread.ATTRIBUTE_CLOSEDATE);
        if (closeDateString != null)
        {
            thread.setCloseDate(DateUtils.parseZonedDateTime(closeDateString));

            thread.setCloseAuthor(_currentUserProvider.getUser());
        }
        else
        {
            thread.setCloseDate(null);

            thread.setCloseAuthor(null);
        }

        return createdTagsJson;
    }

    /**
     * Delete a thread
     * @param id The thread id
     * @return The thread data
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> deleteThread(String id)
    {
        JCRThread thread = _resolver.resolveById(id);
        Project project = _workspaceHelper.getProjectFromRequest();
        
        ModifiableTraversableAmetysObject threadsRoot = _getThreadRoot(project);

        _checkUserRights(threadsRoot, RIGHTS_HANDLE_ALL_THREAD, thread, RIGHTS_DELETE_OWN_THREAD);

        Map<String, Object> threadAsJSON = _threadJSONHelper.threadAsJSON(thread, getSitemapLanguage(), getSiteName(), false);

        _notifyEvent(thread, ObservationConstants.EVENT_THREAD_DELETED);

        thread.remove();
        thread.saveChanges();

        return threadAsJSON;
    }

    private void _checkValidParameters(Map<String, Object> parameters)
    {
        String label = (String) parameters.get(Thread.ATTRIBUTE_TITLE);
        if (StringUtils.isBlank(label))
        {
            throw new IllegalArgumentException("Thread label is mandatory to create a new thread");
        }
        if (StringUtils.length(label) > 100)
        {
            throw new IllegalArgumentException("Thread label should not exceeds 100 characters");
        }

        String category = (String) parameters.get("category");
        if (!EnumUtils.isValidEnum(ThreadCategoryEnum.class, category))
        {
            throw new IllegalArgumentException("Invalid category");
        }
    }

    /**
     * Get the list of threads of a project
     * @param category The category to filter
     * @return The list of threads
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Integer getCategoryThreadCount(String category)
    {
        ThreadCategoryEnum cat = ThreadCategoryEnum.valueOf(category);
        String projectName = getProjectName();
        Project project = _projectManager.getProject(projectName);

        ModifiableResourceCollection moduleRoot = _getModuleRoot(projectName);

        _checkUserReadingRights(moduleRoot);

        return getProjectThreads(project, false).stream()
                .filter(thread -> thread.getCategory().equals(cat))
                .collect(Collectors.toList())
                .size();
    }

    /**
     * Get the threads of current project
     * @param search filter by search query
     * @param category filter by categroy id
     * @param tag filter by tag id
     * @param closed filter by closeInfo field
     * @param accepted filter by accepted answer
     * @param hasNotification filter by whether subjects have notification
     * @return the list of threads
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public List<String> getThreadIds(String search, String category, String tag, Boolean closed, Boolean accepted, Boolean hasNotification)
    {
        String projectName = getProjectName();
        ModifiableResourceCollection moduleRoot = _getModuleRoot(projectName);

        _checkUserReadingRights(moduleRoot);

        Project project = _projectManager.getProject(projectName);

        return getProjectThreads(project, true).stream()
                .filter(thread -> _passFilters(thread, search, category, tag, closed, accepted, hasNotification))
                .map(Thread::getId)
                .toList();
    }

    /**
     * Get the threads of current project
     * @param threadIds list of threads id
     * @return the list of threads
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public List<Map<String, Object>> getThreadsByIds(List<String> threadIds)
    {
        String projectName = getProjectName();
        ModifiableResourceCollection moduleRoot = _getModuleRoot(projectName);

        _checkUserReadingRights(moduleRoot);
        
        String sitemapLanguage = getSitemapLanguage();
        String siteName = getSiteName();
        return threadIds.stream()
                .map(id -> (Thread) _resolver.resolveById(id))
                .map(thread -> _threadJSONHelper.threadAsJSON(thread, sitemapLanguage, siteName, false))
                .toList();
    }

    private boolean _passFilters(Thread thread, String search, String category, String tag, Boolean closed, Boolean accepted, Boolean hasNotification)
    {
        List<ThreadFilter> filters = new ArrayList<>();

        filters.add(new TextSearchFilter(search, _threadJSONHelper));
        filters.add(new CategoryFilter(category));
        filters.add(new TagFilter(tag));
        filters.add(new CloseFilter(closed));
        filters.add(new AcceptedFilter(accepted));
        filters.add(new NotificationFilter(hasNotification, _workspaceThreadUserPrefDAO));

        return filters.stream().allMatch(filter -> filter.passFilter(thread));
    }

    /**
     * Get the threads from project
     * @param id the id of the page
     * @return the list of threads
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> getThread(String id)
    {
        ModifiableResourceCollection moduleRoot = _getModuleRoot();

        _checkUserReadingRights(moduleRoot);

        if (!_resolver.hasAmetysObjectForId(id))
        {
            return null;
        }

        Thread thread = _resolver.resolveById(id);
        Map<String, Object> threadAsJSON = _threadJSONHelper.threadAsJSON(thread, getSitemapLanguage(), getSiteName());

        return threadAsJSON;
    }

    /**
     * Get all thread's threads from given projets
     * @param project the project
     * @param sortByCreationDate true to sort by creation date
     * @return All threads as JSON
     */
    public List<Thread> getProjectThreads(Project project, boolean sortByCreationDate)
    {
        ForumWorkspaceModule forumModule = _workspaceModuleEP.getModule(ForumWorkspaceModule.FORUM_MODULE_ID);
        ModifiableTraversableAmetysObject forumsRoot = forumModule.getModuleRoot(project, true);
        Stream<Thread> stream = forumsRoot.getChildren()
                .stream()
                .filter(Thread.class::isInstance)
                .map(Thread.class::cast);
        if (sortByCreationDate)
        {
            stream = stream.sorted(Comparator.comparing(Thread::getLastContribution).reversed());
        }
        return stream.collect(Collectors.toList());
    }

    /**
     * Get the total number of threads of the project
     * @param project The project
     * @return The number of threads, or null if the module is not activated
     */
    public Long getThreadsCount(Project project)
    {
        return Long.valueOf(getProjectThreads(project, false).size());
    }

    /**
     * Get user rights from project name
     * @return the user rights
     */
    @Callable (rights = Callable.NO_CHECK_REQUIRED)
    public Map<String, Object> getUserRights()
    {
        Map<String, Object> results = new HashMap<>();

        ModifiableResourceCollection threadRoot = _getModuleRoot();

        UserIdentity user = _currentUserProvider.getUser();
        results.put("canHandleAllThread", _rightManager.hasRight(user, RIGHTS_HANDLE_ALL_THREAD, threadRoot) == RightResult.RIGHT_ALLOW);
        results.put("canHandleReports", _rightManager.hasRight(user, RIGHTS_REPORT_NOTIFICATION, threadRoot) == RightResult.RIGHT_ALLOW);
        results.put("canCreateThread", _rightManager.hasRight(user, RIGHTS_CREATE_THREAD, threadRoot) == RightResult.RIGHT_ALLOW);
        results.put("canContribute", _rightManager.hasRight(user, RIGHTS_CONTRIBUTE_THREAD, threadRoot) == RightResult.RIGHT_ALLOW);
        results.put("canDeleteOwnThread", _rightManager.hasRight(user, RIGHTS_DELETE_OWN_THREAD, threadRoot) == RightResult.RIGHT_ALLOW);
        results.put("canDeleteOwnThreadMessages", _rightManager.hasRight(user, RIGHTS_DELETE_OWN_THREAD_ANSWERS, threadRoot) == RightResult.RIGHT_ALLOW);
        results.put("canHandleReportsOwnThread", _rightManager.hasRight(user, RIGHTS_REPORT_NOTIFICATION_OWN_THREAD, threadRoot) == RightResult.RIGHT_ALLOW);

        return results;
    }

    /**
     * Get tags used in any threads
     * @return the tags (id and label)
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Set<Map<String, Object>> getUsedTags()
    {
        String projectName = getProjectName();
        Project project = _projectManager.getProject(projectName);
        
        ModifiableResourceCollection moduleRoot = _getModuleRoot(projectName);
        _checkUserReadingRights(moduleRoot);

        return getProjectThreads(project, false).stream() // Get all threads
                .map(Thread::getTags) // Get all tags of threads
                .flatMap(Set::stream) // Create a stream with tags to process them
                .map(tag -> _tagProviderExtPt.getTag(tag, Collections.emptyMap())) // Get the tag as an object and not a String.
                .filter(Objects::nonNull) // As getTag return null if the tag doesn't exist, filter to remove all nonexisting tags
                .map(tag -> _tagToJSON(tag)) // Convert objet to JSON
                .collect(Collectors.toSet()); // Collect all tags as a Set to remove duplicates
    }

    private Map<String, Object> _tagToJSON(Tag tag)
    {
        Map<String, Object> tagInfo = new HashMap<>();
        tagInfo.put("id", tag.getId());
        tagInfo.put("name", tag.getName());
        tagInfo.put("text", tag.getTitle());
        tagInfo.put("title", tag.getTitle());
        return tagInfo;
    }
    
    private void _notifyEvent(JCRThread thread, String eventId)
    {
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(ObservationConstants.ARGS_THREAD, thread);
        eventParams.put(org.ametys.plugins.explorer.ObservationConstants.ARGS_ID, thread.getId());

        _observationManager.notify(new Event(eventId, _currentUserProvider.getUser(), eventParams));
    }

}
