/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.initialization;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.repository.Content;
import org.ametys.core.observation.Event;
import org.ametys.core.user.directory.ModifiableUserDirectory;
import org.ametys.core.user.directory.UserDirectory;
import org.ametys.core.user.population.PopulationContextHelper;
import org.ametys.core.user.population.UserPopulation;
import org.ametys.core.user.population.UserPopulationDAO;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.data.holder.ModifiableModelAwareDataHolder;
import org.ametys.web.ObservationConstants;
import org.ametys.web.repository.page.ModifiablePage;
import org.ametys.web.repository.page.ModifiableSitemapElement;
import org.ametys.web.repository.page.ModifiableZone;
import org.ametys.web.repository.page.ModifiableZoneItem;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.PageDAO;
import org.ametys.web.repository.page.Zone;
import org.ametys.web.repository.page.ZoneItem;
import org.ametys.web.repository.page.ZoneItem.ZoneType;

/**
 * Initialize a page with user signup service.
 */
public class UserSignupPageInitializer extends DefaultStaticPageInitializer
{
    /** the population context helper */
    protected PopulationContextHelper _populationContextHelper;
    /** the user population DAO */
    protected UserPopulationDAO _userPopulationDAO;
    /** the page DAO */
    protected PageDAO _pageDAO;
    /** the Ametys object resolver */
    protected AmetysObjectResolver _resolver;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _pageDAO = (PageDAO) manager.lookup(PageDAO.ROLE);
        _populationContextHelper = (PopulationContextHelper) manager.lookup(PopulationContextHelper.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _userPopulationDAO = (UserPopulationDAO) manager.lookup(UserPopulationDAO.ROLE);
    }
    
    @Override
    public Optional< ? extends Page> createPage(ModifiableSitemapElement parent)
    {
        Optional< ? extends Page> result = super.createPage(parent);
        if (result.isPresent() && result.get() instanceof ModifiablePage page)
        {
            try (AmetysObjectIterable<ModifiableZone> zones = page.getZones())
            {
                for (ModifiableZone zone : zones)
                {
                    try (AmetysObjectIterable<ModifiableZoneItem> zoneItems = zone.getZoneItems())
                    {
                        for (ModifiableZoneItem zoneItem : zoneItems)
                        {
                            if (StringUtils.equals(zoneItem.getServiceId(), "org.ametys.web.service.UserSignup"))
                            {
                                ModifiableModelAwareDataHolder serviceParams = zoneItem.getServiceParameters();
                                
                                Optional<Content> termsOfUseContent = _getTermsOfUseContent(parent);
                                if (termsOfUseContent.isPresent())
                                {
                                    serviceParams.setValue("terms-of-service-content", termsOfUseContent.get().getId());
                                }
                                
                                Set<String> frontPopulations = _populationContextHelper.getUserPopulationsOnContexts(List.of("/sites-fo/" + parent.getSiteName(), "/sites/" + parent.getSiteName()), false);
                                Optional<UserDirectory> userDirectory = frontPopulations.stream()
                                        .map(_userPopulationDAO::getUserPopulation)
                                        .map(UserPopulation::getUserDirectories)
                                        .flatMap(List::stream)
                                        .filter(ModifiableUserDirectory.class::isInstance)
                                        .findAny();
                                if (userDirectory.isPresent())
                                {
                                    serviceParams.setValue("userdirectory", userDirectory.get().getPopulationId() + "#" + userDirectory.get().getId());
                                }
                                else
                                {
                                    getLogger().warn("No user directory found to set on the signup services. Consider configuring one or remove the sign up page.");
                                }
                                
                                zoneItem.saveChanges();
                                
                                // FIXME we should use api that manage the event instead of doing it manually
                                Map<String, Object> eventParams = new HashMap<>();
                                eventParams.put(ObservationConstants.ARGS_ZONE_ITEM, zoneItem);
                                eventParams.put(ObservationConstants.ARGS_ZONE_ITEM_ID, zoneItem.getId());
                                eventParams.put(ObservationConstants.ARGS_ZONE_ITEM_SERVICE, zoneItem.getServiceId());
                                _observationManager.notify(new Event(ObservationConstants.EVENT_SERVICE_MODIFIED, _currentUserProvider.getUser(), eventParams));
                            }
                        }
                    }
                }
                
            }
            page.saveChanges();
            
        }
        return result;
    }

    private Optional<Content> _getTermsOfUseContent(ModifiableSitemapElement parent)
    {
        for (String pageId : _pageDAO.findPagedIdsByTag(parent.getSiteName(), parent.getSitemapName(), "WORKSPACES_CGU"))
        {
            Page page = _resolver.resolveById(pageId);
            try (AmetysObjectIterable< ? extends Zone> zones = page.getZones())
            {
                return zones.stream()
                        .flatMap(zone -> zone.getZoneItems().stream())
                        .filter(items -> items.getType().equals(ZoneType.CONTENT))
                        .map(ZoneItem::<Content>getContent)
                        .findAny();
            }
        }
        
        return Optional.empty();
    }
}
