/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.project;

import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

import javax.jcr.Node;
import javax.jcr.PathNotFoundException;
import javax.jcr.Property;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.Value;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

import org.ametys.cms.tag.Tag;
import org.ametys.core.cache.AbstractCacheManager;
import org.ametys.core.cache.AbstractCacheManager.CacheType;
import org.ametys.core.cache.Cache;
import org.ametys.core.cache.CacheException;
import org.ametys.core.group.GroupDirectoryContextHelper;
import org.ametys.core.group.GroupIdentity;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.observation.Observer;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.population.PopulationContextHelper;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.LambdaUtils;
import org.ametys.plugins.core.impl.cache.AbstractCacheKey;
import org.ametys.plugins.core.search.UserAndGroupSearchManager;
import org.ametys.plugins.core.user.UserHelper;
import org.ametys.plugins.explorer.ExplorerNode;
import org.ametys.plugins.explorer.resources.ModifiableResourceCollection;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.CollectionIterable;
import org.ametys.plugins.repository.ModifiableAmetysObject;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.plugins.repository.jcr.NodeTypeHelper;
import org.ametys.plugins.repository.provider.AbstractRepository;
import org.ametys.plugins.repository.provider.JackrabbitRepository;
import org.ametys.plugins.repository.provider.WorkspaceSelector;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.plugins.workspaces.ObservationConstants;
import org.ametys.plugins.workspaces.catalog.CatalogSiteType;
import org.ametys.plugins.workspaces.categories.Category;
import org.ametys.plugins.workspaces.categories.CategoryProviderExtensionPoint;
import org.ametys.plugins.workspaces.members.JCRProjectMember;
import org.ametys.plugins.workspaces.members.JCRProjectMember.MemberType;
import org.ametys.plugins.workspaces.members.ProjectMemberManager;
import org.ametys.plugins.workspaces.members.ProjectMemberManager.ProjectMember;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModule;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModuleExtensionPoint;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.project.objects.Project.InscriptionStatus;
import org.ametys.plugins.workspaces.project.rights.ProjectRightHelper;
import org.ametys.plugins.workspaces.tags.ProjectTagProviderExtensionPoint;
import org.ametys.plugins.workspaces.util.StatisticColumn;
import org.ametys.plugins.workspaces.util.StatisticsColumnType;
import org.ametys.runtime.authentication.AccessDeniedException;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.PluginAware;
import org.ametys.web.repository.page.ModifiablePage;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.PageQueryHelper;
import org.ametys.web.repository.page.SitemapElement;
import org.ametys.web.repository.page.ZoneItem;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteDAO;
import org.ametys.web.repository.site.SiteManager;
import org.ametys.web.repository.sitemap.Sitemap;
import org.ametys.web.site.SiteConfigurationManager;

/**
 * Helper component for managing project workspaces
 */
public class ProjectManager extends AbstractLogEnabled implements Serviceable, Component, Contextualizable, PluginAware, Initializable, Observer
{
    /** Avalon Role */
    public static final String ROLE = ProjectManager.class.getName();
    
    /** Constant for the {@link Cache} id (the {@link Cache} is in {@link CacheType#REQUEST REQUEST} attribute) for the {@link Project}s objects
     *  in cache by {@link RequestProjectCacheKey} (composition of project name and workspace name). */
    public static final String REQUEST_PROJECTBYID_CACHE = ProjectManager.class.getName() + "$ProjectById";
    
    /** Constant for the {@link Cache} id for the {@link Project} ids (as {@link String}s) in cache by project name (for whole application). */
    public static final String MEMORY_PROJECTIDBYNAMECACHE = ProjectManager.class.getName() + "$UUID";

    /** Constant for the {@link Cache} id (the {@link Cache} is in {@link CacheType#REQUEST REQUEST} attribute) for the {@link Page}s objects
     *  in cache by {@link RequestModuleCacheKey} (composition of project name and module name). */
    public static final String REQUEST_PAGESBYPROJECTANDMODULE_CACHE = ProjectManager.class.getName() + "$PagesByModule";
    
    /** Constant for the {@link Cache} id for the {@link Project} ids (as {@link String}s) in cache by project name (for whole application). */
    public static final String MEMORY_PAGESBYIDCACHE = ProjectManager.class.getName() + "$PageUUID";

    /** Constant for the {@link Cache} id for the {@link Project} ids (as {@link String}s) in cache by site name (for whole application). */
    public static final String MEMORY_SITEASSOCIATION_CACHE = ProjectManager.class.getName() + "$SiteAssociation";

    /** Workspaces plugin node name */
    private static final String __WORKSPACES_PLUGIN_NODE_NAME = "workspaces";
    
    /** Workspaces plugin node name */
    private static final String __WORKSPACES_PLUGIN_NODE_TYPE = RepositoryConstants.NAMESPACE_PREFIX + ":unstructured";
    
    /** The name of the projects root node */
    private static final String __PROJECTS_ROOT_NODE_NAME = "projects";
    
    /** The type of the projects root node */
    private static final String __PROJECTS_ROOT_NODE_TYPE = RepositoryConstants.NAMESPACE_PREFIX + ":unstructured";
    
    /** Constants for tags metadata */
    private static final String __PROJECTS_TAGS_PROPERTY = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":tags";
    
    private static final String __PAGE_MODULES_VALUE = "workspaces-modules";

    private static final String __IS_CACHE_FILLED = "###iscachefilled###";

    /** Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The i18n utils. */
    protected I18nUtils _i18nUtils;
    
    /** Site manager */
    protected SiteManager _siteManager;
    
    /** Site DAO */
    protected SiteDAO _siteDao;
    
    /** Site configuration manager */
    protected SiteConfigurationManager _siteConfigurationManager;
    
    /** Module Managers EP */
    protected WorkspaceModuleExtensionPoint _moduleManagerEP;

    /** Helper for user population */
    protected PopulationContextHelper _populationContextHelper;
    
    /** Helper for group directory's context */
    protected GroupDirectoryContextHelper _groupDirectoryContextHelper;
    
    /** The project members' manager */
    protected ProjectMemberManager _projectMemberManager;

    /** Avalon context */
    protected Context _context;
    
    private ObservationManager _observationManager;
    
    private CurrentUserProvider _currentUserProvider;

    private ProjectMemberManager _projectMembers;

    private String _pluginName;

    private ProjectRightHelper _projectRightHelper;

    private ProjectTagProviderExtensionPoint _projectTagProviderEP;

    private CategoryProviderExtensionPoint _categoryProviderEP;
    
    private UserHelper _userHelper;

    private AbstractCacheManager _cacheManager;

    private UserAndGroupSearchManager _userAndGroupSearchManager;

    private JackrabbitRepository _repository;

    private WorkspaceSelector _workspaceSelector;

    private RightManager _rightManager;

    private GroupDirectoryContextHelper _directoryContextHelper;

    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _repository = (JackrabbitRepository) manager.lookup(AbstractRepository.ROLE);
        _workspaceSelector = (WorkspaceSelector) manager.lookup(WorkspaceSelector.ROLE);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
        _siteDao = (SiteDAO) manager.lookup(SiteDAO.ROLE);
        _siteConfigurationManager = (SiteConfigurationManager) manager.lookup(SiteConfigurationManager.ROLE);
        _projectMembers = (ProjectMemberManager) manager.lookup(ProjectMemberManager.ROLE);
        _observationManager = (ObservationManager) manager.lookup(ObservationManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _moduleManagerEP = (WorkspaceModuleExtensionPoint) manager.lookup(WorkspaceModuleExtensionPoint.ROLE);
        _projectRightHelper = (ProjectRightHelper) manager.lookup(ProjectRightHelper.ROLE);
        _projectTagProviderEP = (ProjectTagProviderExtensionPoint) manager.lookup(ProjectTagProviderExtensionPoint.ROLE);
        _userHelper = (UserHelper) manager.lookup(UserHelper.ROLE);
        _categoryProviderEP = (CategoryProviderExtensionPoint) manager.lookup(CategoryProviderExtensionPoint.ROLE);
        _cacheManager = (AbstractCacheManager) manager.lookup(AbstractCacheManager.ROLE);
        _populationContextHelper = (PopulationContextHelper) manager.lookup(PopulationContextHelper.ROLE);
        _groupDirectoryContextHelper = (GroupDirectoryContextHelper) manager.lookup(GroupDirectoryContextHelper.ROLE);
        _projectMemberManager = (ProjectMemberManager) manager.lookup(ProjectMemberManager.ROLE);
        _userAndGroupSearchManager = (UserAndGroupSearchManager) manager.lookup(UserAndGroupSearchManager.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _directoryContextHelper = (GroupDirectoryContextHelper) manager.lookup(GroupDirectoryContextHelper.ROLE);
    }
    
    public void initialize() throws Exception
    {
        _createCaches();
        _observationManager.registerObserver(this);
    }
    
    @Override
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _pluginName = pluginName;
    }

    /**
     * Enumeration for the profile to assign for new module
     */
    public enum ProfileForNewModule
    {
        /** No profile assigned for new modules */
        NONE,
        
        /** Default profile assigned for new modules */
        DEFAULT_MEMBER_PROFILE
        
    }
    
    /**
     * Retrieves all projects
     * @return the projects
     */
    public AmetysObjectIterable<Project> getProjects()
    {
        return getProjects(true);
    }
    
    /**
     * Retrieves all projects
     * @param onlyWorking true to retrieve only working projects with non null sites
     * @return the projects
     */
    public AmetysObjectIterable<Project> getProjects(boolean onlyWorking)
    {
        // As cache is computed from default JCR workspace, we need to filter on sites that exist into the current JCR workspace
        Set<Project> projects = _getUUIDCache().values().stream()
                .filter(_resolver::hasAmetysObjectForId)
                .map(_resolver::<Project>resolveById)
                // If needed, check if the site is not null, as it can happen if site was deleted from admin side
                .filter(project -> !onlyWorking || Objects.nonNull(project.getSite()))
                .collect(Collectors.toSet());
        
        return new CollectionIterable<>(projects);
    }
    
    /**
     * Retrieves projects filtered by categories
     * @param filteredCategories the filtered categories. Can be empty to no filter by categories.
     * @return the projects
     */
    public List<Project> getProjects(Set<String> filteredCategories)
    {
        return getProjects(filteredCategories, null, null, true);
    }
    
    /**
     * Retrieves projects filtered by categories and/or keywords
     * @param filteredCategories the filtered categories. Can be empty to no filter by categories.
     * @param filteredKeywords the filtered keywords. Can be empty to no filter by keywords.
     * @param anyMatch true to get projects matching categories OR keywords OR pattern
     * @return the projects
     */
    public List<Project> getProjects(Set<String> filteredCategories, Set<String> filteredKeywords, boolean anyMatch)
    {
        return getProjects(filteredCategories, filteredKeywords, null, anyMatch);
    }
    
    /**
     * Retrieves projects filtered by categories and/or keywords and/or pattern
     * @param filteredCategories the filtered categories. Can be empty to no filter by categories.
     * @param filteredKeywords the filtered keywords. Can be empty to no filter by keywords.
     * @param pattern to filter on pattern. Can be null or empty to no filter on pattern
     * @param anyMatch true to get projects matching categories OR keywords OR pattern
     * @return the projects
     */
    public List<Project> getProjects(Set<String> filteredCategories, Set<String> filteredKeywords, String pattern, boolean anyMatch)
    {
        return getProjects(filteredCategories, filteredKeywords, null, anyMatch, false);
    }
    
    /**
     * Retrieves projects filtered by categories and/or keywords and/or pattern
     * @param filteredCategories the filtered categories. Can be null or empty to no filter by categories.
     * @param filteredKeywords the filtered keywords. Can be empty to no filter by keywords.
     * @param pattern to filter on pattern. Can be null or empty to no filter on pattern
     * @param anyMatch true to get projects matching categories OR keywords OR pattern
     * @param excludePrivate true to exclude private projects
     * @return the projects
     */
    public List<Project> getProjects(Set<String> filteredCategories, Set<String> filteredKeywords, String pattern, boolean anyMatch, boolean excludePrivate)
    {
        List<Predicate<Project>> filters = new ArrayList<>();
        if (filteredCategories != null && !filteredCategories.isEmpty())
        {
            filters.add(p -> !Collections.disjoint(p.getCategories(), filteredCategories));
        }

        if (filteredKeywords != null && !filteredKeywords.isEmpty())
        {
            filters.add(p -> !Collections.disjoint(Arrays.asList(p.getKeywords()), filteredKeywords));
        }

        Pattern patternFilter = StringUtils.isNotEmpty(pattern) ? Pattern.compile(pattern, Pattern.CASE_INSENSITIVE) : null;
        if (patternFilter != null)
        {
            filters.add(p -> p.getTitle() != null && patternFilter.matcher(p.getTitle()).find() || p.getDescription() != null && patternFilter.matcher(p.getDescription()).find());
        }

        Predicate<Project> fullMatch = filters.stream().reduce(anyMatch ? Predicate::or : Predicate::and).orElse(p -> Boolean.TRUE);
        Predicate<Project> matchStatus = p ->  !excludePrivate || p.getInscriptionStatus() != InscriptionStatus.PRIVATE;
        
        return getProjects()
                .stream()
                .filter(matchStatus)
                .filter(fullMatch)
                .collect(Collectors.toList());
    }
    
    /**
     * Get the projects categories
     * @return the projects categories
     */
    public Set<Category> getProjectsCategories()
    {
        return getProjects().stream()
            .map(Project::getCategories)
            .flatMap(Collection::stream)
            .map(id -> _categoryProviderEP.getTag(id, null))
            .filter(Objects::nonNull)
            .collect(Collectors.toSet());
    }
    
    /**
     * Get the user's projects categories
     * @param user the user
     * @return the user's projects categories
     */
    public Set<Category> getUserProjectsCategories(UserIdentity user)
    {
        return getUserProjects(user).keySet()
                .stream()
                .map(Project::getCategories)
                .flatMap(Collection::stream)
                .map(id -> _categoryProviderEP.getTag(id, null))
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());
    }
    
    /**
     * Get the projects modules
     * @return the projects modules
     */
    public Set<WorkspaceModule> getProjectsModules()
    {
        return getProjects().stream()
            .map(Project::getModules)
            .flatMap(Arrays::stream)
            .map(id -> _moduleManagerEP.<WorkspaceModule>getModule(id))
            .filter(Objects::nonNull)
            .collect(Collectors.toSet());
    }
    
    /**
     * Get the projects modules
     * @param user the user
     * @return the projects modules
     */
    public Set<WorkspaceModule> getUserProjectsModules(UserIdentity user)
    {
        return getUserProjects(user).keySet()
            .stream()
            .map(Project::getModules)
            .flatMap(Arrays::stream)
            .map(id -> _moduleManagerEP.<WorkspaceModule>getModule(id))
            .filter(Objects::nonNull)
            .collect(Collectors.toSet());
    }
    
    /**
     * Retrieves all projects for client side
     * @return the projects
     */
    @Callable(rights = "Runtime_Rights_Admin_Access", context = "/admin")
    public List<Map<String, Object>> getProjectsForClientSide()
    {
        return getProjects(false)
                .stream()
                .map(p -> getProjectProperties(p))
                .collect(Collectors.toList());
    }
    
    
    /**
     * Retrieves a project by its name
     * @param projectName The project name
     * @return the project or <code>null</code> if not found
     */
    public Project getProject(String projectName)
    {
        if (StringUtils.isBlank(projectName))
        {
            return null;
        }
        
        Request request = _getRequest();
        if (request == null)
        {
            // There is no request to store cache
            return _computeProject(projectName);
        }
        
        Cache<RequestProjectCacheKey, Project> projectsCache = _getRequestProjectCache();
        
        // The site key in the cache is of the form {site + workspace}.
        String currentWorkspace = _workspaceSelector.getWorkspace();
        RequestProjectCacheKey projectKey = RequestProjectCacheKey.of(projectName, currentWorkspace);
        
        try
        {
            Project project = projectsCache.get(projectKey, __ -> _computeProject(projectName));
            return project;
        }
        catch (CacheException e)
        {
            if (e.getCause() instanceof UnknownAmetysObjectException)
            {
                throw new UnknownAmetysObjectException(e.getMessage());
            }
            else
            {
                throw e;
            }
        }
    }
    
    /**
     * Get the user's projects
     * @param user the user
     * @return the user's projects
     */
    public Map<Project, MemberType> getUserProjects(UserIdentity user)
    {
        return getUserProjects(user, Set.of());
    }
    
    /**
     * Get the user's projects filtered by categories
     * @param user the user
     * @param filteredCategories the filtered categories. Can be empty to no filter by categories
     * @return the user's projects
     */
    public Map<Project, MemberType> getUserProjects(UserIdentity user, Set<String> filteredCategories)
    {
        return getUserProjects(user, filteredCategories, null, null, true);
    }
    
    /**
     * Get the user's projects filtered by categories OR keywords
     * @param user the user
     * @param filteredCategories the filtered categories. Can be empty to no filter by categories
     * @param filteredKeywords the filtered keywords. Can be empty to no filter by keywords
     * @return the user's projects
     */
    public Map<Project, MemberType> getUserProjects(UserIdentity user, Set<String> filteredCategories, Set<String> filteredKeywords)
    {
        return getUserProjects(user, filteredCategories, filteredKeywords, null, true);
    }
    
    /**
     * Get the user's projects filtered by categories, keywords and/or pattern
     * @param user the user
     * @param filteredCategories the filtered categories. Can be empty to no filter by categories.
     * @param filteredKeywords the filtered keywords. Can be empty to no filter by keywords.
     * @param pattern to filter on pattern. Can be null or empty to no filter on pattern
     * @param anyMatch true to get projects matching categories OR keywords OR pattern
     * @return the user's projects
     */
    public Map<Project, MemberType> getUserProjects(UserIdentity user, Set<String> filteredCategories, Set<String> filteredKeywords, String pattern, boolean anyMatch)
    {
        return getUserProjects(user, filteredCategories, filteredKeywords, pattern, true, false);
    }
    
    /**
     * Get the user's projects filtered by categories, keywords and/or pattern
     * @param user the user
     * @param filteredCategories the filtered categories. Can be empty to no filter by categories.
     * @param filteredKeywords the filtered keywords. Can be empty to no filter by keywords.
     * @param pattern to filter on pattern. Can be null or empty to no filter on pattern
     * @param anyMatch true to get projects matching categories OR keywords OR pattern
     * @param excludePrivate true to exclude private projects
     * @return the user's projects
     */
    public Map<Project, MemberType> getUserProjects(UserIdentity user, Set<String> filteredCategories, Set<String> filteredKeywords, String pattern, boolean anyMatch, boolean excludePrivate)
    {
        Map<Project, MemberType> userProjects = new HashMap<>();
        
        List<Project> projects = getProjects(filteredCategories, filteredKeywords, pattern, anyMatch, excludePrivate);
        for (Project project : projects)
        {
            ProjectMember member = _projectMembers.getProjectMember(project, user);
            if (member != null)
            {
                userProjects.put(project, member.getType());
            }
        }
        
        return userProjects;
    }

    /**
     * Get the projects managed by the user
     * @param user the user
     * @return the projects for which the user is a manager
     */
    public List<Project> getManagedProjects(UserIdentity user)
    {
        return getManagedProjects(user, Set.of());
    }
    
    /**
     * Get the projects managed by the user
     * @param user the user
     * @param filteredCategories the filtered categories. Can be empty to no filter by categories.
     * @return the projects for which the user is a manager
     */
    public List<Project> getManagedProjects(UserIdentity user, Set<String> filteredCategories)
    {
        return getProjects(filteredCategories)
            .stream()
            .filter(p -> ArrayUtils.contains(p.getManagers(), user))
            .collect(Collectors.toList());
    }
    
    /**
     * Returns true if the given project exists.
     * @param projectName the project name.
     * @return true if the given project exists.
     */
    public boolean hasProject(String projectName)
    {
        Map<String, String> uuidCache = _getUUIDCache();
        if (uuidCache.containsKey(projectName))
        {
            // As cache is computed from default JCR workspace, we need to check if the project exists into the current JCR workspace
            return _resolver.hasAmetysObjectForId(uuidCache.get(projectName));
        }
        return false;
    }
    
    /**
     * Get all managers
     * @return the managers
     */
    public Set<UserIdentity> getManagers()
    {
        return getProjects()
            .stream()
            .map(Project::getManagers)
            .flatMap(Arrays::stream)
            .collect(Collectors.toSet());
    }
    
    /**
     * Determines if the current user is a manager of at least one project
     * @return true if the user is a manager
     */
    public boolean isManager()
    {
        return isManager(_currentUserProvider.getUser());
    }
    
    /**
     * Determines if the user is a manager of at least one project
     * @param user the user
     * @return true if the user is a manager
     */
    public boolean isManager(UserIdentity user)
    {
        AmetysObjectIterable<Project> projects = getProjects();
        for (Project project : projects)
        {
            if (isManager(project, user))
            {
                return true;
            }
        }
        return false;
    }
    
    /**
     * Determines if the user is a manager of the project
     * @param projectName the project name
     * @param user the user
     * @return true if the user is a manager
     */
    public boolean isManager(String projectName, UserIdentity user)
    {
        Project project = getProject(projectName);
        if (project != null)
        {
            return isManager(project, user);
        }
        return false;
    }
    
    /**
     * Determines if the user is a manager of the project
     * @param project the project
     * @param user the user
     * @return true if the user is a manager
     */
    public boolean isManager(Project project, UserIdentity user)
    {
        return ArrayUtils.contains(project.getManagers(), user);
    }

    /**
     * Can the current user access backoffice on the site of the current project
     * @param project The non null project to analyse
     * @return true if the user can access to the backoffice
     */
    public boolean canAccessBO(Project project)
    {
        Site site = project.getSite();
        if (site == null)
        {
            return false;
        }

        Request request = ContextHelper.getRequest(_context);
        String currentSiteName = (String) request.getAttribute("siteName");
        try
        {
            request.setAttribute("siteName", site.getName()); // Setting temporarily this attribute to check user rights on any object on this site
            return !_rightManager.getUserRights(_currentUserProvider.getUser(), "/cms").isEmpty();
        }
        finally
        {
            request.setAttribute("siteName", currentSiteName);
        }
    }
    
    /**
     * Can the current user leave the project
     * @param project The non null project to analyse
     * @return true if the user can leave the project
     */
    public boolean canLeaveProject(Project project)
    {
        if (project == null)
        {
            return false;
        }
        
        Site site = project.getSite();
        if (site == null)
        {
            return false;
        }
        
        UserIdentity userIdentity = _currentUserProvider.getUser();
        
        // As a user in a group can't leave the project, check if he is registered as User member
        Set<ProjectMember> members = _projectMemberManager.getProjectMembers(project, false);
        ProjectMember projectMember = members.stream()
                .filter(member -> MemberType.USER == member.getType())
                .filter(member -> userIdentity.equals(member.getUser().getIdentity()))
                .findFirst()
                .orElse(null);
        
        // The user is either on a group, or has not been found
        if (projectMember == null)
        {
            return false;
        }

        // The user is the only remaining manager, and therefore can not leave the project
        if (_projectMemberManager.isOnlyManager(project, userIdentity))
        {
            return false;
        }
        
        return true;
    }
    
    /**
     * Retrieves the mapping of all the projects name with their title on which the current user has access
     * @return the map (projectName, projectTitle) for all projects
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public List<Map<String, Object>> getUserProjectsData()
    {
        return getUserProjects(_currentUserProvider.getUser())
                .keySet()
                .stream()
                .map(p -> _project2json(p))
                .collect(Collectors.toList());
    }

    /**
     * Retrieves the users that have not been yet added to a project with a given criteria
     * @param projectName the project name
     * @param limit limit of request
     * @param criteria the criteria of the search
     * @param previousSearchData the previous search data to compute offset. Null if first search
     * @return list of users
     */
    @SuppressWarnings("unchecked")
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> searchUserByProject(String projectName, int limit, String criteria, Map<String, Object> previousSearchData)
    {

        Project project = this.getProject(projectName);

        if (!_projectRightHelper.canAddMember(project))
        {
            throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to do read operation without convenient right");
        }
        
        Map<String, Object> results = new HashMap<>();
        Site site = project.getSite();
        
        Set<String> projectMemberList = _projectMemberManager.getProjectMembers(project, false)
                .stream()
                .map(member ->
                {
                    if (member.getType() == MemberType.USER)
                    {
                        return UserIdentity.userIdentityToString(member.getUser().getIdentity());
                    }
                    else
                    {
                        GroupIdentity groupIdentityAsString = member.getGroup().getIdentity();
                        return GroupIdentity.groupIdentityToString(groupIdentityAsString);
                    }
                })
                .collect(Collectors.toSet());
        
        Set<String> contexts = new HashSet<>(Arrays.asList("/sites/" + site.getName(), "/sites-fo/" + site.getName()));
        
        Map<String, Object> params = new HashMap<>();
        params.put("pattern", criteria);
        
        Map<String, Object> result = new HashMap<>();
        Map<String, Object> searchData = previousSearchData;
        List<Map<String, Object>> memberList = new ArrayList<>();

        Set<String> groupDirectories = _directoryContextHelper.getGroupDirectoriesOnContexts(contexts);
        Set<String> userPopulations = getPopulation(site, true);

        do
        {
            result = _userAndGroupSearchManager.searchUsersAndGroup(userPopulations, groupDirectories, limit - memberList.size(), searchData, params, true);
            List<Map<String, Object>> filteredMembers = ((List<Map<String, Object>>) result.get("results"))
                    .stream()
                    .filter(member ->
                    {
                        return !projectMemberList.contains(member.get("login") + "#" + member.get("populationId")) && !projectMemberList.contains(member.get("id") + "#" + member.get("groupDirectory"));
                    }).collect(Collectors.toList());
            searchData = (Map<String, Object>) result.get("searchData");
            memberList.addAll(filteredMembers);
        }
        while (!result.containsKey("finished") && memberList.size() < limit);

        results.put("searchData", searchData);
        results.put("memberList", memberList);
        return results;
    }
    
    /**
     * Get the populations of the project
     * @param site the project site
     * @param excludeConfigurationPopulations true to exclude populations configured in the catalog site or in the project site
     * @return the populations of the project
     */
    public Set<String> getPopulation(Site site, boolean excludeConfigurationPopulations)
    {
        Set<String> contexts = new HashSet<>(Arrays.asList("/sites/" + site.getName(), "/sites-fo/" + site.getName()));

        Set<String> userPopulations = _populationContextHelper.getUserPopulationsOnContexts(contexts, false, true);
        
        if (excludeConfigurationPopulations)
        {
            Site catalogSite = _siteManager.getSite(getCatalogSiteName());
            
            // Get the excluded population configuration
            String excludedPopulationsString = catalogSite.getValue(CatalogSiteType.PROJECT_EXTERNAL_POPULATIONS_SITE_PARAM);
            
            // Check if project site overrides catalog configuration
            String externalPopulationPolicy = site.getValue(ProjectWorkspaceSiteType.PROJECT_OVERRIDE_EXTERNAL_POPULATION_POLICY_SITE_PARAM, false, ProjectWorkspaceSiteType.ExternalPopulationPolicy.GLOBAL.name());
            if (externalPopulationPolicy.equals(ProjectWorkspaceSiteType.ExternalPopulationPolicy.EXTERNAL_POPULATION_BY_PROJECT.name()))
            {
                excludedPopulationsString = site.getValue(ProjectWorkspaceSiteType.PROJECT_EXTERNAL_POPULATIONS_PROJECT_SITE_PARAM);
            }
            
            List<String> excludedPopulationsList = Arrays.asList(StringUtils.split(StringUtils.defaultString(excludedPopulationsString), ","));
            
            if (!excludedPopulationsList.isEmpty())
            {
                userPopulations.removeAll(excludedPopulationsList);
            }
        }

        return userPopulations;
    }

    /**
     * Retrieves the mapping of all the projects name with their title (regarless user rights)
     * @return the map (projectName, projectTitle) for all projects
     */
    @Callable(rights = "Runtime_Rights_Admin_Access", context = "/admin")
    public List<Map<String, Object>> getProjectsData()
    {
        return getProjects()
                .stream()
                .map(p -> _project2json(p))
                .collect(Collectors.toList());
    }
    
    /**
     * Get the project's main properties as json object
     * @param project the project
     * @return the json representation of project
     */
    protected Map<String, Object> _project2json(Project project)
    {
        Map<String, Object> json = new HashMap<>();
        
        json.put("id", project.getId());
        json.put("name", project.getName());
        json.put("title", project.getTitle());
        json.put("url", getProjectUrl(project, StringUtils.EMPTY));
        
        return json;
    }
    /**
     * Retrieves the project names
     * @return the project names
     */
    @Callable(rights = "Runtime_Rights_Admin_Access", context = "/admin")
    public Collection<String> getProjectNames()
    {
        // As cache is computed from default JCR workspace, we need to filter on sites that exist into the current JCR workspace
        return _getUUIDCache().entrySet().stream()
                .filter(e -> _resolver.hasAmetysObjectForId(e.getValue()))
                .map(Map.Entry::getKey)
                .collect(Collectors.toList());
    }
    
    /**
     * Return the root for projects
     * The root node will be created if necessary
     * @return The root for projects
     */
    public ModifiableTraversableAmetysObject getProjectsRoot()
    {
        try
        {
            ModifiableTraversableAmetysObject pluginsNode = _resolver.resolveByPath("/ametys:plugins");
            ModifiableTraversableAmetysObject workspacesPluginNode = _getOrCreateObject(pluginsNode, __WORKSPACES_PLUGIN_NODE_NAME, __WORKSPACES_PLUGIN_NODE_TYPE);
            return _getOrCreateObject(workspacesPluginNode, __PROJECTS_ROOT_NODE_NAME, __PROJECTS_ROOT_NODE_TYPE);
        }
        catch (AmetysRepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the projects root node.", e);
        }
    }
    
    /**
     * Retrieves the standard information of a project
     * @param projectId Identifier of the project
     * @return The map of information
     */
    @Callable(rights = "Runtime_Rights_Admin_Access", context = "/admin")
    public Map<String, Object> getProjectProperties(String projectId)
    {
        return getProjectProperties((Project) _resolver.resolveById(projectId));
    }
    
    /**
     * Retrieves the standard information of a project
     * @param project The project
     * @return The map of information
     */
    public Map<String, Object> getProjectProperties(Project project)
    {
        Map<String, Object> info = new HashMap<>();

        info.put("id", project.getId());
        info.put("name", project.getName());
        info.put("type", "project");

        info.put("title", project.getTitle());
        info.put("description", project.getDescription());
        info.put("inscriptionStatus", project.getInscriptionStatus().toString());
        info.put("defaultProfile", project.getDefaultProfile());

        info.put("creationDate", project.getCreationDate());

        // check if the project workspace configuration is valid
        Site site = project.getSite();
        boolean valid = site != null && _siteConfigurationManager.isSiteConfigurationValid(site);
        
        Set<String> categories = project.getCategories();
        info.put("categories", categories.stream()
                .map(c -> _categoryProviderEP.getTag(c, new HashMap<>()))
                .filter(Objects::nonNull)
                .map(t -> _tag2json(t))
                .collect(Collectors.toList()));
        
        Set<String> tags = project.getTags();
        info.put("tags", tags.stream()
                .map(c -> _projectTagProviderEP.getTag(c, new HashMap<>()))
                .filter(Objects::nonNull)
                .map(t -> _tag2json(t))
                .collect(Collectors.toList()));
        
        info.put("valid", valid);
        
        UserIdentity[] managers = project.getManagers();
        info.put("managers", Arrays.stream(managers)
                .map(u -> _userHelper.user2json(u))
                .collect(Collectors.toList()));
        
        Map<String, String> siteProps = new HashMap<>();
        // site map with id ,name, title and url property
        // { id: site id, name: site name, title: site title, url: site url }
        if (site != null)
        {
            siteProps.put("id", site.getId());
            siteProps.put("name", site.getName());
            siteProps.put("title", site.getTitle());
            siteProps.put("url", site.getUrl());
        }
        info.put("site", siteProps);

        return info;
    }
    
    private Map<String, Object> _tag2json(Tag tag)
    {
        Map<String, Object> json = new HashMap<>();
        json.put("id", tag.getId());
        json.put("name", tag.getName());
        json.put("title", tag.getTitle());
        return json;
    }

    /**
     * Get the project URL.
     * @param project The project
     * @param defaultValue The default value to use if there is no site
     * @return The project URL if a site is configured, otherwise return the default value.
     */
    public String getProjectUrl(Project project, String defaultValue)
    {
        Site site = project.getSite();
        if (site == null)
        {
            return defaultValue;
        }
        else
        {
            return site.getUrl();
        }
    }
    
    /**
     * Create a project
     * @param name The project name
     * @param title The project title
     * @param description The project description
     * @param emailList Project mailing list
     * @param inscriptionStatus The inscription status of the project
     * @param defaultProfile The default profile for new members
     * @return A map containing the id of the new project or an error key.
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> createProject(String name, String title, String description, String emailList, String inscriptionStatus, String defaultProfile)
    {
        checkRightsForProjectCreation(InscriptionStatus.valueOf(inscriptionStatus.toUpperCase()), null);

        Map<String, Object> result = new HashMap<>();
        List<String> errors = new ArrayList<>();
        
        Map<String, Object> additionalValues = new HashMap<>();
        additionalValues.put("description", description);
        additionalValues.put("emailList", emailList);
        additionalValues.put("inscriptionStatus", inscriptionStatus);
        additionalValues.put("defaultProfile", defaultProfile);
        
        Project project = createProject(name, title, additionalValues, null, errors);
        
        if (CollectionUtils.isEmpty(errors))
        {
            result.put("id", project.getId());
        }
        else
        {
            result.put("error", errors.get(0));
        }
        
        return result;
    }
    
    /**
     * Create a project
     * @param name The project name
     * @param title The project title
     * @param additionalValues A list of optional additional values. Accepted values are : description, mailingList, inscriptionStatus, defaultProfile, tags, categoryTags, keywords and language
     * @param modulesIds The list of modules to activate. Can be null to activate all modules
     * @param errors A list that will be populated with the encountered errors. If null, errors will not be tracked.
     * @return The id of the new project
     */
    public Project createProject(String name, String title, Map<String, Object> additionalValues, Set<String> modulesIds, List<String> errors)
    {
        if (StringUtils.isEmpty(title))
        {
            throw new IllegalArgumentException(String.format("Cannot create project. Title is mandatory"));
        }
        
        ModifiableTraversableAmetysObject projectsRoot = getProjectsRoot();
        
        // Project name should be unique
        if (hasProject(name))
        {
            if (getLogger().isWarnEnabled())
            {
                getLogger().warn(String.format("A project with the name '%s' already exists", name));
            }
            
            if (errors != null)
            {
                errors.add("project-exists");
            }
            
            return null;
        }
        
        Project project = projectsRoot.createChild(name, Project.NODE_TYPE);
        project.setTitle(title);
        String description = (String) additionalValues.getOrDefault("description", null);
        if (StringUtils.isNotEmpty(description))
        {
            project.setDescription(description);
        }
        String mailingList = (String) additionalValues.getOrDefault("emailList", null);
        if (StringUtils.isNotEmpty(mailingList))
        {
            project.setMailingList(mailingList);
        }
        
        String inscriptionStatus = (String) additionalValues.getOrDefault("inscriptionStatus", null);
        if (StringUtils.isNotEmpty(inscriptionStatus))
        {
            project.setInscriptionStatus(inscriptionStatus);
        }
        
        String defaultProfile = (String) additionalValues.getOrDefault("defaultProfile", null);
        if (StringUtils.isNotEmpty(defaultProfile))
        {
            project.setDefaultProfile(defaultProfile);
        }
        
        @SuppressWarnings("unchecked")
        List<String> tags = (List<String>) additionalValues.getOrDefault("tags", null);
        if (tags != null)
        {
            project.setTags(tags);
        }
        @SuppressWarnings("unchecked")
        List<String> categoryTags = (List<String>) additionalValues.getOrDefault("categoryTags", null);
        if (categoryTags != null)
        {
            project.setCategoryTags(categoryTags);
        }
        
        @SuppressWarnings("unchecked")
        List<String> keywords = (List<String>) additionalValues.getOrDefault("keywords", null);
        if (keywords != null)
        {
            project.setKeywords(keywords.toArray(new String[keywords.size()]));
        }

        project.setCreationDate(ZonedDateTime.now());
        
        // Create the project workspace = a site + a set of pages
        _createProjectWorkspace(project, errors);
        
        activateModules(project, modulesIds, additionalValues);
        
        if (CollectionUtils.isEmpty(errors))
        {
            project.saveChanges();
         
            // Notify observers
            Map<String, Object> eventParams = new HashMap<>();
            eventParams.put(ObservationConstants.ARGS_PROJECT, project);
            _observationManager.notify(new Event(ObservationConstants.EVENT_PROJECT_ADDED, _currentUserProvider.getUser(), eventParams));
            
        }
        else
        {
            deleteProject(project);
        }
        
        clearCaches();
        
        return project;
    }
    
    /**
     * Edit a project
     * @param id The project identifier
     * @param title The title to set
     * @param description The description to set
     * @param mailingList Project mailing list
     * @param inscriptionStatus The inscription status of the project
     * @param defaultProfile The default profile for new members
     */
    @Callable(rights = ProjectConstants.RIGHT_PROJECT_EDIT, context = "/admin")
    public void editProject(String id, String title, String description, String mailingList, String inscriptionStatus, String defaultProfile)
    {
        Project project = _resolver.resolveById(id);
        editProject(project, title, description, mailingList, inscriptionStatus, defaultProfile);
    }
    
    /**
     * Edit a project
     * @param project The project
     * @param title The title to set
     * @param description The description to set
     * @param mailingList Project mailing list
     * @param inscriptionStatus The inscription status of the project
     * @param defaultProfile The default profile for new members
     */
    public void editProject(Project project, String title, String description, String mailingList, String inscriptionStatus, String defaultProfile)
    {
        checkRightsForProjectEdition(project, InscriptionStatus.valueOf(inscriptionStatus.toUpperCase()), null);
        
        project.setTitle(title);
        
        if (StringUtils.isNotEmpty(description))
        {
            project.setDescription(description);
        }
        else
        {
            project.removeDescription();
        }
        
        if (StringUtils.isNotEmpty(mailingList))
        {
            project.setMailingList(mailingList);
        }
        else
        {
            project.removeMailingList();
        }

        project.setInscriptionStatus(inscriptionStatus);
        project.setDefaultProfile(defaultProfile);
        
        project.saveChanges();
        
        // Notify observers
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(ObservationConstants.ARGS_PROJECT, project);
        eventParams.put(org.ametys.plugins.workspaces.ObservationConstants.ARGS_PROJECT_ID, project.getId());
        _observationManager.notify(new Event(ObservationConstants.EVENT_PROJECT_UPDATED, _currentUserProvider.getUser(), eventParams));
    }
    
    /**
     * Delete a list of project.
     * @param ids The ids of projects to delete
     * @return The ids of the deleted projects, unknowns projects and the deleted sites
     */
    @Callable(rights = ProjectConstants.RIGHT_PROJECT_DELETE, context = "/admin")
    public Map<String, Object> deleteProjectsByIds(List<String> ids)
    {
        Map<String, Object> result = new HashMap<>();
        List<Map<String, Object>> deleted = new ArrayList<>();
        List<String> unknowns = new ArrayList<>();
        
        for (String id : ids)
        {
            try
            {
                Project project = _resolver.resolveById(id);
                
                Map<String, Object> projectInfo = new HashMap<>();
                projectInfo.put("id", id);
                projectInfo.put("title", project.getTitle());
                projectInfo.put("sites", deleteProject(project));
                
                deleted.add(projectInfo);
            }
            catch (UnknownAmetysObjectException e)
            {
                getLogger().warn(String.format("Unable to delete the definition of id '%s', because it does not exist.", id), e);
                unknowns.add(id);
            }
        }
        
        result.put("deleted", deleted);
        result.put("unknowns", unknowns);
        
        return result;
    }
    
    /**
     * Delete a project.
     * @param projects The list of projects to delete
     * @return list of deleted sites (each list entry contains a data map with
     *         the id and the name of the delete site).
     */
    public List<Map<String, String>> deleteProject(List<Project> projects)
    {
        List<Map<String, String>> deletedSitesInfo = new ArrayList<>();
        
        for (Project project : projects)
        {
            deletedSitesInfo.addAll(deleteProject(project));
        }
        
        return deletedSitesInfo;
    }
    
    /**
     * Delete a project and its sites
     * @param project The project to delete
     * @return list of deleted sites (each list entry contains a data map with
     *         the id and the name of the delete site).
     */
    public List<Map<String, String>> deleteProject(Project project)
    {
        ModifiableAmetysObject parent = project.getParent();
        
        
        // list of map entry with id, name and title property
        // { id: site id, name: site name }
        List<Map<String, String>> deletedSitesInfo = new ArrayList<>();
        
        Site site = project.getSite();
        if (site != null)
        {
            try
            {
                Map<String, String> siteProps = new HashMap<>();
                siteProps.put("id", site.getId());
                siteProps.put("name", site.getName());
                
                _siteDao.deleteSite(site.getId());
                deletedSitesInfo.add(siteProps);
            }
            catch (RepositoryException e)
            {
                String errorMsg = String.format("Error while trying to delete the site '%s' for the project '%s'.", site.getName(), project.getName());
                getLogger().error(errorMsg, e);
            }
        }
        
        String projectId = project.getId();
        Collection<ProjectMember> projectMembers = _projectMembers.getProjectMembers(project, true);
        project.remove();
        parent.saveChanges();
        
        // Notify observers
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(ObservationConstants.ARGS_PROJECT_ID, projectId);
        eventParams.put(ObservationConstants.ARGS_PROJECT_NAME, project.getName());
        eventParams.put(ObservationConstants.ARGS_PROJECT_MEMBERS, projectMembers);
        _observationManager.notify(new Event(ObservationConstants.EVENT_PROJECT_DELETED, _currentUserProvider.getUser(), eventParams));
        
        clearCaches();
        
        return deletedSitesInfo;
    }
    
    /**
     * Utility method to get or create an ametys object
     * @param <A> A sub class of AmetysObject
     * @param parent The parent object
     * @param name The ametys object name
     * @param type The ametys object type
     * @return ametys object
     * @throws AmetysRepositoryException if an repository error occurs
     */
    private <A extends AmetysObject> A _getOrCreateObject(ModifiableTraversableAmetysObject parent, String name, String type) throws AmetysRepositoryException
    {
        A object;
        
        if (parent.hasChild(name))
        {
            object = parent.getChild(name);
        }
        else
        {
            object = parent.createChild(name, type);
            parent.saveChanges();
        }
        
        return object;
    }
    
    /**
     * Get the project of an ametys object inside a project.
     * It can be an explorer node, or any type of resource in a module.
     * @param id The identifier of the ametys object
     * @return the project or null if not found
     */
    public Project getParentProject(String id)
    {
        return getParentProject(_resolver.<AmetysObject>resolveById(id));
    }
    
    /**
     * Get the project of an ametys object inside a project.
     * It can be an explorer node, or any type of resource in a module.
     * @param object The ametys object
     * @return the project or null if not found
     */
    public Project getParentProject(AmetysObject object)
    {
        AmetysObject ametysObject = object;
        // Go back to the local explorer root.
        do
        {
            ametysObject = ametysObject.getParent();
        }
        while (ametysObject instanceof ExplorerNode);
        
        if (!(ametysObject instanceof Project))
        {
            getLogger().warn(String.format("No project found for ametys object with id '%s'", ametysObject.getId()));
            return null;
        }
        
        return (Project) ametysObject;
    }
    
    /**
     * Get the list of project names for a given site
     * @param siteName The site name
     * @return the list of project names
     */
    @Callable(rights = "Runtime_Rights_Admin_Access", context = "/admin")
    public List<String> getProjectsForSite(String siteName)
    {
        Cache<String, List<Pair<String, String>>> cache = _getMemorySiteAssociationCache();
        if (cache.hasKey(siteName))
        {
            return cache.get(siteName).stream()
                    .map(p -> p.getRight())
                    .collect(Collectors.toList());
        }
        else
        {
            List<String> projectNames = new ArrayList<>();
            
            if (_siteManager.hasSite(siteName))
            {
                Site site = _siteManager.getSite(siteName);
                getProjectsForSite(site)
                    .stream()
                    .map(Project::getName)
                    .forEach(projectNames::add);
            }
            
            return projectNames;
        }
    }
    
    /**
     * Get the list of project for a given site
     * @param site The site
     * @return the list of project
     */
    public List<Project> getProjectsForSite(Site site)
    {
        Cache<String, List<Pair<String, String>>> cache = _getMemorySiteAssociationCache();
        if (cache.hasKey(site.getName()))
        {
            return cache.get(site.getName()).stream()
                        .map(p -> _resolver.<Project>resolveById(p.getLeft()))
                        .collect(Collectors.toList());
        }
        
        try
        {
            // Stream over the weak reference properties pointing to this
            // node to find referencing projects
            Iterator<Property> propertyIterator = site.getNode().getWeakReferences();
            Iterable<Property> propertyIterable = () -> propertyIterator;
            
            List<Project> projects = StreamSupport.stream(propertyIterable.spliterator(), false)
                    .map(p ->
                    {
                        try
                        {
                            Node parent = p.getParent();

                            // Check if the parent is a project"
                            if (NodeTypeHelper.isNodeType(parent, "ametys:project"))
                            {
                                Project project = _resolver.resolve(parent, false);
                                return project;
                            }
                        }
                        catch (Exception e)
                        {
                            if (getLogger().isWarnEnabled())
                            {
                                // this weak reference is not from a project
                                String propertyPath = null;
                                try
                                {
                                    propertyPath = p.getPath();
                                }
                                catch (Exception e2)
                                {
                                    // ignore
                                }
                                
                                String warnMsg = String.format("Site '%s' is pointed by a weak reference '%s' which is not representing a relation with project. This reference is ignored.", site.getName(), propertyPath);
                                getLogger().warn(warnMsg);
                            }
                        }
                        
                        return null;
                    })
                    .filter(Objects::nonNull)
                    .collect(Collectors.toList());
            
            List<Pair<String, String>> projectsPairs = projects.stream().map(p -> Pair.of(p.getId(), p.getName())).collect(Collectors.toList());
            cache.put(site.getName(), projectsPairs);
            return projects;
        }
        catch (RepositoryException e)
        {
            getLogger().error(String.format("Unable to find projects for site '%s'", site.getName()), e);
        }
        
        return new ArrayList<>();
    }
    
    /**
     * Create the project workspace for a given project.
     * @param project The project for which the workspace must be created
     * @param errors A list of possible errors to populate. Can be null if the caller is not interested in error tracking.
     * @return The site created for this workspace
     */
    protected Site _createProjectWorkspace(Project project, List<String> errors)
    {
        String initialSiteName = project.getName();
        Site site = null;
        
        Site catalogSite = _siteManager.getSite(getCatalogSiteName());
        String rootId = catalogSite != null ? catalogSite.getId() : null;
        
        Map<String, Object> result = _siteDao.createSite(rootId, initialSiteName, ProjectWorkspaceSiteType.TYPE_ID, true);
        
        String siteId = (String) result.get("id");
        String siteName = (String) result.get("name");
        if (StringUtils.isNotEmpty(siteId))
        {
            // Creation success
            site = _siteManager.getSite(siteName);
            
            setProjectSiteTitle(site, project.getTitle());
          
            // Add site to project
            project.setSite(site);
            
            site.saveChanges();
        }
        
        return site;
    }

    /**
     * Get the project's tags
     * @return The project's tags
     */
    public List<String> getTags()
    {
        AmetysObject projectsRootNode = getProjectsRoot();
        if (projectsRootNode instanceof JCRAmetysObject)
        {
            Node node = ((JCRAmetysObject) projectsRootNode).getNode();
            
            try
            {
                return Arrays.stream(node.getProperty(__PROJECTS_TAGS_PROPERTY).getValues())
                    .map(LambdaUtils.wrap(Value::getString))
                    .collect(Collectors.toList());
            }
            catch (PathNotFoundException e)
            {
                // property is not set, empty list will be returned.
            }
            catch (RepositoryException e)
            {
                throw new AmetysRepositoryException(e);
            }
        }
        
        return new ArrayList<>();
    }

    /**
     * Add project's tags
     * @param newTags The new tags to add
     */
    public synchronized void addTags(Collection<String> newTags)
    {
        if (CollectionUtils.isNotEmpty(newTags))
        {
            AmetysObject projectsRootNode = getProjectsRoot();
            if (projectsRootNode instanceof JCRAmetysObject)
            {
                // Concat existing tags with new lowercased tags
                String[] tags = Stream.concat(getTags().stream(), newTags.stream().map(String::trim).map(String::toLowerCase).filter(StringUtils::isNotEmpty))
                        .distinct()
                        .toArray(String[]::new);
                
                try
                {
                    ((JCRAmetysObject) projectsRootNode).getNode().setProperty(__PROJECTS_TAGS_PROPERTY, tags);
                }
                catch (RepositoryException e)
                {
                    throw new AmetysRepositoryException(e);
                }
            }
        }
    }
    
    /**
     * Get the list of activated modules for a project
     * @param project The project
     * @return The list of activated modules
     */
    public List<WorkspaceModule> getModules(Project project)
    {
        return _moduleManagerEP.getModules().stream()
                .filter(module -> isModuleActivated(project, module.getId()))
                .collect(Collectors.toList());
    }
    
    /**
     * Retrieves the page of the module for all available languages
     * @param project The project
     * @param moduleId The project module id
     * @return the page or null if not found
     */
    public Set<Page> getModulePages(Project project, String moduleId)
    {
        if (_moduleManagerEP.hasExtension(moduleId))
        {
            WorkspaceModule module = _moduleManagerEP.getExtension(moduleId);
            return getModulePages(project, module);
        }
        return null;
    }
    
    /**
     * Return the possible module roots associated to a page
     * @param page The given page
     * @return A non null set of the data of the linked modules
     */
    public Set<ModifiableResourceCollection> pageToModuleRoot(Page page)
    {
        Set<ModifiableResourceCollection> data = new LinkedHashSet<>();
        
        Page rootPage = page;
        SitemapElement parent = page.getParent();
        while (!(parent instanceof Sitemap) && !page.hasValue(__PAGE_MODULES_VALUE))
        {
            rootPage = (Page) parent;
            parent = parent.getParent();
        }
        
        String[] modulesRootsIds = rootPage.getValue(__PAGE_MODULES_VALUE, new String[0]);
        if (modulesRootsIds.length > 0)
        {
            for (String moduleRootId : modulesRootsIds)
            {
                try
                {
                    ModifiableResourceCollection moduleRoot = _resolver.resolveById(moduleRootId);
                    data.add(moduleRoot);
                }
                catch (UnknownAmetysObjectException e)
                {
                    // Ignore obsolete data
                }
            }
        }
        
        return data;
    }
    
    /**
     * Mark the given page as this module page. The modified page will not be saved.
     * @param page The page to change
     * @param moduleRoot The workspace module that use this page
     */
    public void tagProjectPage(ModifiablePage page, ModifiableResourceCollection moduleRoot)
    {
        String[] currentModules = page.getValue(__PAGE_MODULES_VALUE, new String[0]);
        
        Set<String> modules = new LinkedHashSet<>(Arrays.asList(currentModules));
        modules.add(moduleRoot.getId());
        
        String[] newModules = new String[modules.size()];
        modules.toArray(newModules);
        
        page.setValue(__PAGE_MODULES_VALUE, newModules);
    }

    /**
     * Remove the mark on the given page of this module. The modified page will not be saved.
     * @param page The page to change
     * @param moduleRoot The workspace module that use this page
     */
    public void untagProjectPage(ModifiablePage page, ModifiableResourceCollection moduleRoot)
    {
        if (moduleRoot != null)
        {
            String[] currentModules = page.getValue(__PAGE_MODULES_VALUE, new String[0]);
            
            Set<String> modules = new LinkedHashSet<>(Arrays.asList(currentModules));
            modules.remove(moduleRoot.getId());
            
            String[] newModules = new String[modules.size()];
            modules.toArray(newModules);
            
            page.setValue(__PAGE_MODULES_VALUE, newModules);
        }
    }

    /**
     * Get a page in the site of a given project with a specific tag
     * @param project The project
     * @param workspaceModule the module
     * @return The module's pages
     */
    public Set<Page> getModulePages(Project project, WorkspaceModule workspaceModule)
    {
        Request request = _getRequest();
        if (request == null)
        {
            // There is no request to store cache
            return _computePages(project, workspaceModule);
        }
        
        Cache<RequestModuleCacheKey, Set<Page>> pagesCache = _getRequestPageCache();
        
        // The site key in the cache is of the form {site + workspace}.
        String currentWorkspace = _workspaceSelector.getWorkspace();
        RequestModuleCacheKey pagesKey = RequestModuleCacheKey.of(project.getName(), workspaceModule.getId(), currentWorkspace);
        
        try
        {
            return pagesCache.get(pagesKey, __ -> _computePages(project, workspaceModule));
        }
        catch (CacheException e)
        {
            if (e.getCause() instanceof UnknownAmetysObjectException)
            {
                throw (UnknownAmetysObjectException) e.getCause();
            }
            else
            {
                throw new RuntimeException("An error occurred while computing page of module " + workspaceModule.getModuleName() + " in project " + project.getName(), e);
            }
        }
    }
    
    private Set<Page> _computePages(Project project, WorkspaceModule workspaceModule)
    {
        Set<String> pagesUUids = _getMemoryPageCache().get(ModuleCacheKey.of(project.getName(), workspaceModule.getId()), __ -> _computePagesIds(project, workspaceModule));
        if (pagesUUids != null)
        {
            return pagesUUids.stream().map(uuid -> _resolver.<Page>resolveById(uuid)).collect(Collectors.toSet());
        }
        else
        {
            // Project may be present in cache for 'default' workspace but does not exist in current JCR workspace
            throw new UnknownAmetysObjectException("There is no pages for '" + project.getName() + "', module '" + workspaceModule.getModuleName() + "'");
        }
    }
    
    private Set<String> _computePagesIds(Project project, WorkspaceModule workspaceModule)
    {
        Site site = project.getSite();
        String siteName = site != null ? site.getName() : null;
        if (StringUtils.isEmpty(siteName))
        {
            return null;
        }
        
        ModifiableResourceCollection moduleRoot = workspaceModule.getModuleRoot(project, false);
        if (moduleRoot != null)
        {
            Expression expression = new StringExpression(__PAGE_MODULES_VALUE, Operator.EQ, moduleRoot.getId());
            String query = PageQueryHelper.getPageXPathQuery(siteName, null, null, expression, null);
    
            return StreamSupport.stream(_resolver.query(query).spliterator(), false)
                        .map(page -> page.getId())
                        .collect(Collectors.toSet());
        }
        else
        {
            return Set.of();
        }
    }

    /**
     * Activate the list of module of the project
     * @param project The project
     * @param moduleIds The list of modules. Can be null to activate all modules
     * @param additionalValues A list of optional additional values. Accepted values are : description, mailingList, inscriptionStatus, defaultProfile, tags, categoryTags, keywords and language
     */
    public void activateModules(Project project, Set<String> moduleIds, Map<String, Object> additionalValues)
    {
        Set<String> modules = moduleIds == null ? _moduleManagerEP.getExtensionsIds() : moduleIds;
        
        for (String moduleId : modules)
        {
            WorkspaceModule module = _moduleManagerEP.getModule(moduleId);
            if (module != null && !isModuleActivated(project, moduleId))
            {
                module.activateModule(project, additionalValues);
                project.addModule(moduleId);
            }
        }
        
        _setDefaultProfileForMembers(project, modules);
        
        project.saveChanges();
    }

    private void _setDefaultProfileForMembers(Project project, Set<String> modules)
    {
        String profileForNewModule = StringUtils.defaultString(Config.getInstance().getValue("workspaces.profile.new.module"));
        
        if (profileForNewModule.equals(ProfileForNewModule.DEFAULT_MEMBER_PROFILE.name()))
        {
            Set<String> defaultProfiles = Set.of(StringUtils.defaultString(Config.getInstance().getValue("workspaces.profile.default")));
            Map<JCRProjectMember, Object> projectMembers = _projectMembers.getJCRProjectMembers(project);
            
            for (String moduleId : modules)
            {
                WorkspaceModule module = _moduleManagerEP.getModule(moduleId);
                for (JCRProjectMember member : projectMembers.keySet())
                {
                    
                    _projectMembers.setProfileOnModule(member, project, module, defaultProfiles);
                }
            }
        }
    }
    
    /**
     * Initialize the sitemap with the active module of the project
     * @param project The project
     * @param sitemap The sitemap
     */
    public void initializeModulesSitemap(Project project, Sitemap sitemap)
    {
        Set<String> modules = _moduleManagerEP.getExtensionsIds();
        
        for (String moduleId : modules)
        {
            if (_moduleManagerEP.hasExtension(moduleId))
            {
                WorkspaceModule module = _moduleManagerEP.getExtension(moduleId);
                
                if (isModuleActivated(project, moduleId))
                {
                    module.initializeSitemap(project, sitemap);
                }
            }
        }
    }
    
    /**
     * Determines if a module is activated
     * @param project The project
     * @param moduleId The id of module
     * @return true if the module the currently activated
     */
    public boolean isModuleActivated(Project project, String moduleId)
    {
        return ArrayUtils.contains(project.getModules(), moduleId);
    }
    
    /**
     * Remove the explorer root node of the project module, remove all events
     * related to that module and set it to deactivated
     * @param project The project
     * @param moduleIds The id of module to activate
     */
    public void deactivateModules(Project project, Set<String> moduleIds)
    {
        for (String moduleId : moduleIds)
        {
            WorkspaceModule module = _moduleManagerEP.getModule(moduleId);
            if (module != null && isModuleActivated(project, moduleId))
            {
                module.deactivateModule(project);
                project.removeModule(moduleId);
            }
        }
        
        project.saveChanges();
    }
    
    
    /**
     * Get the list of profiles configured for the workspaces' projects
     * @return The list of profiles as JSON
     */
    @Callable(rights = {ProjectConstants.RIGHT_PROJECT_CREATE_PRIVATE, ProjectConstants.RIGHT_PROJECT_CREATE_PUBLIC_MODERATED, ProjectConstants.RIGHT_PROJECT_CREATE_PUBLIC_OPENED, ProjectConstants.RIGHT_PROJECT_EDIT}, context = "/admin")
    public Map<String, Object> getProjectProfiles()
    {
        Map<String, Object> result = new HashMap<>();
        List<Map<String, Object>> profiles = _projectRightHelper.getProfiles().stream().map(p -> p.toJSON()).collect(Collectors.toList());
        result.put("profiles", profiles);
        return result;
    }
    
    /**
     * Get the site name holding the catalog of projects
     * @return the catalog's site name
     * @throws UnknownCatalogSiteException when the config is invalid
     */
    public String getCatalogSiteName() throws UnknownCatalogSiteException
    {
        String catalogSiteName = Config.getInstance().getValue("workspaces.catalog.site.name");
        if (!_siteManager.hasSite(catalogSiteName))
        {
            throw new UnknownCatalogSiteException("Unknown site '" + catalogSiteName + "'. The global Ametys configuration is invalid for the parameter 'workspaces.catalog.site.name'");
        }
        return catalogSiteName;
    }
    
    /**
     * Get the site name holding the users directory
     * @return the users directory's site name
     * @throws UnknownUserDirectorySiteException when the config is invalid
     */
    public String getUsersDirectorySiteName() throws UnknownUserDirectorySiteException
    {
        String udSiteName = Config.getInstance().getValue("workspaces.member.userdirectory.site.name");
        if (!_siteManager.hasSite(udSiteName))
        {
            throw new UnknownUserDirectorySiteException("Unknown site '" + udSiteName + "'. The global Ametys configuration is invalid for the parameter 'workspaces.member.userdirectory.site.name'");
        }
        return udSiteName;
    }
    
    @Override
    public boolean supports(Event event)
    {
        return event.getId().equals(ObservationConstants.EVENT_PROJECT_DELETED)
                || event.getId().equals(ObservationConstants.EVENT_PROJECT_UPDATED)
                || event.getId().equals(ObservationConstants.EVENT_PROJECT_ADDED)
                || event.getId().equals(org.ametys.web.ObservationConstants.EVENT_PAGE_ADDED)
                || event.getId().equals(org.ametys.web.ObservationConstants.EVENT_PAGE_DELETED);
    }

    public int getPriority()
    {
        return 0;
    }

    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        clearCaches();
    }

    /**
     * Prefix project title
     * @param site the site
     * @param title the title
     */
    public void setProjectSiteTitle(Site site, String title)
    {
        I18nizableText i18nSiteTitle = new I18nizableText("plugin." + _pluginName, "PLUGINS_WORKSPACES_PROJECT_DEFAULT_PROJECT_WORKSPACE_TITLE", Arrays.asList(title));
        site.setTitle(_i18nUtils.translate(i18nSiteTitle));
    }
    
    private Project _computeProject(String projectName)
    {
        if (hasProject(projectName))
        {
            String uuid = _getUUIDCache().get(projectName);
            return _resolver.<Project>resolveById(uuid);
        }
        else
        {
            // Project may be present in cache for 'default' workspace but does not exist in current JCR workspace
            throw new UnknownAmetysObjectException("There is no site named '" + projectName + "'");
        }
    }
    
    /**
     * Check rights to create project
     * @param inscriptionStatus the inscription status
     * @param zoneItem the zoneItem containing catalog service. Must be a catalog service if not null
     */
    public void checkRightsForProjectCreation(InscriptionStatus inscriptionStatus, ZoneItem zoneItem)
    {
        SitemapElement catalogPage = zoneItem != null ? zoneItem.getZone().getSitemapElement() : null;
        
        if (catalogPage != null && !_projectRightHelper.hasCatalogReadAccess(zoneItem))
        {
            throw new AccessDeniedException("User " + _currentUserProvider.getUser() + " tried to create project from page '" + catalogPage.getId() + "' without sufficient rights");
        }
        
        switch (inscriptionStatus)
        {
            case PRIVATE:
                boolean hasRightToCreatePrivateProjetOnPages = catalogPage != null ? _rightManager.currentUserHasRight(ProjectConstants.RIGHT_PROJECT_CREATE_PRIVATE, catalogPage) == RightResult.RIGHT_ALLOW : false;
                if (!hasRightToCreatePrivateProjetOnPages && _rightManager.currentUserHasRight(ProjectConstants.RIGHT_PROJECT_CREATE_PRIVATE, "/${WorkspaceName}") != RightResult.RIGHT_ALLOW)
                {
                    throw new AccessDeniedException("User " + _currentUserProvider.getUser() + " tried to create private project without sufficient rights");
                }
                break;
            case MODERATED:
                boolean hasRightToCreateModeratedProjetOnPages = catalogPage != null ? _rightManager.currentUserHasRight(ProjectConstants.RIGHT_PROJECT_CREATE_PUBLIC_MODERATED, catalogPage) == RightResult.RIGHT_ALLOW : false;
                if (!hasRightToCreateModeratedProjetOnPages && _rightManager.currentUserHasRight(ProjectConstants.RIGHT_PROJECT_CREATE_PUBLIC_MODERATED, "/${WorkspaceName}") != RightResult.RIGHT_ALLOW)
                {
                    throw new AccessDeniedException("User " + _currentUserProvider.getUser() + " tried to create public project with moderation without sufficient rights");
                }
                break;
            case OPEN:
                boolean hasRightToCreateOpenProjetOnPages = catalogPage != null ? _rightManager.currentUserHasRight(ProjectConstants.RIGHT_PROJECT_CREATE_PUBLIC_OPENED, catalogPage) == RightResult.RIGHT_ALLOW : false;
                if (!hasRightToCreateOpenProjetOnPages && _rightManager.currentUserHasRight(ProjectConstants.RIGHT_PROJECT_CREATE_PUBLIC_OPENED, "/${WorkspaceName}") != RightResult.RIGHT_ALLOW)
                {
                    throw new AccessDeniedException("User " + _currentUserProvider.getUser() + " tried to create public project without sufficient rights");
                }
                break;
            default:
                throw new IllegalArgumentException("Inscription status '" + inscriptionStatus.toString() + "' is unknown");
        }
    }
    
    /**
     * Check rights to edit project
     * @param project the project
     * @param inscriptionStatus the inscription status
     * @param zoneItem the zoneItem containing catalog service. Must be a catalog service if not null
     */
    public void checkRightsForProjectEdition(Project project, InscriptionStatus inscriptionStatus, ZoneItem zoneItem)
    {
        InscriptionStatus oldInscriptionStatus = project.getInscriptionStatus();
        SitemapElement catalogPage = zoneItem != null ? zoneItem.getZone().getSitemapElement() : null;
        
        if (catalogPage != null && !_projectRightHelper.hasCatalogReadAccess(zoneItem))
        {
            throw new AccessDeniedException("User " + _currentUserProvider.getUser() + " tried to edit project from page '" + catalogPage.getId() + "' without sufficient rights");
        }
        
        if (oldInscriptionStatus != inscriptionStatus)
        {
            boolean canCreatePrivateProjet = _rightManager.currentUserHasRight(ProjectConstants.RIGHT_PROJECT_CREATE_PRIVATE, "/${WorkspaceName}") == RightResult.RIGHT_ALLOW
                    || (catalogPage != null ? _rightManager.currentUserHasRight(ProjectConstants.RIGHT_PROJECT_CREATE_PRIVATE, catalogPage) == RightResult.RIGHT_ALLOW : false);
            boolean canCreatePublicProjetWithModeration = _rightManager.currentUserHasRight(ProjectConstants.RIGHT_PROJECT_CREATE_PUBLIC_MODERATED, "/${WorkspaceName}") == RightResult.RIGHT_ALLOW
                    || (catalogPage != null ? _rightManager.currentUserHasRight(ProjectConstants.RIGHT_PROJECT_CREATE_PUBLIC_MODERATED, catalogPage) == RightResult.RIGHT_ALLOW : false);
            boolean canCreatePublicProjet = _rightManager.currentUserHasRight(ProjectConstants.RIGHT_PROJECT_CREATE_PUBLIC_OPENED, "/${WorkspaceName}") == RightResult.RIGHT_ALLOW
                    || (catalogPage != null ? _rightManager.currentUserHasRight(ProjectConstants.RIGHT_PROJECT_CREATE_PUBLIC_OPENED, catalogPage) == RightResult.RIGHT_ALLOW : false);
            
            switch (oldInscriptionStatus)
            {
                case PRIVATE:
                    if (!canCreatePrivateProjet)
                    {
                        throw new AccessDeniedException("User " + _currentUserProvider.getUser() + " tried to edit private project without sufficient rights");
                    }
                    break;
                case MODERATED:
                    if (!canCreatePublicProjetWithModeration)
                    {
                        throw new AccessDeniedException("User " + _currentUserProvider.getUser() + " tried to edit public project with moderation without sufficient rights");
                    }
                    break;
                case OPEN:
                    if (!canCreatePublicProjet)
                    {
                        throw new AccessDeniedException("User " + _currentUserProvider.getUser() + " tried to edit public project without sufficient rights");
                    }
                    break;
                default:
                    throw new IllegalArgumentException("Inscription status '" + oldInscriptionStatus.toString() + "' is unknown");
            }
            
            switch (inscriptionStatus)
            {
                case PRIVATE:
                    if (!canCreatePrivateProjet)
                    {
                        throw new AccessDeniedException("User " + _currentUserProvider.getUser() + " tried to edit project to private project without sufficient rights");
                    }
                    break;
                case MODERATED:
                    if (!canCreatePublicProjetWithModeration)
                    {
                        throw new AccessDeniedException("User " + _currentUserProvider.getUser() + " tried to edit project to public project with moderation without sufficient rights");
                    }
                    break;
                case OPEN:
                    if (!canCreatePublicProjet)
                    {
                        throw new AccessDeniedException("User " + _currentUserProvider.getUser() + " tried to edit project to public project without sufficient rights");
                    }
                    break;
                default:
                    throw new IllegalArgumentException("Inscription status '" + inscriptionStatus.toString() + "' is unknown");
            }
        }
    }
    
    /**
     * Clear the site cache
     */
    public void clearCaches ()
    {
        _getMemorySiteAssociationCache().invalidateAll();
        _getMemoryProjectCache().invalidateAll();
        _getMemoryPageCache().invalidateAll();
        _getRequestProjectCache().invalidateAll();
        _getRequestPageCache().invalidateAll();
    }
    
    private Cache<String, List<Pair<String, String>>> _getMemorySiteAssociationCache()
    {
        return _cacheManager.get(MEMORY_SITEASSOCIATION_CACHE);
    }
    
    private Cache<String, String> _getMemoryProjectCache()
    {
        return _cacheManager.get(MEMORY_PROJECTIDBYNAMECACHE);
    }
    
    private Cache<ModuleCacheKey, Set<String>> _getMemoryPageCache()
    {
        return _cacheManager.get(MEMORY_PAGESBYIDCACHE);
    }
    
    private Cache<RequestProjectCacheKey, Project> _getRequestProjectCache()
    {
        return _cacheManager.get(REQUEST_PROJECTBYID_CACHE);
    }
    
    private Cache<RequestModuleCacheKey, Set<Page>> _getRequestPageCache()
    {
        return _cacheManager.get(REQUEST_PAGESBYPROJECTANDMODULE_CACHE);
    }

    
    /**
     * Creates the caches
     */
    protected void _createCaches()
    {
        _cacheManager.createMemoryCache(MEMORY_SITEASSOCIATION_CACHE,
                new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_CACHE_PROJECT_MANAGER_LABEL"),
                new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_CACHE_PROJECT_MANAGER_DESCRIPTION"),
                true,
                null);
        _cacheManager.createMemoryCache(MEMORY_PROJECTIDBYNAMECACHE,
                new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_PROJECT_MANAGER_UUID_CACHE_LABEL"),
                new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_PROJECT_MANAGER_UUID_CACHE_DESCRIPTION"),
                true,
                null);
        _cacheManager.createMemoryCache(MEMORY_PAGESBYIDCACHE,
                new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_PROJECT_MANAGER_PAGEUUID_CACHE_LABEL"),
                new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_PROJECT_MANAGER_PAGEUUID_CACHE_DESCRIPTION"),
                true,
                null);
        _cacheManager.createRequestCache(REQUEST_PROJECTBYID_CACHE,
                new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_PROJECT_MANAGER_REQUEST_CACHE_LABEL"),
                new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_PROJECT_MANAGER_REQUEST_CACHE_DESCRIPTION"),
                false);
        _cacheManager.createRequestCache(REQUEST_PAGESBYPROJECTANDMODULE_CACHE,
                new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_PROJECT_MANAGER_PAGEREQUEST_CACHE_LABEL"),
                new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_PROJECT_MANAGER_PAGEREQUEST_CACHE_DESCRIPTION"),
                false);
    }
    
    private synchronized Map<String, String> _getUUIDCache()
    {
        if (!_getMemoryProjectCache().hasKey(__IS_CACHE_FILLED))
        {
            Session defaultSession = null;
            try
            {
                // Force default workspace to execute query
                defaultSession = _repository.login(RepositoryConstants.DEFAULT_WORKSPACE);
                
                String jcrQuery = "//element(*, ametys:project)";
                
                AmetysObjectIterable<Project> projects = _resolver.query(jcrQuery, defaultSession);
                
                for (Project project : projects)
                {
                    _getMemoryProjectCache().put(project.getName(), project.getId());
                }
                
                _getMemoryProjectCache().put(__IS_CACHE_FILLED, null);
            }
            catch (RepositoryException e)
            {
                throw new AmetysRepositoryException(e);
            }
            finally
            {
                if (defaultSession != null)
                {
                    defaultSession.logout();
                }
            }
        }
        
        Map<String, String> cacheAsMap = _getMemoryProjectCache().asMap();
        cacheAsMap.remove(__IS_CACHE_FILLED);
        return cacheAsMap;
    }
    
    private static final class RequestProjectCacheKey extends AbstractCacheKey
    {
        private RequestProjectCacheKey(String projectName, String workspaceName)
        {
            super(projectName, workspaceName);
        }
        
        static RequestProjectCacheKey of(String projectName, String workspaceName)
        {
            return new RequestProjectCacheKey(projectName, workspaceName);
        }
    }
    
    private static final class ModuleCacheKey extends AbstractCacheKey
    {
        private ModuleCacheKey(String projectName, String moduleId)
        {
            super(projectName, moduleId);
        }
        
        static ModuleCacheKey of(String projectName, String moduleId)
        {
            return new ModuleCacheKey(projectName, moduleId);
        }
    }
    
    private static final class RequestModuleCacheKey extends AbstractCacheKey
    {
        private RequestModuleCacheKey(String projectName, String moduleId, String workspaceName)
        {
            super(projectName, moduleId, workspaceName);
        }
        
        static RequestModuleCacheKey of(String projectName, String moduleId, String workspaceName)
        {
            return new RequestModuleCacheKey(projectName, moduleId, workspaceName);
        }
    }
    
    private Request _getRequest ()
    {
        try
        {
            return (Request) _context.get(ContextHelper.CONTEXT_REQUEST_OBJECT);
        }
        catch (ContextException ce)
        {
            getLogger().info("Unable to get the request", ce);
            return null;
        }
    }

    /**
     * Retrieves all projects for client side
     * @return the projects
     */
    @Callable(rights = "Runtime_Rights_Admin_Access", context = "/admin")
    public List<Map<String, Object>> getProjectsStatisticsForClientSide()
    {
        return getProjects()
                .stream()
                .map(p -> getProjectStatistics(p))
                .collect(Collectors.toList());
    }
    
    /**
     * Retrieves the standard information of a project
     * @param project The project
     * @return The map of information
     */
    public Map<String, Object> getProjectStatistics(Project project)
    {
        Map<String, Object> statistics = new HashMap<>();

        statistics.put("title", project.getTitle());

        long totalSize = 0;
        for (WorkspaceModule moduleManager : _moduleManagerEP.getModules())
        {
            Map<String, Object> moduleStatistics = moduleManager.getStatistics(project);
            statistics.putAll(moduleStatistics);
            Long size = (Long) moduleStatistics.get(moduleManager.getModuleSizeKey());
            totalSize += (size != null && size >= 0) ? (Long) moduleStatistics.get(moduleManager.getModuleSizeKey()) : 0;
        }

        statistics.put("totalSize", totalSize);

        ZonedDateTime creationDate = project.getCreationDate();
        
        statistics.put("creationDate", creationDate);
        statistics.put("managers", Arrays.stream(project.getManagers())
                .map(u -> _userHelper.user2json(u))
                .collect(Collectors.toList()));

        return statistics;
    }

    /**
     * Retrieves all projects for client side
     * @return the projects
     */
    @Callable(rights = "Runtime_Rights_Admin_Access", context = "/admin")
    public List<Map<String, Object>> getProjectsStatisticsColumnsModel()
    {
        return getStatisticHeaders()
                .stream()
                .map(p -> p.convertToJSON())
                .collect(Collectors.toList());
    }

    private List<StatisticColumn> getStatisticHeaders()
    {

        List<StatisticColumn> flatStatisticHeaders = getFlatStatisticHeaders();
        List<StatisticColumn> headers = new ArrayList<>();
        for (StatisticColumn statisticColumn : flatStatisticHeaders)
        {
            // this column have a parent, we have to find it and attach it
            if (statisticColumn.getGroup() != null)
            {
                Optional<StatisticColumn> parent = flatStatisticHeaders.stream()
                                                    .filter(column -> column.getId().equals(statisticColumn.getGroup()))
                                                    .findAny();
                if (parent.isPresent())
                {
                    parent.get().addSubColumn(statisticColumn);
                }
            }
            else
            {
                headers.add(statisticColumn);
            }
        }
        
        return headers;
    }
    
    private List<StatisticColumn> getFlatStatisticHeaders()
    {
        List<StatisticColumn> headers = new ArrayList<>();
        headers.add(new StatisticColumn("title", new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_PROJECT_STATISTICS_TOOL_COLUMN_TITLE"))
                .withType(StatisticsColumnType.STRING)
                .withWidth(200)
                .withRenderer("Ametys.plugins.workspaces.project.tool.ProjectsGridHelper.renderTitle"));
        headers.add(new StatisticColumn("creationDate", new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_PROJECT_STATISTICS_TOOL_COLUMN_CREATION"))
                .withType(StatisticsColumnType.DATE)
                .withWidth(150));
        headers.add(new StatisticColumn("managers", new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_PROJECT_STATISTICS_TOOL_COLUMN_MANAGERS"))
                .withRenderer("Ametys.grid.GridColumnHelper.renderUser")
                .withFilter(false));
        for (WorkspaceModule moduleManager : _moduleManagerEP.getModules())
        {
            headers.addAll(moduleManager.getStatisticModel());
        }
        
        StatisticColumn elements = new StatisticColumn(WorkspaceModule.GROUP_HEADER_ELEMENTS_ID, new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_PROJECT_STATISTICS_TOOL_COLUMN_ELEMENTS"))
                .withFilter(false);
        headers.add(elements);

        StatisticColumn activatedModules = new StatisticColumn(WorkspaceModule.GROUP_HEADER_ACTIVATED_ID, new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_PROJECT_STATISTICS_TOOL_COLUMN_ACTIVE_MODULES"))
                .isHidden(true)
                .withFilter(false);
        headers.add(activatedModules);
        
        StatisticColumn lastActivity = new StatisticColumn(WorkspaceModule.GROUP_HEADER_LAST_ACTIVITY_ID, new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_PROJECT_STATISTICS_TOOL_COLUMN_LAST_ACTIVITY")).isHidden(true);
        headers.add(lastActivity);

        StatisticColumn modulesSize = new StatisticColumn(WorkspaceModule.GROUP_HEADER_SIZE_ID, new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_PROJECT_STATISTICS_TOOL_COLUMN_MODULES_SIZE"))
                .withFilter(false);
        modulesSize.addSubColumn(new StatisticColumn("totalSize", new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_PROJECT_STATISTICS_TOOL_COLUMN_MODULES_SIZE_TOTAL"))
                .withRenderer("Ametys.plugins.workspaces.project.tool.ProjectsGridHelper.renderSize")
                .withType(StatisticsColumnType.LONG));
        
        headers.add(modulesSize);
        
        return headers;
    }
    
    /**
     * Check if the user is in one of the populations of project
     * @param project the project
     * @param user the user
     * @return true if the user is in one of the populations of project
     */
    public boolean isUserInProjectPopulations(Project project, UserIdentity user)
    {
        Site site = project.getSite();

        if (site == null)
        {
            throw new IllegalArgumentException("Cannot determine if user " + UserIdentity.userIdentityToString(user) + " can connect to the project " + project.getName() + " since the project has no associated site to determine the populations");
        }
        String siteName = site.getName();

        Set<String> populations = _populationContextHelper.getUserPopulationsOnContext("/sites/" + siteName, false);
        Set<String> frontPopulations = _populationContextHelper.getUserPopulationsOnContext("/sites-fo/" + siteName, false);
        
        return populations.contains(user.getPopulationId()) || frontPopulations.contains(user.getPopulationId());
    }
    
    /**
     * Thrown to indicate that the catalog site is unknown
     */
    public class UnknownCatalogSiteException extends IllegalArgumentException
    {
        /**
         * Construct a {@code UnknownCatalogSiteException} with the specified message
         * @param message the message
         */
        public UnknownCatalogSiteException(String message)
        {
            super(message);
        }
    }
    
    /**
     * Thrown to indicate that the user directory site is unknown
     */
    public class UnknownUserDirectorySiteException extends IllegalArgumentException
    {
        /**
         * Construct a {@code UnknownUserDirectorySiteException} with the specified message
         * @param message the message
         */
        public UnknownUserDirectorySiteException(String message)
        {
            super(message);
        }
    }
}
