/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.project.generators;

import java.io.IOException;
import java.util.Arrays;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.plugins.core.user.UserHelper;
import org.ametys.plugins.repository.activities.Activity;
import org.ametys.plugins.workspaces.activities.AbstractWorkspacesActivityType;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModule;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModuleExtensionPoint;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.web.renderingcontext.RenderingContextHandler;
import org.ametys.web.repository.page.Page;

/**
 * Abstract generator for mail notification
 */
public abstract class AbstractMailNotifierGenerator extends ServiceableGenerator
{
    /** The name of the root element */
    public static final String ROOT_ELEMENT = "notification";

    /** The workspace module managers EP */
    protected WorkspaceModuleExtensionPoint _moduleManagerEP;
    /** The project manager */
    protected ProjectManager _projectManager;
    /** The rendering context handler */
    protected RenderingContextHandler _renderingContextHandler;
    /** The user Helper */
    protected UserHelper _userHelper;
    /** The user Manager */
    protected UserManager _userManager;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _moduleManagerEP = (WorkspaceModuleExtensionPoint) smanager.lookup(WorkspaceModuleExtensionPoint.ROLE);
        _projectManager = (ProjectManager) smanager.lookup(ProjectManager.ROLE);
        _renderingContextHandler = (RenderingContextHandler) smanager.lookup(RenderingContextHandler.ROLE);
        _userHelper = (UserHelper) smanager.lookup(UserHelper.ROLE);
        _userManager = (UserManager) smanager.lookup(UserManager.ROLE);
    }
    
    public void generate() throws IOException, SAXException, ProcessingException
    {
        contentHandler.startDocument();

        @SuppressWarnings("unchecked")
        Map<String, Object> parentContextAttr = (Map<String, Object>) objectModel.get(ObjectModelHelper.PARENT_CONTEXT);
        Activity activity = (Activity) parentContextAttr.get(AbstractWorkspacesActivityType.ACTIVITY_CONTEXT_PARAM);
        
        String eventId = activity.getEventType();
        
        UserIdentity issuerIdentity = activity.getAuthor();
        
        Project project = _projectManager.getProject(activity.getValue(AbstractWorkspacesActivityType.PROJECT_NAME));
        
        XMLUtils.startElement(contentHandler, ROOT_ELEMENT);
        
        XMLUtils.createElement(contentHandler, "eventKey", getEventIdAsI18nKey(eventId));
        
        saxProject(project);
        
        Page modulePage = getModulePage(project);
        if (modulePage != null)
        {
            XMLUtils.createElement(contentHandler, "modulePageId", modulePage.getId());
        }
        
        _saxProjectUserIdentity(issuerIdentity, project, "issuer");
        
        activity.toSAX(contentHandler);
        
        saxAdditionalData(activity);
        
        XMLUtils.endElement(contentHandler, ROOT_ELEMENT);
        
        contentHandler.endDocument();
    }
    
    /**
     * Use this method to provide additional information not stored directly in the activity
     * @param activity the activity
     * @throws SAXException When a error occurred
     */
    protected void saxAdditionalData(Activity activity) throws SAXException
    {
        // Nothing to add by default
    }

    /**
     * Sax the user and project information relative to the user such as manager
     * @param issuerIdentity the identity of the user
     * @param project the project concerned
     * @param tagname the name of the element
     * @throws SAXException if an error occurred while saxing
     */
    protected void _saxProjectUserIdentity(UserIdentity issuerIdentity, Project project, String tagname) throws SAXException
    {
        XMLUtils.startElement(contentHandler, tagname);
        _userHelper.saxUserIdentity(issuerIdentity, contentHandler);
        if (ArrayUtils.contains(project.getManagers(), issuerIdentity))
        {
            XMLUtils.createElement(contentHandler, "manager", "true");
        }
        else
        {
            XMLUtils.createElement(contentHandler, "manager", "false");
        }
        XMLUtils.endElement(contentHandler, tagname);
    }
    
    /**
     * Sax the project basic informations
     * @param project the project to SAX
     * @throws SAXException if an error occurred while saxing
     */
    protected void saxProject(Project project) throws SAXException
    {
        AttributesImpl attr = new AttributesImpl();
        attr.addCDATAAttribute("id", project.getId());
        XMLUtils.startElement(contentHandler, "project", attr);
        
        XMLUtils.createElement(contentHandler, "name", project.getName());
        XMLUtils.createElement(contentHandler, "title", project.getTitle());
        XMLUtils.createElement(contentHandler, "description", StringUtils.defaultString(project.getDescription()));
        XMLUtils.createElement(contentHandler, "managers", getProjectManagersNames(project));
        XMLUtils.createElement(contentHandler, "url", getProjectUrl(project));
        
        XMLUtils.endElement(contentHandler, "project");
    }
    
    /**
     * Get the list of names of the project managers
     * @param project The project
     * @return The list of names, as a single string, comma separated
     */
    protected String getProjectManagersNames(Project project)
    {
        return Arrays.stream(project.getManagers())
                .map(_userManager::getUser)
                .filter(Objects::nonNull)
                .map(User::getFullName)
                .collect(Collectors.joining(", "));
    }
    
    /**
     * Get the URL of module to insert in email body
     * @return The full URL
     */
    protected abstract String _getModuleId();
    
    /**
     * Get the module's page
     * @param project The project
     * @return The page or <code>null</code> if not found
     */
    protected Page getModulePage(Project project)
    {
        String moduleId = _getModuleId();
        Set<Page> pages = null;
        
        WorkspaceModule module = _moduleManagerEP.getModule(moduleId);
        if (module != null && _projectManager.isModuleActivated(project, moduleId))
        {
            pages = _projectManager.getModulePages(project, module);
        }
        
        
        if (pages != null && pages.size() > 0)
        {
            return pages.iterator().next();
        }
        
        return null;
    }
    
    /**
     * Get the absolute url of project
     * @param project The project
     * @return the project's url
     */
    protected String getProjectUrl(Project project)
    {
        return _projectManager.getProjectUrl(project, StringUtils.EMPTY);
    }
    
    /**
     * Transform the event Id in the I18n key format
     * @param eventId the event Id
     * @return the Id as an I18n key
     */
    protected String getEventIdAsI18nKey(String eventId)
    {
        return eventId.toUpperCase().replace(".", "_");
    }
}
