/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.project.notification;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;
import org.apache.excalibur.source.SourceUtil;

import org.ametys.core.group.GroupManager;
import org.ametys.core.observation.AsyncObserver;
import org.ametys.core.observation.Event;
import org.ametys.core.user.UserManager;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.language.UserLanguagesManager;
import org.ametys.core.util.mail.SendMailHelper;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.workspaces.ObservationConstants;
import org.ametys.plugins.workspaces.WorkspacesHelper;
import org.ametys.plugins.workspaces.members.ProjectMemberManager;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.runtime.plugin.component.PluginAware;
import org.ametys.web.WebConstants;
import org.ametys.web.population.PopulationContextHelper;
import org.ametys.web.renderingcontext.RenderingContext;
import org.ametys.web.renderingcontext.RenderingContextHandler;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;

import jakarta.mail.MessagingException;

/**
 * Abstract observer for sending mail to members
 */
public abstract class AbstractMemberMailNotifierObserver  extends AbstractLogEnabled implements AsyncObserver, PluginAware, Serviceable, Contextualizable
{
    /** The avalon context */
    protected Context _context;
    /** The name of current plugin */
    protected String _pluginName;
    /** The Ametys Object resolver */
    protected AmetysObjectResolver _resolver;
    /** The i18n utils */
    protected I18nUtils _i18nUtils;
    /** The project member manager */
    protected ProjectMemberManager _projectMemberManager;
    /** Project manager */
    protected ProjectManager _projectManager;
    /** Site manager */
    protected SiteManager _siteManager;
    /** Population context helper */
    protected PopulationContextHelper _populationContextHelper;
    /** Source Resolver */
    protected SourceResolver _srcResolver;
    /** User manager */
    protected UserManager _userManager;
    /** Group manager */
    protected GroupManager _groupManager;
    /** The rendering context */
    protected RenderingContextHandler _renderingContextHandler;
    /** The user languages manager */
    protected UserLanguagesManager _userLanguagesManager;
    /** The workspaces helper */
    protected WorkspacesHelper _workspacesHelper;
    
    
    @Override
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _pluginName = pluginName;
    }
    
    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
        _projectMemberManager = (ProjectMemberManager) manager.lookup(ProjectMemberManager.ROLE);
        _projectManager = (ProjectManager) manager.lookup(ProjectManager.ROLE);
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
        _populationContextHelper = (PopulationContextHelper) manager.lookup(org.ametys.core.user.population.PopulationContextHelper.ROLE);
        _srcResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
        _userManager = (UserManager) manager.lookup(UserManager.ROLE);
        _groupManager = (GroupManager) manager.lookup(GroupManager.ROLE);
        _renderingContextHandler = (RenderingContextHandler) manager.lookup(RenderingContextHandler.ROLE);
        _userLanguagesManager = (UserLanguagesManager) manager.lookup(UserLanguagesManager.ROLE);
        _workspacesHelper = (WorkspacesHelper) manager.lookup(WorkspacesHelper.ROLE);
    }
    
    @Override
    public int getPriority()
    {
        return MIN_PRIORITY;
    }
    
    @Override
    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        Map<String, Object> args = event.getArguments();
        
        String projectId = (String) args.get(ObservationConstants.ARGS_PROJECT_ID);
        Project project = _resolver.resolveById(projectId);
        Site site = project.getSite();
        String lang = _workspacesHelper.getLang(project, _userLanguagesManager.getDefaultLanguage());
        // Compute subject and body
        I18nizableText i18nSubject = getI18nSubject(event, project);
        
        // Send mail to removed members
        Map<String, List<String>> recipientsByLanguage = getUserToNotifyByLanguage(event, project);
        
        for (String language : recipientsByLanguage.keySet())
        {
            String userLanguage = StringUtils.defaultIfBlank(language, lang);
            
            String subject = _i18nUtils.translate(i18nSubject, userLanguage);
            
            String mailBody = null;
            Source source = null;
            RenderingContext currentContext = _renderingContextHandler.getRenderingContext();
            try
            {
                // Force rendering context.FRONT to resolve URI
                _renderingContextHandler.setRenderingContext(RenderingContext.FRONT);
                Request request = ContextHelper.getRequest(_context);
                request.setAttribute("forceAbsoluteUrl", true);
                request.setAttribute("lang", userLanguage);
                request.setAttribute(WebConstants.REQUEST_ATTR_SITE, site);
                request.setAttribute(WebConstants.REQUEST_ATTR_SITE_NAME, site.getName());
                request.setAttribute(WebConstants.REQUEST_ATTR_SKIN_ID, site.getSkinId());
                source = _srcResolver.resolveURI(getMailBodyURI(event, project), null, Map.of("event", event, "project", project));
                try (InputStream is = source.getInputStream())
                {
                    ByteArrayOutputStream bos = new ByteArrayOutputStream();
                    SourceUtil.copy(is, bos);
                    
                    mailBody = bos.toString("UTF-8");
                }
            }
            catch (IOException e)
            {
                throw new RuntimeException("Failed to create mail body", e);
            }
            finally
            {
                _renderingContextHandler.setRenderingContext(currentContext);
                
                if (source != null)
                {
                    _srcResolver.release(source);
                }
            }
            
            try
            {
                SendMailHelper.newMail()
                    .withSubject(subject)
                    .withHTMLBody(mailBody)
                    .withRecipients(recipientsByLanguage.get(language))
                    .withAsync(true)
                    .withInlineCSS(false)
                    .sendMail();
            }
            catch (MessagingException | IOException e)
            {
                getLogger().warn("Could not send a notification e-mail to " + recipientsByLanguage + " following his removal from the project " + project.getTitle(), e);
            }
        }
    }
    
    /**
     * Get recipients' emails sorted by language
     * @param event the event
     * @param project the project
     * @return the recipients' emails sorted by language
     */
    protected abstract Map<String, List<String>> getUserToNotifyByLanguage(Event event, Project project);
    
    /**
     * Returns the URI for HTML mail body
     * @param event the event
     * @param project the project
     * @return The URI for HTML mail body
     */
    protected String getMailBodyURI(Event event, Project project)
    {
        return "cocoon://_plugins/workspaces/notification-mail-member";
    }
    
    /**
     * Get the {@link I18nizableText} for mail subject
     * @param event the event
     * @param project the project
     * @return the {@link I18nizableText} for subject
     */
    protected abstract I18nizableText getI18nSubject(Event event, Project project);
}
