/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.signup;

import java.util.List;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.core.user.User;
import org.ametys.plugins.workspaces.members.ProjectInvitationHelper;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.repository.page.Page;
import org.ametys.web.usermanagement.UserManagementException;

import com.google.common.collect.Multimap;

/**
 * This component overrides {@link org.ametys.plugins.userdirectory.signup.UserSignupManager} to handle invitations by mail for workspaces
 */
public class UserSignupManager extends org.ametys.plugins.userdirectory.signup.UserSignupManager
{
    private ProjectManager _projectManager;
    private ProjectInvitationHelper _invitationHelper;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _projectManager = (ProjectManager) serviceManager.lookup(ProjectManager.ROLE);
        _invitationHelper = (ProjectInvitationHelper) serviceManager.lookup(ProjectInvitationHelper.ROLE);
    }
    
    @Override
    public void temporarySignup(String siteName, String language, String email, String population, String userDirectoryId) throws UserManagementException
    {
        String signupSiteName = _getSignupSiteName(siteName);
        boolean sendMail = signupSiteName.equals(siteName);
        
        super.temporarySignup(signupSiteName, language, email, population, userDirectoryId, sendMail);
    }
    
    @Override
    public void additionalSignupOperations(User createdUser, Multimap<String, I18nizableText> errors) throws UserManagementException
    {
        super.additionalSignupOperations(createdUser, errors);
        
        _invitationHelper.createMemberFromInvitations(createdUser);
    }
    
    @Override
    public boolean isPublicSignupAllowed(String siteName)
    {
        return super.isPublicSignupAllowed(_getSignupSiteName(siteName));
    }
    
    @Override
    public Page getSignupPage(String siteName, String language)
    {
        return super.getSignupPage(_getSignupSiteName(siteName), language);
    }
    
    @Override
    public List<Page> getSignupPages(String siteName, String language)
    {
        return super.getSignupPages(_getSignupSiteName(siteName), language);
    }
    
    @Override
    public Page getPwdChangePage(String siteName, String language)
    {
        return super.getPwdChangePage(_getSignupSiteName(siteName), language);
    }
    
    @Override
    public List<Page> getPwdChangePages(String siteName, String language)
    {
        return super.getPwdChangePages(_getSignupSiteName(siteName), language);
    }
    
    private String _getSignupSiteName(String siteName)
    {
        String signupSite = siteName;
        
        List<String> projectNames = _projectManager.getProjectsForSite(siteName);
        if (!projectNames.isEmpty())
        {
            signupSite = _projectManager.getCatalogSiteName();
        }
        
        return signupSite;
    }
}
