<?xml version="1.0" encoding="UTF-8"?>
<!--
   Copyright 2023 Anyware Services

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
   -->
<xsl:stylesheet version="1.0"
                xmlns:xsl="http://www.w3.org/1999/XSL/Transform" 
                xmlns:i18n="http://apache.org/cocoon/i18n/2.1"
                xmlns:exsl="http://exslt.org/common"
                xmlns:math="java.lang.Math"
                xmlns:ametys="org.ametys.web.transformation.xslt.AmetysXSLTHelper"
                xmlns:resolver="org.ametys.cms.transformation.xslt.ResolveURIComponent"
                xmlns:escaper="org.apache.commons.lang.StringEscapeUtils"
                extension-element-prefixes="math ametys resolver escaper">
    
    <xsl:import href="plugin:user-directory://pages/services/user-signup/signup.xsl"/>
    
    <xsl:variable name="stepCssClassName">
        <xsl:choose>
            <xsl:when test="$step = ''">step-temp-signup</xsl:when>
            <xsl:otherwise>step-<xsl:value-of select="$step"/></xsl:otherwise>
        </xsl:choose>
    </xsl:variable>
    <xsl:variable name="common-service-css-class-name">user-account user-signup <xsl:value-of select="$stepCssClassName"/></xsl:variable>
    
    <xsl:variable name="withTosContent" select="normalize-space($tos-mode) = 'CONTENT' and /user-signup/tos/content"/>
    
    <xsl:template name="common-service-head-css">
        <link rel="stylesheet" href="{ametys:skinURL('zones/signup/scss/main.css')}" />
    </xsl:template>
    
    <xsl:template name="common-service-head-js">
        <xsl:call-template name="strong-password-head-js"/>

        <script src="{ametys:skinURL(concat('js/services/signup/step-form.', ametys:lang(), '.js'))}"></script>
        
        <xsl:call-template name="common-dropdown-js"/>
                
        <script type="text/javascript">
             $j(function() {
                var $input = $j(".user-signup form .field input:not([type=search]):not([type=checkbox])");
                $input.each(function() {
                    var $this = $j(this),
                        label = $this.closest('.field').find('.label label').text();
                        
                    $this.attr('placeholder', label);
                    $this.attr('title', label);
                })
             });
        </script>
    </xsl:template>
    
    <xsl:template name="common-dropdown-js">
        <script type="text/javascript">
            function toggleDropdown($cbox) {
                if (isOpenDropdown($cbox)) {
					closeDropdown($cbox);
					$cbox.trigger('focus');
				} else {
					openDropdown($cbox);
				}
            }
            
            function isOpenDropdown($cbox) {
                return $cbox.attr('aria-expanded') == 'true';
            }
            
            function openDropdown($cbox) {
                // close all open combobox
                closeDropdown($j('.user-account form [role=combobox]'));
                
                $cbox.attr('aria-expanded', true);
                $cbox.siblings(".select-dropdown").show()
                $cbox.next().addClass("active");
            }
            
            function closeDropdown($cbox) {
                $cbox.attr('aria-expanded', false);
                $cbox.siblings(".select-dropdown").hide();
                $cbox.next().removeClass("active");
                clearActiveDescendant($cbox.siblings(".select-dropdown").find('[role="listbox"]'));
            }
        
            // On select option in criteria dropdown
            function onSelectOptions($option, multiple) {
                const $dropdownList = $option.closest(".select-dropdown");
                if ($option.attr('aria-selected') == 'false') {
                    $option.attr('aria-selected', 'true');
                } else {
                    $option.attr('aria-selected', 'false');
                }
                const optValue = $option.attr("data-value");
                const $cbox = $dropdownList.parent().find("[role=combobox]");
                const $textbox = $cbox.find("[role=textbox]");
                const $input = $dropdownList.parent().find("select");
            
                if (multiple)
                {
                    let $selectedOpts = $j('li[aria-selected="true"]', $dropdownList);
                    if ($selectedOpts.length == 0) {
                        // no option selected => check 'all' option
                        $textbox.text($cbox.attr('aria-label'));
                        $input.val("");
                    } else {
                         var count = $dropdownList.find("input:checked").length;
            
                        var vals = [];
                        var labels = [];
                        $selectedOpts.each(function (index) {
                            vals.push($j(this).attr("data-value"));
                            labels.push($j(this).text());
                        });
                        $textbox.text(labels.join(", "));
                        $input.val(vals);
                    }
                }
                else
                {
                    if ($option.attr('aria-selected') == 'true') {
                        // Unselect all other options
                        $j('li[aria-selected="true"]', $dropdownList).not($option).attr('aria-selected', 'false');
                        if (optValue)
                        {
                            $textbox.text($option.text());
                            $input.val(optValue);
                        }
                        else
                        {
                            $textbox.text($cbox.attr('aria-label'));
                            $input.val("");
                        }
                    } else {
                        // No option selected
                        $textbox.text($cbox.attr('aria-label'));
                        $input.val("");
                    }
                }
            }
            
            function getActiveDescendant($lbox) {
                if ($lbox.attr('aria-activedescendant')) {
                    return $j('#' + $lbox.attr('aria-activedescendant'));
                }
            }
            function setActiveDescendant($option) {
                let $lbox = $option.parent();
                if (getActiveDescendant($lbox)) {
                    getActiveDescendant($lbox).removeClass('activedescendant');
                } 
                $lbox.attr('aria-activedescendant', $option.attr('id'));
                getActiveDescendant($lbox).addClass('activedescendant');
            }
            function clearActiveDescendant($lbox) {
                if (getActiveDescendant($lbox)) {
                    getActiveDescendant($lbox).removeClass('activedescendant');
                }       
                $lbox.attr('aria-activedescendant', '');
            }
            
            function getDefaultOption($lbox) {
                if ($j('li[aria-selected="true"]', $lbox).length > 0) {
                    return $j('li[aria-selected="true"]', $lbox).eq(0);
                } else {
                    return $j('li:first-child', $lbox);
                }
            }
            function getNextOption($lbox) {
                if (getActiveDescendant($lbox)) {
                    var ad_index = getActiveDescendant($lbox).index();
                    if (ad_index == $j('li', $lbox).length - 1) {
                        return $j('li:first-child', $lbox);
                    } else {
                        return $j('li', $lbox).eq(ad_index + 1);
                    }
                } else {
                    return getDefaultOption($lbox);
                }
            }
            function getPreviousOption($lbox) {
                if (getActiveDescendant($lbox)) {
                    var ad_index = getActiveDescendant($lbox).index();
                    if (ad_index === 0) {
                        return $j('li:last-child', $lbox);
                    } else {
                        return $j('li', $lbox).eq(ad_index - 1);
                    }
                } else {
                    return getDefaultOption($lbox);
                }
            }
        
            // Filter on dropdown items
            function filterDropdown() {
                var value = $j(this).val().toLowerCase();
                $j(this)
                    .parent()
                    .parent()
                    .find(".select-dropdown-item")
                    .filter(function () {
                        $j(this).toggle($j(this).text().toLowerCase().indexOf(value) > -1);
                    });
            }
        
            $j(document).on('click', function (e) {
                if ( $j(e.target).closest(".select-dropdown").length === 0) {
                    // close all opened dropdown
                    closeDropdown($j('.user-account form [role=combobox]'));
                }
            });
            
            $j(function() {
                $j(".select-dropdown-search input").on("keyup", filterDropdown);
            });
        </script>
    </xsl:template>
    
    <xsl:template name="common-service-body-nonempty-content-title-rendering">
        <xsl:param name="title"/>
        
        <xsl:call-template name="step-form-progress"/>
        
        <h1 class="service-title">
            <xsl:choose>
                <xsl:when test="$step = 'signup' and $status = 'success'"><i18n:text i18n:key="SKIN_SERVICE_SIGNUP_SUCCESS_TITLE" i18n:catalogue="skin.{$skin}"/></xsl:when>
                <xsl:otherwise><xsl:copy-of select="$title"/></xsl:otherwise>
            </xsl:choose>
        </h1>
   </xsl:template>
   
   
   <xsl:template name="step-form-progress">
        <xsl:if test="$step != 'signup'">
            <ul class="signup-steps" id="progressStepForm" data-step="{$step}">
                <li class="step step-1" data-to-step="#tempSignUp">
                    <xsl:if test="$step = '' or $step = 'temp-signup'">
                        <xsl:attribute name="class">step step-1 active</xsl:attribute>
                        <xsl:attribute name="aria-current">step</xsl:attribute>
                    </xsl:if>
                    <div class="step-marker">
                        <div class="step-marker-round"></div>
                    </div>
                    <div class="step-number"><i18n:text i18n:key="SKIN_SERVICE_SIGNUP_DISPLAY_STEP1" i18n:catalogue="skin.{$skin}"/></div>
                </li>
            
                <li class="step step-2" data-to-step="#signUp">
                    <xsl:if test="$step = 'password'">
                        <xsl:attribute name="class">step step-2 active</xsl:attribute>
                        <xsl:attribute name="aria-current">step</xsl:attribute>
                    </xsl:if>
                    <div class="step-marker">
                        <div class="step-marker-round"></div>
                    </div>
                    <div class="step-number"><i18n:text i18n:key="SKIN_SERVICE_SIGNUP_DISPLAY_STEP2" i18n:catalogue="skin.{$skin}"/></div>
                </li>
            
                <xsl:if test="normalize-space($tos-mode) = 'CONTENT' and /user-signup/tos/content">
                    <li class="step step-3" data-to-step="#stepFormTermsOfUse">
                        <div class="step-marker">
                            <div class="step-marker-round"></div>
                        </div>
                        <div class="step-number"><i18n:text i18n:key="SKIN_SERVICE_SIGNUP_DISPLAY_STEP3" i18n:catalogue="skin.{$skin}"/></div>
                    </li>
                </xsl:if>
            </ul>
        </xsl:if>
    </xsl:template>
    
    <!-- Override to reorganize fields -->
    <xsl:template name="continue-signup-fields">
        <div id="signUp">
            <div class="signup-flex-profil">
                <div class="signup-flex-profil-fields">
                    <xsl:call-template name="firstname-field">
                        <xsl:with-param name="value" select="/user-signup/@firstname"/>
                    </xsl:call-template>
                    <xsl:call-template name="lastname-field">
                        <xsl:with-param name="value" select="/user-signup/@lastname"/>
                    </xsl:call-template>
               </div>
           </div>
           
           <div class="signup-flex-fields">
                <xsl:call-template name="password-form-fields">
                    <xsl:with-param name="uniqueId" select="$uniqueId"/>
                    <xsl:with-param name="email" select="/user-signup/@email"/>
                    <xsl:with-param name="token" select="/user-signup/@token"/>
                    <xsl:with-param name="errors" select="/user-signup/errors"/>
                </xsl:call-template>
           </div>
            
           <!-- User fields -->
           <xsl:call-template name="user-fields"/>  
           
           <xsl:if test="$withTosContent">
                <div class="button">
                    <button type="button" class="next" data-next-step="#stepFormTermsOfUse" aria-label="skin.{$skin}:SKIN_SERVICE_SIGNUP_GO_TO_NEXT_STEP" i18n:attr="aria-label">
                        <i18n:text i18n:key="SKIN_SERVICE_SIGNUP_NEXT" i18n:catalogue="skin.{$skin}"/>
                    </button>
                </div>
           </xsl:if>
             
       </div>
       
       <div id="stepFormTermsOfUse">
           <xsl:if test="not($withTosContent)"><xsl:attribute name="class">active</xsl:attribute></xsl:if>
           <xsl:if test="$withTosContent">
                <xsl:attribute name="style">display:none;</xsl:attribute>
                <xsl:call-template name="tos-title"/>
                
                <div class="cgu-part ametys-cms-content">
                    <xsl:call-template name="tos-content">
                        <xsl:with-param name="tos-content-id" select="$tos-content-id"/>
                    </xsl:call-template>
                </div>
            </xsl:if>
            
            <xsl:if test="normalize-space($tos-mode) != 'NONE'">
                <xsl:call-template name="tos-field">
                    <xsl:with-param name="tos-mode" select="$tos-mode"/>
                    <xsl:with-param name="tos-page-id" select="$tos-page-id"/>
                    <xsl:with-param name="tos-link-label" select="$tos-link-label"/>
                </xsl:call-template>
            </xsl:if>
        
            <xsl:if test="$withTosContent">
                <script type="text/javascript">
                    // check terms of use
                    $j('.field.tos input[type=checkbox]').prop('checked', 'checked');
                </script>
                <div class="button precede-step">
                    <button type="button" class="precede" data-precede-step="#signUp" aria-label="skin.{$skin}:SKIN_SERVICE_SIGNUP_GO_TO_PREVIOUS_STEP" i18n:attr="aria-label">
                        <i18n:text i18n:key="SKIN_SERVICE_SIGNUP_PRECEDE" i18n:catalogue="skin.{$skin}"/>
                    </button>
                </div>
            </xsl:if>
        </div>
    </xsl:template>
    
    <xsl:template name="tos-title">
        <h1 class="service-title">
            <i18n:text i18n:key="SKIN_SERVICE_SIGNUP_CGU_TITLE" i18n:catalogue="skin.{$skin}"/>
        </h1>
    </xsl:template>
    
    <!-- Override for handle user profile image profile -->
    <xsl:template match="metadata[@type='composite' and @name='illustration']" mode="contenttype-form-field">
        <xsl:param name="fieldId" select="translate(@path, '/', '-')"/>
        <xsl:param name="fieldName" select="translate(@path, '/', '.')"/>
        <xsl:param name="values"/>
        <xsl:param name="contentValues"/>
        <xsl:param name="value" select="$values/*[local-name() = $fieldName]"/>
        <xsl:param name="errors" />
        <xsl:param name="uniqueId" />
        
        <div class="profile-picture">
            <button type="button" class="profil-picture-btn profil-picture-btn-add" onclick="selectProfilePicture()" aria-label="skin.{$skin}:SKIN_SIGNUP_SELECT_IMAGE" i18n:attr="aria-label">
                <span aria-hidden="true" class="fas fa-camera-retro"></span>
            </button>
            <button type="button"
                class="profil-picture-btn profil-picture-btn-remove"
                onclick="removePicture()"
                aria-label="skin.{$skin}:SKIN_SIGNUP_DELETE_IMAGE"
                i18n:attr="aria-label"
                style="display: none">
                <span aria-hidden="true" class="fas fa-times"></span>
            </button>
            <input
                name="illustration.image"
                type="file"
                id="profilePictureInput"
                accept="image/*"
                onchange="showPreview(event)"
             />
        </div>
        
        <script>
            $j(".profile-picture").appendTo(".signup-flex-profil");
            
            function selectProfilePicture() {
                document.getElementById("profilePictureInput").trigger('click');
            }
        
            function showPreview(event) {
           	    const input = event.target;
                if (input.files &amp;&amp; input.files[0]) 
                {
                    const reader = new FileReader();
                    reader.onload = function (e) {
                        const preview = document.querySelector(".profile-picture");
                        preview.style.backgroundImage = `url(${e.target.result})`;
                        preview.style.backgroundSize = "cover";
                        $j(".profil-picture-btn-add").hide();
                        $j(".profil-picture-btn-remove").show();
                    };
                    reader.readAsDataURL(input.files[0]);
                }
            }
    
            function removePicture() {
                const preview = document.querySelector(".profile-picture");
                preview.removeAttribute("style");
                document.getElementById("profilePictureInput").value = "";
                $j(".profil-picture-btn-add").show();
                $j(".profil-picture-btn-remove").hide();
            }
        </script>
    </xsl:template>
    
    <!-- Override of select field (enumeration) -->
    <xsl:template match="metadata[@type='string' and enumeration]" mode="contenttype-form-field">
        <xsl:param name="fieldId" select="translate(@path, '/', '-')"/>
        <xsl:param name="fieldName" select="translate(@path, '/', '.')"/>
        <xsl:param name="values"/>
        <xsl:param name="errors" />
        <xsl:param name="uniqueId" />
        <xsl:param name="value" select="$values/*[local-name() = $fieldName]"/>
        
        <xsl:variable name="error"><xsl:value-of select="$errors[@name = $fieldName]/error"/></xsl:variable>
        
        <div id="field-{$fieldId}-{$uniqueId}" class="field field-select field-{$fieldId}">
            <xsl:if test="$error">
                <xsl:call-template name="contenttype-form-field-error" >
                    <xsl:with-param name="error" select="$error" />
                </xsl:call-template>
            </xsl:if>
            
            <xsl:apply-templates select="." mode="contenttype-form-field-label"/>
             
            <div id="input-{$fieldId}-{$uniqueId}" class="input">
                <xsl:if test="$errors[@name = $fieldName]/error">
                    <xsl:attribute name="class">select invalid</xsl:attribute>
                </xsl:if>
                
                <select style="display: none" name="{$fieldName}">
                    <xsl:if test="@multiple = 'true'"><xsl:attribute name="multiple">multiple</xsl:attribute></xsl:if>
                    <xsl:if test="value != ''"><xsl:attribute name="value"><xsl:value-of select="$value"/></xsl:attribute></xsl:if>
                    <option value=""><i18n:text i18n:key="PLUGINS_WEB_FRONTOFICE_FORM_SELECT_NONE" i18n:catalogue="plugin.web"/></option>
                    <xsl:for-each select="enumeration/entry">
                        <option value="{value}"><xsl:copy-of select="label/node()"/></option>
                    </xsl:for-each>
                </select>
                
                <div aria-controls="listbox-{$fieldId}-{$uniqueId}"
                       aria-expanded="false"
                       aria-haspopup="listbox"
                       aria-label="{label}"
                       id="combo-{$fieldId}-{$uniqueId}"
                       role="combobox"
                       tabindex="0">
                       
                   <div id="textbox-{$fieldId}-{$uniqueId}" role="textbox" aria-readonly="true" aria-multiline="false" aria-autocomplete="none" aria-controls="listbox-{$fieldId}-{$uniqueId}"><xsl:copy-of select="label/node()"/></div>
              </div>
                  
              <button type="button" class="toggle" aria-hidden="true" i18n:attr="aria-label" tabindex="-1">
                <xsl:attribute name="aria-label">
                    <xsl:choose>
                        <xsl:when test="@multiple != 'true'">skin.<xsl:value-of select="$skin"/>:SKIN_SERVICE_SIGNUP_OPEN_DROPDOWN</xsl:when>
                        <xsl:otherwise>skin.<xsl:value-of select="$skin"/>:SKIN_SERVICE_SIGNUP_OPEN_DROPDOWN_MULTIPLE</xsl:otherwise>
                    </xsl:choose>
                </xsl:attribute>
              </button>
                  <div class="select-dropdown">
                    <xsl:if test="@multiple != 'true'"><xsl:attribute name="class">select-dropdown monoselect</xsl:attribute></xsl:if>
                    
                    <xsl:if test="count(enumeration/entry) > 15">
                        <div class="select-dropdown-search">
                            <input type="search" title="skin.{$skin}:SKIN_SERVICE_SIGNUP_FILTER" placeholder="skin.{$skin}:SKIN_SERVICE_SIGNUP_FILTER" i18n:attr="title placeholder"/>
                        </div>
                    </xsl:if>
                        
                    <ul class="select-dropdown-list" id="listbox-{$fieldId}-{$uniqueId}" role="listbox" tabindex="-1">
                        <xsl:choose>
                            <xsl:when test="@multiple != 'true'">
                                <li class="select-dropdown-item" role="option" aria-selected="false" data-value="" id="{$fieldName}-opt-none">
                                    <span class="check" aria-hidden="true"></span>
                                    <i18n:text i18n:key="PLUGINS_WEB_FRONTOFICE_FORM_SELECT_NONE" i18n:catalogue="plugin.web"/>
                                </li>
                            </xsl:when>
                            <xsl:otherwise>
                                <xsl:attribute name="aria-multiselectable">true</xsl:attribute>
                            </xsl:otherwise>
                        </xsl:choose>
                        
                        <xsl:for-each select="enumeration/entry">
                            <xsl:sort  select="label/node()"/>
                            <li class="select-dropdown-item" role="option" aria-selected="false" data-value="{value}" id="{$fieldName}-opt-{position()}">
                                <span class="check" aria-hidden="true"></span>
                                <xsl:copy-of select="label/node()"/>
                            </li>
                        </xsl:for-each>
                    </ul>
                  </div>
            </div>
       </div>
       
       <xsl:call-template name="select-js">
            <xsl:with-param name="fieldId" select="$fieldId"/>
            <xsl:with-param name="fieldName" select="$fieldName"/>
            <xsl:with-param name="uniqueId" select="$uniqueId"/>
            <xsl:with-param name="multiple" select="@multiple"/>
        </xsl:call-template>
    </xsl:template>
    
    <!-- Override of select fields (content) -->
    <xsl:template match="metadata[@type='content']" mode="contenttype-form-field">
        <xsl:param name="fieldId" select="translate(@path, '/', '-')"/>
        <xsl:param name="fieldName" select="translate(@path, '/', '.')"/>
        <xsl:param name="values"/>
        <xsl:param name="contentValues"/>
        <xsl:param name="value" select="$values/*[@path = translate(@path, '/', '.')]"/>
        <xsl:param name="errors" />
        <xsl:param name="uniqueId" />
        
        <div id="field-{$fieldId}-{$uniqueId}" class="field field-select field-{$fieldId}">
             <xsl:apply-templates select="." mode="contenttype-form-field-label"/>
             
             <div id="input-{$fieldId}-{$uniqueId}" class="input">
                 <xsl:if test="$errors[@name = $fieldId]/error">
                    <xsl:attribute name="class">select invalid</xsl:attribute>
                 </xsl:if>
                 
                 <select style="display:none" name="{$fieldName}" >
                     <xsl:if test="@multiple = 'true'"><xsl:attribute name="multiple">multiple</xsl:attribute></xsl:if>
                     <xsl:if test="$value != ''"><xsl:attribute name="value"><xsl:value-of select="$value"/></xsl:attribute></xsl:if>
                     <option value=""><i18n:text i18n:key="PLUGINS_WEB_FRONTOFICE_FORM_SELECT_NONE" i18n:catalogue="plugin.web"/></option>
                     
                     <xsl:for-each select="$contentValues[@name = $fieldName]/enumeration/item">
                        <xsl:sort  select="label/node()"/>
                        <option value="{@value}">
                            <xsl:copy-of select="label/node()"/>
                        </option>
                     </xsl:for-each>
                 </select>
                 
                 <div aria-controls="listbox-{$fieldId}-{$uniqueId}"
                       aria-expanded="false"
                       aria-haspopup="listbox"
                       aria-label="{label}"
                       id="combo-{$fieldId}-{$uniqueId}"
                       role="combobox"
                       tabindex="0">
                       
                       <div id="textbox-{$fieldId}-{$uniqueId}" role="textbox" aria-readonly="true" aria-multiline="false" aria-autocomplete="none" aria-controls="listbox-{$fieldId}-{$uniqueId}"><xsl:copy-of select="label/node()"/></div>
                  </div>
                  
                  <button typ="button" class="toggle" aria-hidden="true" i18n:attr="aria-label" tabindex="-1">
                    <xsl:attribute name="aria-label">
                        <xsl:choose>
                            <xsl:when test="@multiple != 'true'">skin.<xsl:value-of select="$skin"/>:SKIN_SERVICE_SIGNUP_OPEN_DROPDOWN</xsl:when>
                            <xsl:otherwise>skin.<xsl:value-of select="$skin"/>:SKIN_SERVICE_SIGNUP_OPEN_DROPDOWN_MULTIPLE</xsl:otherwise>
                        </xsl:choose>
                    </xsl:attribute>
                  </button>
                  
                  <div class="select-dropdown">
                    <xsl:if test="@multiple != 'true'"><xsl:attribute name="class">select-dropdown monoselect</xsl:attribute></xsl:if>
                    
                    <xsl:if test="count($contentValues[@name = $fieldName]/enumeration/item) > 15">
                        <div class="select-dropdown-search">
                            <input type="search" title="skin.{$skin}:SKIN_SERVICE_SIGNUP_FILTER" placeholder="skin.{$skin}:SKIN_SERVICE_SIGNUP_FILTER" i18n:attr="placeholder title"/>
                        </div>
                    </xsl:if>
                    
                    <ul class="select-dropdown-list" id="listbox-{$fieldId}-{$uniqueId}" role="listbox" tabindex="-1">
                        <xsl:choose>
                            <xsl:when test="@multiple != 'true'">
                                <li class="select-dropdown-item" role="option" aria-selected="false" data-value="" id="{$fieldName}-opt-none">
                                    <span class="check" aria-hidden="true"></span>
                                    <i18n:text i18n:key="PLUGINS_WEB_FRONTOFICE_FORM_SELECT_NONE" i18n:catalogue="plugin.web"/>
                                </li>
                            </xsl:when>
                            <xsl:otherwise>
                                <xsl:attribute name="aria-multiselectable">true</xsl:attribute>
                            </xsl:otherwise>
                        </xsl:choose>
                            
                        <xsl:for-each select="$contentValues[@name = $fieldName]/enumeration/item">
                            <xsl:sort  select="label/node()"/>
                            <li class="select-dropdown-item" role="option" aria-selected="false" data-value="{@value}" id="{$fieldName}-opt-{position()}">
                                <span class="check" aria-hidden="true"></span>
                                <xsl:copy-of select="label/node()"/>
                            </li>
                        </xsl:for-each>
                    </ul>
                  </div>
             </div>
        </div>
        
        <xsl:call-template name="select-js">
            <xsl:with-param name="fieldId" select="$fieldId"/>
            <xsl:with-param name="fieldName" select="$fieldName"/>
            <xsl:with-param name="uniqueId" select="$uniqueId"/>
            <xsl:with-param name="multiple" select="@multiple"/>
        </xsl:call-template>
    </xsl:template>
    
    <xsl:template name="select-js">
        <xsl:param name="fieldId"/>
        <xsl:param name="fieldName"/>
        <xsl:param name="uniqueId"/>
        <xsl:param name="multiple"/>
            
        <script>
              {
                const $cbox = $j("#input-<xsl:value-of select="$fieldId"/>-<xsl:value-of select="$uniqueId"/> [role=combobox]");
                const $lbox = $j("#input-<xsl:value-of select="$fieldId"/>-<xsl:value-of select="$uniqueId"/> [role=listbox]");
                const $btn = $j("#input-<xsl:value-of select="$fieldId"/>-<xsl:value-of select="$uniqueId"/> .toggle");
                const $options = $j('li', $lbox);
                
                $cbox.on('click', function(e) {
                    toggleDropdown($cbox);
                    e.stopPropagation();
                });
                
                var altPressed = false;
              
                $cbox.on('keydown', function (e) {
                    switch (e.which) {
                        case 18:
                            altPressed = true;
                            break;
                        case 9:
                            if (isOpenDropdown($cbox)) {
                                closeDropdown($cbox);
                            }
                            break;
                        case 13:
                        case 32:
                            toggleDropdown($cbox);
                            if (isOpenDropdown($cbox)) {
                                $lbox.trigger('focus');
                            }
                            e.preventDefault();
                            break;
                        case 38:
                            if (altPressed == true) {
                                closeDropdown($cbox);
                            } else if (!isOpenDropdown($cbox)) {
                                openDropdown($cbox);       
                                $lbox.trigger('focus');
                                setActiveDescendant(getDefaultOption($lbox));
                                e.preventDefault();                 
                            }               
                            break;
                        case 40:
                            if (!isOpenDropdown($cbox)) {
                                openDropdown($cbox);                           
                            }               
                            $lbox.trigger('focus');
                            setActiveDescendant(getDefaultOption($lbox));
                            e.preventDefault();
                            break;
                        default:
                            return true;
                    }
                    
                });
                
                $lbox.on('keydown', function (e) {
                    switch (e.which) {
                        case 18:
                            altPressed = true;
                            break;
                        case 9:
                            if (isOpenDropdown($cbox)) {
                                closeDropdown($cbox);
                            }
                            break;
                       case 27:
                            if (isOpenDropdown($cbox)) {
                                closeDropdown($cbox);
                                $cbox.trigger('focus');
                            }
                            break;
                        case 32:
                            onSelectOptions(getActiveDescendant($j(this)), <xsl:value-of select="$multiple"/>);
                            e.preventDefault();
                            break;
                        case 38: /* arrow up */
                            setActiveDescendant(getPreviousOption($j(this)));
                            e.preventDefault(); 
                            break;   
                        case 40: /* arrow down */
                            setActiveDescendant(getNextOption($j(this)));
                            e.preventDefault();     
                            break;
                        default:
                            return true;
                    }
                });
              
                $btn.on('click', function(e) {
                    toggleDropdown($cbox);
                    $cbox.trigger('focus');
                    e.stopPropagation();
                })
                
                $options.on('click', function(e) {
                    onSelectOptions($j(this), <xsl:value-of select="$multiple"/>);
                    setActiveDescendant($j(this));
                    e.stopProgation();
                })
                
                $j('html').on('keydown', function (e) {
                    if (e.which == 27 &amp;&amp; isOpenDropdown($cbox)) {
                        closeDropdown($cbox);
                    } 
                });
              }
            </script>
    </xsl:template>
    
    <xsl:template match="*" mode="contenttype-form-field-js">
        <!-- Neutralize default JS validation of input (handled by step-form.js) -->
    </xsl:template>
    
    <!-- Override checkbox rendering (parent div flex)-->
    <xsl:template match="metadata[@type='boolean']" mode="contenttype-form-field">
        <xsl:param name="fieldId" select="translate(@path, '/', '-')"/>
        <xsl:param name="fieldName" select="translate(@path, '/', '.')"/>
        <xsl:param name="values"/>
        <xsl:param name="value" select="$values/*[local-name() = $fieldName]"/>
        <xsl:param name="errors" />
        <xsl:param name="uniqueId" />
        
        <div id="field-{$fieldId}-{$uniqueId}" class="field field-checkbox field-{$fieldId}">
             <div class="field-checkbox-flex input">
                <xsl:if test="$errors[@name = $fieldName]/error">
                    <xsl:attribute name="class">field-checkbox-flex input invalid</xsl:attribute>
                 </xsl:if>
                <input id="{$fieldId}-{$uniqueId}" type="checkbox" name="{$fieldName}">
                    <xsl:if test="$value = 'true'"><xsl:attribute name="checked">checked</xsl:attribute></xsl:if>
                </input>
                <label for="{$fieldId}-{$uniqueId}">
                    <xsl:copy-of select="label/node()" />
                    <xsl:call-template name="contenttype-form-field-mandatory" />
                </label>
             </div> 
        </div>
    </xsl:template>
    
    <!-- Override to add parent div flex -->
    <xsl:template name="tos-field">
        <xsl:param name="tos-mode"/>
        <xsl:param name="tos-page-id"/>
        <xsl:param name="tos-link-label"/>
        
        <div class="field field-checkbox tos">
            <xsl:if test="/user-signup/errors/field[@name = 'tos']/error">
                <xsl:call-template name="error-field" >
                    <xsl:with-param name="error" select="/user-signup/errors/field[@name = 'tos']/error" />
                </xsl:call-template>
            </xsl:if>
            
            <div class="field-checkbox-flex">
                <input type="checkbox" name="tos" id="tos-{$uniqueId}" value="true"/>
                <label for="tos-{$uniqueId}">
                    <xsl:choose>
                        <xsl:when test="$tos-mode = 'CONTENT'">
                            <i18n:text i18n:key="PLUGINS_WEB_SERVICE_USER_SIGNUP_VIEW_ACCEPT_TOS_CONTENT" i18n:catalogue="plugin.web"/>
                        </xsl:when>
                        <xsl:when test="$tos-link-label != ''">
                            <a href="{resolver:resolve('page', $tos-page-id)}" target="_blank">
                                <xsl:value-of select="$tos-link-label"/>
                            </a>
                        </xsl:when>
                        <xsl:otherwise>
                            <i18n:text i18n:key="PLUGINS_WEB_SERVICE_USER_SIGNUP_VIEW_ACCEPT_TOS" i18n:catalogue="plugin.web"/>
                            <xsl:text> </xsl:text>
                            <a href="{resolver:resolve('page', $tos-page-id)}" target="_blank">
                                <i18n:text i18n:key="PLUGINS_WEB_SERVICE_USER_SIGNUP_VIEW_ACCEPT_TOS_LINK" i18n:catalogue="plugin.web"/>
                            </a>
                        </xsl:otherwise>
                    </xsl:choose>
                </label>
            </div>
        </div>
    </xsl:template>
    
    <!-- Override to use <button> instead of <input> -->
    <xsl:template name="common-utils-input-submit">
        <xsl:param name="class-name"/>
        <xsl:param name="text"/>
        <xsl:param name="i18nkey"/>
        <xsl:param name="name"/>   
        <xsl:param name="i18ntitle"/>  
        <xsl:param name="title"/>             
        <xsl:param name="id"/>        
        <xsl:param name="onclick"/>        
        <xsl:param name="width"/>        
        <xsl:param name="type">submit</xsl:param>        
        <xsl:param name="disabled" select="false()"/>        
        
        <div class="button">
            <button type="{$type}">
                <xsl:if test="$class-name != ''">
                    <xsl:attribute name="class"><xsl:value-of select="$class-name"/></xsl:attribute>
                </xsl:if>
                <xsl:if test="$name != ''">
                    <xsl:attribute name="name"><xsl:value-of select="$name"/></xsl:attribute>
                </xsl:if>
                <xsl:if test="$id != ''">
                    <xsl:attribute name="id"><xsl:value-of select="$id"/></xsl:attribute>
                </xsl:if>
                <xsl:if test="$onclick">
                    <xsl:attribute name="onclick"><xsl:value-of select="$onclick"/></xsl:attribute>
                </xsl:if>
                <xsl:if test="$disabled">
                    <xsl:attribute name="disabled">disabled</xsl:attribute>
                </xsl:if>
                <xsl:if test="$width != ''">
                    <xsl:attribute name="style">width: <xsl:value-of select="$width"/>px;</xsl:attribute>
                </xsl:if>
                <xsl:choose>
                    <xsl:when test="$i18nkey != ''">
                        <i18n:text i18n:key="{substring-after($i18nkey, ':')}" i18n:catalogue="{substring-before($i18nkey, ':')}"/>
                    </xsl:when>
                    <xsl:otherwise><xsl:value-of select="$text"/></xsl:otherwise>                
                </xsl:choose>
            </button>
        </div>        
    </xsl:template>        
    
</xsl:stylesheet>
