// /src/ressources/components/header/header.js
(function () {
  function waitForElm(selector, { timeout = 10000 } = {}) {
    return new Promise((resolve, reject) => {
      const found = document.querySelector(selector);
      if (found) return resolve(found);

      const obs = new MutationObserver(() => {
        const el = document.querySelector(selector);
        if (el) {
          obs.disconnect();
          resolve(el);
        }
      });

      obs.observe(document.documentElement, { childList: true, subtree: true });

      if (timeout) {
        setTimeout(() => {
          obs.disconnect();
          reject(new Error(`Timeout: ${selector} introuvable`));
        }, timeout);
      }
    });
  }

  async function boot() {
    try {
      await waitForElm("#main-menu");

      initMenu();
      initTopBar();
      initMenuMobile();
      initSearchBar();
    } catch (e) {
      console.warn(e.message);
    }
  }

  function initSearchBar() {
    const searchForm = document.querySelector(".search__form__fixed");
    const searchInput = document.querySelector(".search__input");
    const overlaySearch = document.querySelector(".overlay-search");
    searchInput.addEventListener("click", () => {
      searchForm.classList.add("open");
      overlaySearch.classList.add("open");
    });

    overlaySearch.addEventListener("click", () => {
      searchForm.classList.remove("open");
      overlaySearch.classList.remove("open");
    });

  }

  function initTopBar() {
    const topBar = document.querySelector(".top-links__dropdown");
    const windowWidth = window.innerWidth;
    const isMobile = windowWidth < 750.02;

    if (isMobile) {
      topBar.removeAttribute("open");
    } else {
      topBar.setAttribute("open", "");
    }
  }

  window.addEventListener("resize", () => {
    initTopBar();
  });


  function initMenuMobile() {
    const menuDesktopToCopy = document.querySelector(".main-bar__nav");
    const menuMobileContainer = document.querySelector(".mobile-menu");
    const menuMobile = document.querySelector(".mobile-menu__content");
    const hamburger = document.querySelector(".main-bar__hamburger");
    const closeBtn = document.querySelector(".mobile-menu__close");
    const mobileMenuReturn = document.querySelector(".mobile-menu__return");
    const mobileMenuLogo = document.querySelector(".mobile-menu__logo");

    if (!menuDesktopToCopy || !menuMobileContainer || !menuMobile) return;

    if (menuMobileContainer.dataset.enhanced === "true") return;
    menuMobileContainer.dataset.enhanced = "true";

    menuMobile.innerHTML = menuDesktopToCopy.innerHTML;

    hamburger?.addEventListener("click", () => {
      menuMobileContainer.classList.toggle("open");
    });

    closeBtn?.addEventListener("click", () => {
      menuMobileContainer.classList.remove("open");
      closeAllSubmenus(menuMobile);
      mobileMenuReturn && (mobileMenuReturn.style.display = "none");
      mobileMenuLogo && (mobileMenuLogo.style.display = "block");

      menuMobile
        .querySelectorAll(".submenu__list__subtitle")
        .forEach((t) => (t.style.display = "block"));
      menuMobile
        .querySelectorAll(".submenu__list__item")
        .forEach((i) => (i.style.display = "none"));
      menuMobile
        .querySelectorAll(".submenu__list")
        .forEach((ul) => (ul.style.display = "block"));
      if (subMenuLevel === 2) {
        const titleEl = menuMobile.querySelector(".submenu__title");
        if (titleEl && lastSubMenuTitle !== null)
          titleEl.innerHTML = lastSubMenuTitle;
      }
      subMenuLevel = 0;
      currentSubmenu = null;
    });

    if (mobileMenuReturn && !mobileMenuReturn.dataset.bound) {
      mobileMenuReturn.addEventListener("click", () => {
        if (subMenuLevel === 2) {
          menuMobile
            .querySelectorAll(".submenu__list__subtitle")
            .forEach((t) => (t.style.display = "block"));
          menuMobile
            .querySelectorAll(".submenu__list")
            .forEach((ul) => (ul.style.display = "block"));
          menuMobile
            .querySelectorAll(".submenu__list__item")
            .forEach((i) => (i.style.display = "none"));
          const titleEl = menuMobile.querySelector(".submenu__title");
          if (titleEl && lastSubMenuTitle !== null)
            titleEl.innerHTML = lastSubMenuTitle;
          subMenuLevel = 1;
          return;
        }

        if (subMenuLevel === 1 && currentSubmenu) {
          currentSubmenu.hidden = true;
          currentSubmenu.classList.remove("open");

          const parentBtn = menuMobile.querySelector(
            `.menubtn[aria-controls="${currentSubmenu.id}"]`
          );
          parentBtn && parentBtn.setAttribute("aria-expanded", "false");

          mobileMenuReturn.style.display = "none";
          mobileMenuLogo && (mobileMenuLogo.style.display = "block");
          subMenuLevel = 0;
          currentSubmenu = null;
        }
      });
      mobileMenuReturn.dataset.bound = "true";
    }

    const buttons = menuMobile.querySelectorAll(".menubtn");
    buttons.forEach((btn) => {
      btn.addEventListener("click", () => {
        const controls = btn.getAttribute("aria-controls");
        const isOpen = btn.getAttribute("aria-expanded") === "true";

        if (
          isOpen &&
          subMenuLevel === 1 &&
          currentSubmenu &&
          currentSubmenu.id === controls
        ) {
          return;
        }

        if (isOpen) {
          closeAllSubmenus(menuMobile);
          mobileMenuReturn && (mobileMenuReturn.style.display = "none");
          mobileMenuLogo && (mobileMenuLogo.style.display = "block");
          subMenuLevel = 0;
          currentSubmenu = null;
          return;
        }

        const submenuMobile = controls
          ? menuMobile.querySelector(`#${controls}`)
          : null;
        if (!submenuMobile) return;

        openSubmenu(btn, submenuMobile, menuMobile);
        currentSubmenu = submenuMobile;
        mobileMenuReturn && (mobileMenuReturn.style.display = "block");
        mobileMenuLogo && (mobileMenuLogo.style.display = "none");
        subMenuLevel = 1;
      });
    });

    menuMobile
      .querySelectorAll(".submenu__list__subtitle")
      .forEach((titleEl) => {
        titleEl.addEventListener("click", () => {
          menuMobile
            .querySelectorAll(".submenu__list__subtitle")
            .forEach((t) => (t.style.display = "none"));
          const titleNode = menuMobile.querySelector(".submenu__title");
          if (titleNode) {
            lastSubMenuTitle = titleNode.innerHTML;
            titleNode.innerHTML = titleEl.innerHTML;
          }
          const parentUl = titleEl.parentElement;
          parentUl?.querySelectorAll(".submenu__list__item").forEach((item) => {
            item.style.display = "block";
          });

          const othersUls =
            titleEl.parentElement.parentElement.querySelectorAll(
              ".submenu__list"
            );
          othersUls.forEach((ul) => {
            ul.style.display = "none";
          });

          parentUl.style.display = "block";

          subMenuLevel = 2;
        });
      });
  }

  function closeAllSubmenus(scopeEl) {
    scopeEl.querySelectorAll(".submenu").forEach((submenu) => {
      submenu.hidden = true;
      submenu.classList.remove("open");
    });
    scopeEl.querySelectorAll(".menubtn").forEach((btn) => {
      btn.setAttribute("aria-expanded", "false");
    });
  }

  function openSubmenu(btn, submenuMobile, scopeEl) {
    closeAllSubmenus(scopeEl);
    submenuMobile.hidden = false;
    submenuMobile.classList.add("open");
    btn.setAttribute("aria-expanded", "true");
  }

  function initMenu() {
    const menu = document.getElementById("main-menu");
    if (!menu) return;

    const items = menu.querySelectorAll(".has-submenu");
    const triggers = menu.querySelectorAll(".has-submenu > .menubtn");

    function closeAll(except = null) {
      items.forEach((li) => {
        const btn = li.querySelector(".menubtn");
        const panel = document.getElementById(
          btn.getAttribute("aria-controls")
        );
        if (li !== except) {
          btn.setAttribute("aria-expanded", "false");
          panel.hidden = true;
          li.dataset.open = "false";
        }
      });
    }

    function open(li) {
      const btn = li.querySelector(".menubtn");
      const panel = document.getElementById(btn.getAttribute("aria-controls"));
      btn.setAttribute("aria-expanded", "true");
      panel.hidden = false;
      li.dataset.open = "true";
    }

    triggers.forEach((btn) => {
      btn.addEventListener("click", () => {
        const li = btn.closest(".has-submenu");
        const expanded = btn.getAttribute("aria-expanded") === "true";
        expanded ? closeAll() : (closeAll(), open(li));
      });

      btn.addEventListener("keydown", (e) => {
        const li = btn.closest(".has-submenu");
        const panel = document.getElementById(
          btn.getAttribute("aria-controls")
        );
        const firstItem = panel.querySelector('a[role="menuitem"]');

        if (e.key === "Enter" || e.key === " ") {
          e.preventDefault();
          const expanded = btn.getAttribute("aria-expanded") === "true";
          expanded ? closeAll() : (closeAll(), open(li), firstItem?.focus());
        } else if (e.key === "ArrowDown") {
          e.preventDefault();
          if (btn.getAttribute("aria-expanded") !== "true") open(li);
          firstItem?.focus();
        } else if (e.key === "ArrowUp") {
          e.preventDefault();
          if (btn.getAttribute("aria-expanded") !== "true") open(li);
          const links = panel.querySelectorAll('a[role="menuitem"]');
          links[links.length - 1]?.focus();
        } else if (e.key === "Escape") {
          closeAll();
          btn.focus();
        } else if (e.key === "ArrowRight" || e.key === "ArrowLeft") {
          e.preventDefault();
          const list = Array.from(triggers);
          const idx = list.indexOf(btn);
          const next =
            e.key === "ArrowRight"
              ? list[(idx + 1) % list.length]
              : list[(idx - 1 + list.length) % list.length];
          closeAll();
          next.focus();
        }
      });
    });

    menu.addEventListener("keydown", (e) => {
      const link = e.target.closest(
        '.submenu a[role="menuitem"], .submenu__cta a[role="menuitem"]'
      );
      if (!link) return;

      const container = link.closest(".submenu, .submenu__cta");
      if (!container) return;

      const submenu = container.closest(".submenu") || container;

      const links = Array.from(submenu.querySelectorAll('a[role="menuitem"]'));
      const i = links.indexOf(link);

      const parentBtn = menu.querySelector(
        `.menubtn[aria-controls="${submenu.id}"]`
      );

      if (e.key === "ArrowDown") {
        e.preventDefault();
        (links[i + 1] || links[0])?.focus();
      } else if (e.key === "ArrowUp") {
        e.preventDefault();
        (links[i - 1] || links[links.length - 1])?.focus();
      } else if (e.key === "Home") {
        e.preventDefault();
        links[0]?.focus();
      } else if (e.key === "End") {
        e.preventDefault();
        links[links.length - 1]?.focus();
      } else if (e.key === "Escape") {
        e.preventDefault();
        closeAll();
        parentBtn?.focus();
      } else if (e.key === "ArrowLeft" || e.key === "ArrowRight") {
        e.preventDefault();
        const list = Array.from(
          document.querySelectorAll(".has-submenu > .menubtn")
        );
        const idx = list.indexOf(parentBtn);
        const next =
          e.key === "ArrowRight"
            ? list[(idx + 1) % list.length]
            : list[(idx - 1 + list.length) % list.length];
        closeAll();
        next?.focus();
      }
    });

    document.addEventListener("click", (e) => {
      const insideSubmenu = !!e.target.closest(".submenu");
      const insideSubmenuCta = !!e.target.closest(".submenu__cta");
      const onMenuBtn = !!e.target.closest(".menubtn");

      if (!insideSubmenu && !insideSubmenuCta && !onMenuBtn) {
        closeAll();
      }
    });
    /* menu.addEventListener("focusout", () => {
      setTimeout(() => {
        if (!menu.contains(document.activeElement)) closeAll();
      }, 0);
    }); */
  }

  if (document.readyState === "loading") {
    document.addEventListener("DOMContentLoaded", boot, { once: true });
  } else {
    boot();
  }
})();
