(function () {
  function initA11ySelect(root) {
    const multiple = root.dataset.multiple === "true";
    const placeholder = root.dataset.placeholder || "Sélectionner";
    const searchable = root.dataset.searchable === "true";
    const btn = root.querySelector('.a11y-select__button');
    const list = root.querySelector('.a11y-select__list');
    const hidden = root.querySelector('.a11y-select__input');
    const options = Array.from(root.querySelectorAll('.a11y-select__option'));

    let activeIndex = -1;
    let searchInput = null;

    if (searchable && multiple) {
      const searchWrapper = document.createElement('div');
      searchWrapper.className = 'a11y-select__search';
      searchWrapper.innerHTML = '<input type="text" placeholder="Filtrer les critères ci-dessous" class="a11y-select__search-input" />';
      list.insertBefore(searchWrapper, list.firstChild);
      searchInput = searchWrapper.querySelector('input');

      const field = root.closest('.field');
      if (field) {
        const gridForm = field.parentElement;
        if (gridForm && gridForm.classList.contains('grid-form')) {
          const fields = Array.from(gridForm.querySelectorAll('.field'));
          const index = fields.indexOf(field);
          const totalFields = fields.length;

          if ((index + 1) % 3 === 0 || index === totalFields - 1) {
            root.classList.add('a11y-select--align-right');
          }
        }
      }

      searchInput.addEventListener('input', (e) => {
        const query = e.target.value.toLowerCase().trim();
        options.forEach(li => {
          const text = li.textContent.toLowerCase();
          if (text.includes(query)) {
            li.style.display = '';
          } else {
            li.style.display = 'none';
          }
        });
        list.querySelectorAll('.sub-title').forEach(subtitle => {
          const text = subtitle.textContent.toLowerCase();
          subtitle.style.display = text.includes(query) ? '' : 'none';
        });
      });

      searchInput.addEventListener('click', (e) => {
        e.stopPropagation();
      });

      searchInput.addEventListener('keydown', (e) => {
        if (e.key === 'Escape') {
          e.preventDefault();
          close();
        } else if (e.key === 'ArrowDown') {
          e.preventDefault();
          const firstVisible = options.find(opt => opt.style.display !== 'none');
          if (firstVisible) {
            const idx = options.indexOf(firstVisible);
            setActive(idx);
            list.focus();
          }
        }
      });
    }

    function open() {
      list.classList.add('is-open');
      btn.setAttribute('aria-expanded', 'true');

      if (searchInput) {
        //searchInput.focus({ preventScroll: true });
      } else {
        list.focus({ preventScroll: true });
        const firstSelected = options.findIndex(li => li.getAttribute('aria-selected') === 'true');
        setActive(firstSelected >= 0 ? firstSelected : 0);
      }

      document.addEventListener('mousedown', onDocClick);
      document.addEventListener('keydown', onDocKeydown);
    }
    function close() {
      list.classList.remove('is-open');
      btn.setAttribute('aria-expanded', 'false');
      btn.focus({ preventScroll: true });
      document.removeEventListener('mousedown', onDocClick);
      document.removeEventListener('keydown', onDocKeydown);
      activeIndex = -1;
    }
    function toggle() { list.classList.contains('is-open') ? close() : open(); }

    function onDocClick(e) { if (!root.contains(e.target)) close(); }
    function onDocKeydown(e) { if (e.key === 'Escape') close(); }

    function setActive(i) {
      if (i < 0 || i >= options.length) return;
      if (activeIndex >= 0) options[activeIndex].classList.remove('is-active');
      activeIndex = i;
      options[i].classList.add('is-active');
      const el = options[i];
      const r = el.getBoundingClientRect();
      const lr = list.getBoundingClientRect();
      if (r.top < lr.top) el.scrollIntoView(true);
      if (r.bottom > lr.bottom) el.scrollIntoView(false);
    }

    function commitValue() {
      if (multiple) {
        const selected = options
          .filter(li => li.getAttribute('aria-selected') === 'true')
          .map(li => li.dataset.value);
        hidden.value = selected.join(',');
        renderButtonLabel();
      } else {
        const sel = options.find(li => li.getAttribute('aria-selected') === 'true');
        hidden.value = sel ? sel.dataset.value : '';
        renderButtonLabel();
      }
    }

    function renderButtonLabel() {
      const selectedLis = options.filter(li => li.getAttribute('aria-selected') === 'true');
      if (!selectedLis.length) {
        btn.textContent = placeholder;
        btn.classList.add('is-placeholder');
        return;
      }
      btn.classList.remove('is-placeholder');
      if (!multiple) {
        btn.textContent = selectedLis[0].textContent.trim();
        return;
      }
      const labels = selectedLis.map(li => li.textContent.trim());
      btn.innerHTML = '<span class="a11y-select__button-badges">'
        + labels.map(t => `<span class="a11y-select__badge">${t}</span>`).join('')
        + '</span>';
    }

    function selectIndex(i) {
      const li = options[i];
      if (!li) return;
      const selected = li.getAttribute('aria-selected') === 'true';
      if (multiple) {
        li.setAttribute('aria-selected', String(!selected));
        const cb = li.querySelector('input[type="checkbox"]');
        if (cb) cb.checked = !selected;
      } else {
        options.forEach(o => o.setAttribute('aria-selected', 'false'));
        li.setAttribute('aria-selected', 'true');
      }
      commitValue();
      if (!multiple) close();
    }

    btn.addEventListener('click', toggle);
    options.forEach((li, i) => {
      li.addEventListener('click', (e) => {
        if (e.target.matches('input[type="checkbox"]')) {
          li.setAttribute('aria-selected', e.target.checked ? 'true' : 'false');
          commitValue();
          return;
        }

        const label = e.target.closest('label');
        if (label && multiple) {
          const cb = li.querySelector('input[type="checkbox"]');
          if (cb) {
            e.preventDefault();
            const newChecked = !cb.checked;
            cb.checked = newChecked;
            li.setAttribute('aria-selected', String(newChecked));
            commitValue();
            return;
          }
        }

        selectIndex(i);
      });
      li.addEventListener('mousemove', () => setActive(i));
    });

    btn.addEventListener('keydown', (e) => {
      if (['ArrowDown', 'ArrowUp', 'Enter', ' '].includes(e.key)) {
        e.preventDefault();
        open();
      }
    });

    list.addEventListener('keydown', (e) => {
      if (e.key === 'ArrowDown') { e.preventDefault(); setActive(Math.min(options.length - 1, activeIndex + 1)); }
      else if (e.key === 'ArrowUp') { e.preventDefault(); setActive(Math.max(0, activeIndex - 1)); }
      else if (e.key === 'Home') { e.preventDefault(); setActive(0); }
      else if (e.key === 'End') { e.preventDefault(); setActive(options.length - 1); }
      else if (e.key === 'Enter' || e.key === ' ') { e.preventDefault(); selectIndex(activeIndex); }
      else if (e.key === 'Escape') { e.preventDefault(); close(); }
      else if (e.key === 'Tab') { close(); }
    });

    options.forEach(li => {
      const cb = li.querySelector('input[type="checkbox"]');
      if (cb) {
        const ariaSelected = li.getAttribute('aria-selected') === 'true';
        const checkboxChecked = cb.checked;

        if (checkboxChecked && !ariaSelected) {
          li.setAttribute('aria-selected', 'true');
        }
        else if (ariaSelected) {
          cb.checked = true;
        }

        cb.addEventListener('click', (e) => e.stopPropagation());
      }
    });
    commitValue();
    renderButtonLabel();

    return {
      get value() { return hidden.value; },
      open, close, set value(v) {
        if (multiple) {
          const wanted = new Set(String(v).split(',').filter(Boolean));
          options.forEach(li => li.setAttribute('aria-selected', wanted.has(li.dataset.value) ? 'true' : 'false'));
          options.forEach(li => { const cb = li.querySelector('input'); if (cb) cb.checked = (li.getAttribute('aria-selected') === 'true'); });
        } else {
          options.forEach(li => li.setAttribute('aria-selected', (li.dataset.value === v) ? 'true' : 'false'));
        }
        commitValue();
      }
    };
  }

  document.addEventListener('DOMContentLoaded', () => {
    setTimeout(() => {
      
      document.querySelectorAll('.a11y-select').forEach(initA11ySelect);
    }, 500);
  });
})();
