(function () {
  function waitForElm(selector, { timeout = 10000 } = {}) {
    return new Promise((resolve, reject) => {
      const found = document.querySelector(selector);
      if (found) return resolve(found);

      const obs = new MutationObserver(() => {
        const el = document.querySelector(selector);
        if (el) {
          obs.disconnect();
          resolve(el);
        }
      });

      obs.observe(document.documentElement, { childList: true, subtree: true });

      if (timeout) {
        setTimeout(() => {
          obs.disconnect();
          reject(new Error(`Timeout: ${selector} introuvable`));
        }, timeout);
      }
    });
  }

  async function boot() {
    try {
      await waitForElm(".nav-formation");
      initFormationNav();
    } catch (e) {
      console.warn(e.message);
    }
  }

  function initFormationNav() {
    const nav = document.querySelector(".nav-formation");
    if (!nav) return;

    const links = nav.querySelectorAll("a[href^='#']");
    const sections = Array.from(links)
      .map((link) => document.querySelector(link.getAttribute("href")))
      .filter(Boolean);
    const navOffsetTop = nav.offsetTop;
    const heightHeader = document.querySelector(".header").offsetHeight;

    function handleFixedNav() {
        console.timeLog(window.scrollY);
        console.log(navOffsetTop);
      if (window.scrollY + heightHeader >= navOffsetTop) {
        nav.classList.add("fixed");
        nav.style.top = `${heightHeader}px`;
      } else {
        nav.classList.remove("fixed");
      }
    }

    links.forEach((link) => {
      link.addEventListener("click", (e) => {
        e.preventDefault();
        const targetId = link.getAttribute("href");
        const targetSection = document.querySelector(targetId);
        if (!targetSection) return;
        const navHeight = nav.offsetHeight;

        window.scrollTo({
          top: targetSection.offsetTop - navHeight - 120,
          behavior: "smooth",
        });
      });
    });

    function updateAriaCurrent() {
      let current = null;

      sections.forEach((section) => {
        const rect = section.getBoundingClientRect();
        if (
          rect.top <= window.innerHeight * 0.3 &&
          rect.bottom >= window.innerHeight * 0.3
        ) {
          current = section;
        }
      });

      links.forEach((link) => {
        if (current && link.getAttribute("href") === `#${current.id}`) {
          link.setAttribute("aria-current", "step");
        } else {
          link.removeAttribute("aria-current");
        }
      });
    }

    function onScroll() {
      handleFixedNav();
      updateAriaCurrent();
    }

    window.addEventListener("scroll", onScroll);

    handleFixedNav();
    updateAriaCurrent();
  }

  if (document.readyState === "loading") {
    document.addEventListener("DOMContentLoaded", boot, { once: true });
  } else {
    boot();
  }
})();
