/**
 * Sitemap (Plan du site)
 * @description Gestion des accordéons du plan du site avec accessibilité
 */

(function() {
    'use strict';

    /**
     * Fonction d'attente d'un élément dans le DOM
     */
    function waitForElm(selector, { timeout = 10000 } = {}) {
        return new Promise((resolve, reject) => {
            const found = document.querySelector(selector);
            if (found) return resolve(found);

            const obs = new MutationObserver(() => {
                const el = document.querySelector(selector);
                if (el) {
                    obs.disconnect();
                    resolve(el);
                }
            });

            obs.observe(document.documentElement, { childList: true, subtree: true });

            if (timeout) {
                setTimeout(() => {
                    obs.disconnect();
                    reject(new Error(`Timeout: ${selector} introuvable`));
                }, timeout);
            }
        });
    }

    /**
     * Initialisation du sitemap
     */
    function initSitemap() {
        Promise.all([
            waitForElm('.sitemap'),
        ])
        .then(() => {
            // Gestion des clics sur les boutons toggle
            document.addEventListener('click', (e) => {
                const btn = e.target.closest('.sitemap__toggle');
                if (!btn) return;

                const expanded = btn.getAttribute('aria-expanded') === 'true';
                const targetId = btn.getAttribute('aria-controls');
                const target = document.getElementById(targetId);

                if (!target) return;

                btn.setAttribute('aria-expanded', String(!expanded));
                target.hidden = expanded;

                // Mise à jour du label pour l'accessibilité
                btn.setAttribute('aria-label', expanded ? 'Afficher les sous-pages' : 'Masquer les sous-pages');
            });

            // Accessibilité clavier : ENTER / SPACE togglent aussi
            document.addEventListener('keydown', (e) => {
                const btn = e.target.closest('.sitemap__toggle');
                if (btn && (e.key === ' ' || e.key === 'Enter')) {
                    e.preventDefault();
                    btn.click();
                }
            });
        })
        .catch(err => {
            console.error('[Sitemap]', err);
        });
    }

    // Initialiser après 500ms
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', () => {
            setTimeout(() => {
                initSitemap();
            }, 500);
        });
    } else {
        setTimeout(() => {
            initSitemap();
        }, 500);
    }

})();
