<?xml version="1.0" encoding="UTF-8"?>
<!--
   Copyright 2025 Anyware Services

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
   -->
<xsl:stylesheet version="1.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform" 
                            xmlns:i18n="http://apache.org/cocoon/i18n/2.1" 
                            xmlns:ametys="org.ametys.web.transformation.xslt.AmetysXSLTHelper"
                            xmlns:resolver="org.ametys.cms.transformation.xslt.ResolveURIComponent"
                            xmlns:cms="http://ametys.org/cms/3.0"
                            xmlns:func="http://exslt.org/functions"
                            xmlns:docbook="http://docbook.org/ns/docbook" 
                            xmlns:dyn="http://exslt.org/dynamic"
                            exclude-result-prefixes="ametys resolver cms dyn func">
    
    <xsl:param name="isODFContent" select="false()"/>
    
    <!-- +
         | This function tests if a fieldset is not empty, 
         | ie. it is composed at least of a valued metadata or a non-empty fieldset.
         | Be careful, metadata of type 'boolean' are excluded for the test.
         | So if the fieldset has only boolean metadata that are valued, it will be considered as empty.
         | Parameters are:
         | - fieldset: the fieldset node
         | - [prefix]: the xpath of the node holding the values. By default /view/content/metadata/
         + -->
    <func:function name="cms:isNotFieldsetEmpty">
        <xsl:param name="fieldset"/>
        <xsl:param name="prefix">/view/content/metadata/</xsl:param>
        
        <xsl:variable name="xpath">
            <xsl:call-template name="compute-xpath">
                <xsl:with-param name="fieldset" select="$fieldset"/>
                <xsl:with-param name="prefix" select="$prefix"/>
            </xsl:call-template>
        </xsl:variable>
        
        <func:result select="dyn:evaluate($xpath)"/>
    </func:function>
    
    <!-- +
         | This template compute the xpath to be evaluated to test if the fieldset is empty
         | Parameters are:
         | - fieldset: the fieldset node
         | - [prefix]: the xpath of the node holding the values. By default /view/content/metadata/
         + -->
    <xsl:template name="compute-xpath">
        <xsl:param name="fieldset"/>
        <xsl:param name="prefix">/view/content/metadata/</xsl:param>
        
        <xsl:for-each select="$fieldset/fieldset|$fieldset/metadata[@type != 'boolean']">
            <xsl:if test="position() != 1"> or </xsl:if>
            <xsl:choose>
                <xsl:when test="@type = 'repeater'"><xsl:value-of select="concat($prefix, @name)"/>[not(@entryCount) or @entryCount != '0']</xsl:when>
                <xsl:when test="@type = 'rich-text'">cms:isNotRichTextEmpty(<xsl:value-of select="concat($prefix, @name)"/>)</xsl:when>
                <xsl:when test="@type = 'composite'">
                    <xsl:call-template name="compute-xpath">
                        <xsl:with-param name="fieldset" select="."/>
                        <xsl:with-param name="prefix" select="concat($prefix, @name, '/')"/>
                    </xsl:call-template>
                </xsl:when>
                <xsl:when test="local-name() = 'fieldset'">
                    <xsl:call-template name="compute-xpath">
                        <xsl:with-param name="fieldset" select="."/>
                        <xsl:with-param name="prefix" select="$prefix"/>
                    </xsl:call-template>
                </xsl:when>
                <xsl:otherwise><xsl:value-of select="concat($prefix, @name)"/></xsl:otherwise>
            </xsl:choose>
        </xsl:for-each>
    </xsl:template>
    
    <!-- +
         | Default template for display of a tab.
         + -->
    <xsl:template match="fieldset[@role='tab']" mode="display">
        <xsl:variable name="self" select="."/>
        
        <xsl:if test="cms:isNotFieldsetEmpty($self)">
            <section id="tab-{generate-id()}">
                <xsl:apply-templates select="." mode="section-class"/>
                <div class="container">
                    <div class="ametys-formation-section__col">
                        <h2 class="ametys-formation-title"><xsl:value-of select="label"/></h2>
                        
                        <xsl:apply-templates select="fieldset|metadata" mode="display"/> 
                    </div>
                </div>
            </section>
        </xsl:if>
    </xsl:template>
    
    <xsl:template match="fieldset[@role='tab']" mode="section-class">
        <xsl:attribute name="class">ametys-formation-section ametys-richtext</xsl:attribute>
    </xsl:template>
    
    <xsl:template match="fieldset[@role='tab' and position() mod 2 = 0]" mode="section-class">
        <xsl:attribute name="class">ametys-formation-section with-bg  ametys-richtext</xsl:attribute>
    </xsl:template>
    
    <xsl:template match="fieldset[@role='fieldset']" mode="display">
        <xsl:apply-templates select="fieldset|metadata" mode="display"/>
    </xsl:template>
    
    <xsl:template match="metadata[@type = 'repeater']" mode="display">
        <xsl:param name="name" select="@name"/>
        <xsl:param name="value" select="/view/content/metadata/*[local-name() = $name]"/>
        <xsl:param name="cssClassName">repeater</xsl:param>
         
        <xsl:if test="$value/@entryCount != '0'">
            <div class="{$cssClassName}" data-ametys-content-repeater="true">
                <div class="{$cssClassName}-label" data-ametys-content-repeater-label="true"><strong><xsl:value-of select="label"/></strong></div>
                <div class="{$cssClassName}-value" data-ametys-content-repeater-value="true">
                    <xsl:variable name="definition" select="."/>
                    
                    <xsl:for-each select="$value">
                        <xsl:if test="position() != 1"><br/></xsl:if>
                        <xsl:apply-templates select="$definition" mode="value">
                            <xsl:with-param name="value" select="."/>
                        </xsl:apply-templates>
                    </xsl:for-each>
                </div>
            </div>
        </xsl:if>
    </xsl:template>
    
    <xsl:template match="metadata[@type = 'composite']" mode="display">
        <xsl:param name="name" select="@name"/>
        <xsl:param name="value" select="/view/content/metadata/*[local-name() = $name]"/>
        <xsl:param name="cssClassName">composite</xsl:param>
         
        <xsl:if test="$value/*">
            <div class="{$cssClassName}" data-ametys-content-composite="true">
                <div class="{$cssClassName}-label" data-ametys-content-composite-label="true"><strong><xsl:value-of select="label"/></strong></div>
                <div class="{$cssClassName}-value" data-ametys-content-composite-value="true">
                    <xsl:variable name="definition" select="."/>
                    
                    <xsl:for-each select="$value/*">
                        <xsl:variable name="name" select="local-name()"/>
                        <xsl:apply-templates select="$definition/metadata[@name = $name]" mode="display">
                            <xsl:with-param name="value" select="."/>
                        </xsl:apply-templates>
                    </xsl:for-each>
                </div>
            </div>
        </xsl:if>
    </xsl:template>
    
    <xsl:template match="metadata[@type='rich-text']" mode="display">
        <xsl:param name="name" select="@name"/>
        <xsl:param name="value" select="/view/content/metadata/*[local-name() = $name]"/>
        
        <xsl:if test="$value/docbook:article">
            <div class="ametys-formation-{@name}">
                <h3 class="ametys-richtext-title-2"><xsl:value-of select="label"/></h3>
                
                <xsl:variable name="definition" select="."/>                      
                <xsl:apply-templates select="$definition" mode="value">
                    <xsl:with-param name="value" select="$value"/>
                </xsl:apply-templates>
            </div>
        </xsl:if>
    </xsl:template>
    
    <xsl:template match="metadata[@name='presentation' and @type='rich-text']" mode="display">
        <xsl:param name="name" select="@name"/>
        <xsl:param name="value" select="/view/content/metadata/*[local-name() = $name]"/>
        
        <xsl:if test="$value/docbook:article">
            <xsl:variable name="definition" select="."/>                      
            <xsl:apply-templates select="$definition" mode="value">
                <xsl:with-param name="value" select="$value"/>
            </xsl:apply-templates>
        </xsl:if>
    </xsl:template>
    
    <xsl:template match="metadata" mode="display">
        <xsl:param name="name" select="@name"/>
        <xsl:param name="value" select="/view/content/metadata/*[local-name() = $name]"/>
        <xsl:param name="cssClassName">field</xsl:param>
        
        <xsl:if test="$value">
            <div class="ametys-richtext">
                <h4 class="ametys-richtext-title-3"><xsl:value-of select="label"/></h4>
                
                <xsl:variable name="definition" select="."/>
                <xsl:choose>
                    <xsl:when test="count($value) > 1">
                        <ul class="custom">
                            <xsl:for-each select="$value">
                                <li>
                                    <xsl:apply-templates select="$definition" mode="value">
                                        <xsl:with-param name="value" select="."/>
                                    </xsl:apply-templates>
                                </li>
                            </xsl:for-each>
                        </ul>
                    </xsl:when>
                    <xsl:otherwise>
                        <p>
                            <xsl:apply-templates select="$definition" mode="value">
                                <xsl:with-param name="value" select="$value"/>
                            </xsl:apply-templates>
                        </p>
                    </xsl:otherwise>
                </xsl:choose>
            </div>
    </xsl:if>
    </xsl:template>
    
    <xsl:template match="metadata[@type='content']" mode="value">
        <xsl:param name="value"/>
        
        <xsl:variable name="contentId" select="$value/@id" />
        <xsl:if test="$contentId != ''">
            <xsl:choose>
                <xsl:when test="ametys:isReferenceTableContent($contentId)">
                    <xsl:value-of select="$value/@title"/>
                </xsl:when>
                <xsl:when test="$isODFContent">
                    <a href="{resolver:resolve('odf', $contentId)}">
                        <xsl:value-of select="$value/@title"/>
                    </a>
                </xsl:when>
                <xsl:otherwise>
                    <a href="{resolver:resolve('content', $contentId)}">
                        <xsl:value-of select="$value/@title"/>
                    </a>
                </xsl:otherwise>
            </xsl:choose>
        </xsl:if>
    </xsl:template>
    
   <xsl:template match="metadata[@type='boolean']" mode="value">
        <xsl:param name="value"/>
        
        <xsl:choose>
            <xsl:when test="$value = 'true'"><i18n:text i18n:key="CONTENT_TRUE_LABEL" i18n:catalogue="skin.{$skin}"/></xsl:when>
            <xsl:otherwise><i18n:text i18n:key="CONTENT_FALSE_LABEL" i18n:catalogue="skin.{$skin}"/></xsl:otherwise>
        </xsl:choose>
    </xsl:template>
        
    <xsl:template match="metadata[@type='long']" mode="value">
        <xsl:param name="value"/>
        
        <xsl:value-of select="$value"/>
    </xsl:template>
    
    <xsl:template match="metadata[@type='double']" mode="value">
        <xsl:param name="value"/>
        
        <xsl:value-of select="$value"/>
    </xsl:template>
    
    <xsl:template match="metadata[@type='date']" mode="value">
        <xsl:param name="value"/>
        
        <i18n:date pattern="medium" src-pattern="yyyy-MM-dd'T'HH:mm">
            <xsl:value-of select="$value"/>
        </i18n:date>
    </xsl:template>
        
    <xsl:template match="metadata[@type='datetime']" mode="value">
        <xsl:param name="value"/>
        
        <i18n:date pattern="medium" src-pattern="yyyy-MM-dd'T'HH:mm">
            <xsl:value-of select="$value"/>
        </i18n:date>
    </xsl:template>
    
    <xsl:template match="metadata[@type='reference']" mode="value">
        <xsl:param name="value"/>
        <a>
            <xsl:attribute name="href"><xsl:value-of select="$value"/></xsl:attribute>
            <xsl:value-of select="$value"/>
        </a>
    </xsl:template>
        
    <xsl:template match="metadata[@type='file']" mode="value">
        <xsl:param name="value"/>
        <a href="{resolver:resolve($value/@type, $value/@path, 'true')}" title="{$value/@filename} ({$value/@size})">
            <xsl:value-of select="$value/@filename"/>
        </a>
    </xsl:template>
    
    <xsl:template match="metadata[@type='geocode']" mode="value">
        <xsl:param name="value"/>
        
        <xsl:value-of select="concat(substring-before($value/@latitude, '.'), '.', substring(substring-after($value/@latitude, '.'), 1, 4))"/><xsl:text>°N , </xsl:text><xsl:value-of select="concat(substring-before($value/@longitude, '.'), '.', substring(substring-after($value/@longitude, '.'), 1, 4))"/><xsl:text>°E</xsl:text>
        <br/>
        <a external="external">
            <xsl:attribute name="href">https://maps.google.com/maps?q={$value/@latitude},{$value/@longitude}&amp;z=30&amp;ll={$value/@latitude},{$value/@longitude}</xsl:attribute>
            <i18n:text i18n:key="CONTENT_VIEW_GOOGLEMAPS_LINK" i18n:catalogue="plugin.cms"/>
        </a>
    </xsl:template>
    
    <xsl:template match="metadata[@type='rich-text']" mode="value">
        <xsl:param name="value"/>
        <xsl:param name="level" select="2"/>
        <xsl:param name="uniqueId" select="generate-id()"/>
        
        <xsl:apply-templates select="$value/docbook:article">
            <xsl:with-param name="level" select="$level"/>
        </xsl:apply-templates>
    </xsl:template>
    
    <xsl:template match="metadata[@type='repeater']" mode="value">
        <xsl:param name="value"/>
        
        <xsl:variable name="repeaterDef" select="."/>
                    
        <xsl:for-each select="$value/entry">
            <div data-ametys-content-repeater-entry="true">
                <xsl:for-each select="*">
                    <xsl:variable name="name" select="local-name()"/>
                    <xsl:apply-templates select="$repeaterDef/metadata[@name = $name]" mode="display">
                        <xsl:with-param name="value" select="."/>
                    </xsl:apply-templates>
                </xsl:for-each>
            </div>
        </xsl:for-each>
    </xsl:template>
        
    <xsl:template match="metadata" mode="value">
        <xsl:param name="value"/>
        
        <xsl:value-of select="$value"/>
    </xsl:template>
    
    
    <!-- +
         | Specific rendering into accordion
         + -->
    <xsl:template match="metadata[@type='rich-text']" mode="faq-details">
        <xsl:param name="name" select="@name"/>
        <xsl:param name="value" select="/view/content/metadata/*[local-name() = $name]"/>
        <xsl:param name="entryUniqueId"/>
        
        <xsl:apply-templates select="." mode="value">
            <xsl:with-param name="value" select="$value"/>
            <xsl:with-param name="uniqueId" select="concat($entryUniqueId, '-', generate-id())"/>
        </xsl:apply-templates>
    </xsl:template>   
    
    <xsl:template match="metadata" mode="faq-details">
        <xsl:param name="name" select="@name"/>
        <xsl:param name="value" select="/view/content/metadata/*[local-name() = $name]"/>
        
        <xsl:if test="$value">
            <p>
                <strong><xsl:value-of select="label"/> : </strong>
                <xsl:variable name="definition" select="."/>
                <xsl:for-each select="$value">
                    <xsl:if test="position() != 1"><xsl:text>, </xsl:text></xsl:if>
                    <xsl:apply-templates select="$definition" mode="value">
                        <xsl:with-param name="value" select="."/>
                    </xsl:apply-templates>
                </xsl:for-each>
            </p>
        </xsl:if>
    </xsl:template>  
    
    <!-- +
         | Specific rendering in table
         + -->
    <xsl:template match="metadata[@type='rich-text']" mode="table-row">
        <xsl:param name="name" select="@name"/>
        <xsl:param name="value" select="/view/content/metadata/*[local-name() = $name]"/>
        <xsl:param name="entryUniqueId"/>
        
        <xsl:if test="$value">
            <tr>
                <th style="width: 28.5%;">
                    <p><xsl:value-of select="label"/></p>
                </th>
                <td>
                    <xsl:apply-templates select="." mode="value">
                        <xsl:with-param name="value" select="$value"/>
                        <xsl:with-param name="uniqueId" select="concat($entryUniqueId, '-', generate-id())"/>
                    </xsl:apply-templates>
                </td>
            </tr>
        </xsl:if>
    </xsl:template>   
    
    <xsl:template match="metadata" mode="table-row">
        <xsl:param name="name" select="@name"/>
        <xsl:param name="value" select="/view/content/metadata/*[local-name() = $name]"/>
        
        <xsl:if test="$value">
            <tr>
                <th style="width: 28.5%;">
                    <p><xsl:value-of select="label"/></p>
                </th>
                <td>
                    <p>
                        <xsl:variable name="definition" select="."/>
                        <xsl:for-each select="$value">
                            <xsl:if test="position() != 1"><xsl:text>, </xsl:text></xsl:if>
                            <xsl:apply-templates select="$definition" mode="value">
                                <xsl:with-param name="value" select="."/>
                            </xsl:apply-templates>
                        </xsl:for-each>
                    </p>
                </td>
            </tr>
        </xsl:if>
    </xsl:template>  
    
</xsl:stylesheet>