/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odforientation;

import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Optional;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.LocaleUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.cms.search.advanced.AbstractTreeNode;
import org.ametys.cms.search.advanced.TreeLeaf;
import org.ametys.core.util.LambdaUtils;
import org.ametys.odf.enumeration.OdfReferenceTableEntry;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.odf.enumeration.OdfReferenceTableHelper.SortField;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.orgunit.OrgUnitFactory;
import org.ametys.odf.orgunit.RootOrgUnitProvider;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.query.QueryHelper;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.runtime.model.View;
import org.ametys.runtime.model.type.DataContext;
import org.ametys.web.frontoffice.search.SearchService;
import org.ametys.web.frontoffice.search.instance.SearchServiceInstance;
import org.ametys.web.frontoffice.search.instance.SearchServiceInstanceManager;
import org.ametys.web.frontoffice.search.metamodel.impl.ContentReferencingSearchServiceCriterionDefinition;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.Page.PageType;
import org.ametys.web.repository.page.ZoneItem;
import org.ametys.web.repository.page.ZoneItem.ZoneType;

/**
 * Generates the exhaustive list of orgunits
 */
public class OdfReferenceTableItemsGenerator extends ServiceableGenerator
{
    private AmetysObjectResolver _resolver;
    private RootOrgUnitProvider _rootOrgUnitProvider;
    private ContentTypesHelper _cTypeHelper;
    private SearchServiceInstanceManager _searchServiceInstanceManager;
    private OdfReferenceTableHelper _odfReferenceTableHelper;

    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _rootOrgUnitProvider = (RootOrgUnitProvider) serviceManager.lookup(RootOrgUnitProvider.ROLE);
        _cTypeHelper = (ContentTypesHelper) serviceManager.lookup(ContentTypesHelper.ROLE);
        _searchServiceInstanceManager = (SearchServiceInstanceManager) serviceManager.lookup(SearchServiceInstanceManager.ROLE);
        _odfReferenceTableHelper = (OdfReferenceTableHelper) serviceManager.lookup(OdfReferenceTableHelper.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        String lang = parameters.getParameter("lang", (String) request.getAttribute("sitemapLanguage"));
        Locale defaultLocale = LocaleUtils.toLocale(lang);
        
        ZoneItem zoneItem = (ZoneItem) request.getAttribute(ZoneItem.class.getName());
        
        String refTableId = zoneItem.getServiceParameters().getValue("refTableId");
        
        contentHandler.startDocument();
        
        XMLUtils.startElement(contentHandler, "items");
        
        List<OdfReferenceTableEntry> items = _odfReferenceTableHelper.getItems(refTableId, false, new SortField("order", true));
        
        items.stream().forEach(LambdaUtils.wrapConsumer(e -> {
            
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("id", e.getId());
            attrs.addCDATAAttribute("code", e.getCode());
            
            XMLUtils.createElement(contentHandler, "item", attrs, e.getLabel(lang));
        }));
        
        _saxSearchForm(zoneItem, refTableId);
        
        XMLUtils.endElement(contentHandler, "items");
        
        contentHandler.endDocument();
    }
    
    private void _saxSearchForm(ZoneItem zoneItem, String refTableId) throws SAXException
    {
        String criterionInputName = _getRefTableCriterionInputName(zoneItem, refTableId);
        if (criterionInputName != null)
        {
            XMLUtils.startElement(contentHandler, "search-form");
            XMLUtils.createElement(contentHandler, "criterion", criterionInputName);
            XMLUtils.endElement(contentHandler, "search-form");
        }
    }
    
    private String _getRefTableCriterionInputName(ZoneItem zoneItem, String refTableId)
    {
        String searchPageId = zoneItem.getServiceParameters().getValue("search-page");
        if (StringUtils.isNotEmpty(searchPageId))
        {
            Page page = _resolver.resolveById(searchPageId);
            if (page.getType() == PageType.CONTAINER) 
            {
                Optional< ? extends ZoneItem> searchZoneItem = page.getZones().stream()
                    .flatMap(z -> z.getZoneItems().stream())
                    .filter(zi -> zi.getType() == ZoneType.SERVICE)
                    .filter(zi -> zi.getServiceId().equals(SearchService.ROLE))
                    .findFirst();
                    
                if (searchZoneItem.isPresent())
                {
                    SearchServiceInstance serviceInstance = _searchServiceInstanceManager.get(searchZoneItem.get().getId());
                    return serviceInstance.getCriterionTree()
                            .map(AbstractTreeNode::getFlatLeaves)
                            .orElseGet(Collections::emptyList)
                            .stream()
                            .map(TreeLeaf::getValue)
                            .filter(c -> c.getCriterionDefinition() instanceof ContentReferencingSearchServiceCriterionDefinition && ((ContentReferencingSearchServiceCriterionDefinition) c.getCriterionDefinition()).getContentTypeId().equals(refTableId))
                            .findFirst()
                            .map(c -> c.getName())
                            .orElseGet(null);
                }
            }
        }
        
        getLogger().warn("No criteria found related to selected reference table '" + refTableId + "'");
        return null;
    }
    
}
