/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odforientation;

import java.io.IOException;
import java.util.Collections;
import java.util.Locale;
import java.util.Optional;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.LocaleUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.cms.search.advanced.AbstractTreeNode;
import org.ametys.cms.search.advanced.TreeLeaf;
import org.ametys.core.util.LambdaUtils;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.orgunit.OrgUnitFactory;
import org.ametys.odf.orgunit.RootOrgUnitProvider;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.query.QueryHelper;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.runtime.model.View;
import org.ametys.runtime.model.type.DataContext;
import org.ametys.web.frontoffice.search.SearchService;
import org.ametys.web.frontoffice.search.instance.SearchServiceInstance;
import org.ametys.web.frontoffice.search.instance.SearchServiceInstanceManager;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.Page.PageType;
import org.ametys.web.repository.page.ZoneItem;
import org.ametys.web.repository.page.ZoneItem.ZoneType;

/**
 * Generates the exhaustive list of orgunits
 */
public class OrgUnitsGenerator extends ServiceableGenerator
{
    private AmetysObjectResolver _resolver;
    private RootOrgUnitProvider _rootOrgUnitProvider;
    private ContentTypesHelper _cTypeHelper;
    private SearchServiceInstanceManager _searchServiceInstanceManager;

    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _rootOrgUnitProvider = (RootOrgUnitProvider) serviceManager.lookup(RootOrgUnitProvider.ROLE);
        _cTypeHelper = (ContentTypesHelper) serviceManager.lookup(ContentTypesHelper.ROLE);
        _searchServiceInstanceManager = (SearchServiceInstanceManager) serviceManager.lookup(SearchServiceInstanceManager.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        String lang = parameters.getParameter("lang", (String) request.getAttribute("sitemapLanguage"));
        Locale defaultLocale = LocaleUtils.toLocale(lang);
        
        ZoneItem zoneItem = (ZoneItem) request.getAttribute(ZoneItem.class.getName());
        
        String contentView = zoneItem.getServiceParameters().getValue("contentView");
        View view = _cTypeHelper.getView(contentView, new String[] {OrgUnitFactory.ORGUNIT_CONTENT_TYPE}, new String[0]);
        
        
        contentHandler.startDocument();
        
        XMLUtils.startElement(contentHandler, "orgunits");
        
        Expression expr = new ContentTypeExpression(Operator.EQ, OrgUnitFactory.ORGUNIT_CONTENT_TYPE);
        String xPathQuery = QueryHelper.getXPathQuery(null, OrgUnitFactory.ORGUNIT_NODETYPE, expr);
        AmetysObjectIterable<OrgUnit> orgUnits = _resolver.query(xPathQuery);
        
        orgUnits.stream().filter(ou -> !_rootOrgUnitProvider.isRoot(ou)).forEach(LambdaUtils.wrapConsumer(ou -> {
            
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("id", ou.getId());
            
            XMLUtils.startElement(contentHandler, "orgunit", attrs);
            ou.dataToSAX(contentHandler, view, DataContext.newInstance().withLocale(defaultLocale).withEmptyValues(false));
            XMLUtils.endElement(contentHandler, "orgunit");
        }));
        
        _saxSearchForm(zoneItem);
        
        XMLUtils.endElement(contentHandler, "orgunits");
        
        contentHandler.endDocument();
    }
    
    private void _saxSearchForm(ZoneItem zoneItem) throws SAXException
    {
        String criterionInputName = _getOrgUnitCriterionInputName(zoneItem);
        if (criterionInputName != null)
        {
            XMLUtils.startElement(contentHandler, "search-form");
            XMLUtils.createElement(contentHandler, "criterion", criterionInputName);
            XMLUtils.endElement(contentHandler, "search-form");
        }
    }
    
    private String _getOrgUnitCriterionInputName(ZoneItem zoneItem)
    {
        String searchPageId = zoneItem.getServiceParameters().getValue("search-page");
        if (StringUtils.isNotEmpty(searchPageId))
        {
            Page page = _resolver.resolveById(searchPageId);
            if (page.getType() == PageType.CONTAINER) 
            {
                Optional< ? extends ZoneItem> searchZoneItem = page.getZones().stream()
                    .flatMap(z -> z.getZoneItems().stream())
                    .filter(zi -> zi.getType() == ZoneType.SERVICE)
                    .filter(zi -> zi.getServiceId().equals(SearchService.ROLE))
                    .findFirst();
                    
                if (searchZoneItem.isPresent())
                {
                    SearchServiceInstance serviceInstance = _searchServiceInstanceManager.get(searchZoneItem.get().getId());
                    return serviceInstance.getCriterionTree()
                            .map(AbstractTreeNode::getFlatLeaves)
                            .orElseGet(Collections::emptyList)
                            .stream()
                            .map(TreeLeaf::getValue)
                            .filter(c -> StringUtils.endsWith(c.getCriterionDefinition().getName(), "$org.ametys.plugins.odf.Content.abstractProgram$orgUnit"))
                            .findFirst()
                            .map(c -> c.getName())
                            .orElseGet(null);
                    
                }
            }
        }
        
        return null;
    }
    
}
