/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/*
 * Dataviz helper to create charts using Chart.js v4.5.1
 * Requires Chart.js and chartjs-plugin-datalabels
*/
var Dataviz = {
    
	/** Maximum width in pixels to consider a small screen (for responsive design) */
	SMALL_SCREEN_MAX_WIDTH: 600,
	
	/** The default options for chart */
	defaultOpts: {
		// slice colors ordered to biggest to smallest values
		backgroundColor: [
            '#FF6384', // red color of index 0
            '#FF9F40', // orange color of index 1
            '#4BC0C0', // green color of index 2
            '#36A2EB', // blue color of index 3
            '#9966FF', // purple color of index 4
            '#C9CBCF', // grey color of index 5
            '#FFCD56' // yellow color of index 6
        ],
		// formatting options for the ticks (x and y axes) (see https://www.chartjs.org/docs/latest/axes/#tick-configuration for more options)
		scales: {
			ticks: {
				color: '#000',
				font: {
					size: 12,
					family: "'Helvetica', 'Arial', sans-serif"
				}
			}
		},
		// formatting options for the legend (see https://www.chartjs.org/docs/latest/configuration/legend.html for more options)
		legend: {
			position: 'right',
			labels: {
				color: '#000',
				font: {
                    size: 12,
                    family: "'Helvetica', 'Arial', sans-serif",
                },
				useBorderRadius: true,
				borderRadius: 5,
				boxHeight: 22,
				boxWidth: 40
			},				
			align: 'start',
		},
		// formatting options for the data labels displayed on the chart's slices (see https://chartjs-plugin-datalabels.netlify.app/guide/options.html for more options)
		datalabels: {
            color: '#000',
			backgroundColor: '#fff',
            font: {
                size: 12,
                family: "'Helvetica', 'Arial', sans-serif",
            },
			weight: 'normal',
			align: 'center',
			textAlign: 'center'
        }
	},
	
    /**
     * Create chart results
     * @param {String} canvasId The id of the canvas
	 * @param {String} type The type of the chart (doughnut, pie, etc.)
     * @param {Object|Object[]} dataset the chart dataset ({label: 'Label', data: [{label: 'label1', value: 10}, {label: 'label2', value: 20}, ...]})
	 * @param {Boolean} percentage true if data values expressed as a percentage
	 * @param {Boolean} maxSlicesToShow maximum number of slices to show on the chart (others will be grouped into "Others"). Only supported for single dataset charts.
     * @param {Object} [opts] additional options to customize the chart (colors, legend options, datalabels options, etc.)
     */
    createChart: function(canvasId, type, dataset, percentage, maxSlicesToShow, opts)
    {
		// Merge default options with user options
		var opts = $j.extend(true, {}, Dataviz.defaultOpts, opts || {});
		
        var labels = [];
		var datasets = [];
		
		var maxSlicesToShow = maxSlicesToShow || -1;
					
		var multipleDatasets = $j.isArray(dataset) && dataset.length > 1;
		dataset = $j.isArray(dataset) ? dataset : [dataset]; // Ensure dataset is an array
        
		dataset.forEach((ds, index) => {
            // Transform data: sort, convert to percentage if needed and group small slices into "Others" if needed (only for single dataset charts !)
            var data = this._transformData(ds.data, percentage, !multipleDatasets, maxSlicesToShow);
            
			var chartData = [];
            data.forEach((d) => {
                if (index === 0)
                {
					// Fill labels only once
                    labels.push(window.innerWidth > Dataviz.SMALL_SCREEN_MAX_WIDTH ? d.label : d.shortLabel || d.label); // Use shortLabel if defined on small screen
                }
                chartData.push(d.value);
            });
            
            ds.data = chartData;
			ds = $j.extend({
				backgroundColor: opts.backgroundColor
			}, ds);
			           
            datasets.push(ds);
       });
        
        // Chart configuration		
		var config = {
            type: type || 'doughnut',
			
			plugins: [ChartDataLabels], // Enable the datalabels plugin to display values on the chart's slices
			
            data: {
                datasets: datasets,
                labels: labels
            },
            options: {
                responsive: true,
				maintainAspectRatio: true,
				plugins: {
					// Change options for ALL labels of THIS CHART
			      	datalabels: $j.extend(opts.datalabels, {
						formatter: function(value, context) {
							// return context.chart.data.labels[context.dataIndex] + "\n" + value + "%"; // Show label and percentage value with % sign on slices
							return value + "%"; // Show percentage value with % sign on slices
                        }
					})
				}
			}
        };
		
		if (type === 'doughnut' || type === 'pie')
        {
			// Specific options for pie/doughnut chart
            config.options = $j.extend(true, config.options, this._getPieOptions(opts));
        }
		else if (type === 'bar')
		{
			// Specific options for horizontal bar chart
			config.options = $j.extend(true, config.options, this._getHorizontalBarOptions(opts));
		}
		
		// Show chart
        var ctx = document.getElementById(canvasId).getContext('2d');
        new Chart(ctx, config);
    },
	
	/**
	 * Sort data, add an "Others" slice if needed and convert values to percentage
     * @param {Object[]} data the chart data [{label: 'label1', value: 10}, {label: 'label2', value: 20}, ...]
     * @param {Boolean} percentage true if data values expressed as a percentage
	 * @param {Boolean} groupIfNeeded true to group small slices into "Others" if needed
     * @param {Boolean} maxSlicesToShow maximum number of slices to show on the chart (others will be grouped into "Others")
     * @return {Object[]} the transformed data
	 */
	_transformData: function(data, percentage, groupIfNeeded, maxSlicesToShow)
	{
		// Retrieve the total number
		let totalNb = data.reduce((sum, entry) => sum + entry.value, 0);
		let remainPercentage = percentage ? Math.round((100 - totalNb)*100)/100 : 0; // Calculate the remaining percentage to reach 100%
		
		if (groupIfNeeded)
		{
			// Sort data to have the smallest values at the end
			data.sort((a, b) => a.value - b.value);
			
			// If maxSlicesToShow is defined and the data length exceeds it, group the smallest slices into "Others"
	        if (maxSlicesToShow > 0 && data.length > maxSlicesToShow)
	        {
	            // Get the slices to group
	            let othersData = data.slice(0, data.length - maxSlicesToShow + 1);
	            
	            // Calculate the total value of the grouped slices
	            let othersValue = othersData.reduce((sum, entry) => sum + entry.value, 0);
				othersValue += remainPercentage;
	            
	            // Keep only the biggest slices and add the "Others" slice
	            data = data.slice(data.length - maxSlicesToShow + 1);
				
				// Sort data to have the biggest values first
				data.sort((a, b) => b.value - a.value);
				
				if (othersValue > 0)
				{
					// Insert the "Others" slice at the end
		            let othersSlice = { label: '{{i18n PROGRAM_DATAVIZ_OTHERS_SLICE_LABEL}}', value: othersValue };
					data.push(othersSlice);
				}
	        }
			else if (remainPercentage > 0) // If there is a remaining percentage to reach 100%, add an "Others" slice
	        {
				// Sort data to have the biggest values first
				data.sort((a, b) => b.value - a.value);
			
				// Insert the "Others" slice at the end			
	            let othersSlice = { label: '{{i18n PROGRAM_DATAVIZ_OTHERS_SLICE_LABEL}}', value: remainPercentage };
	            data.push(othersSlice);
	        }
			else
			{
	            // Sort data to display the biggest values first
	            data.sort((a, b) => b.value - a.value);
	        }
		}
		else if (remainPercentage > 0) // If there is a remaining percentage to reach 100%, add an "Others" slice
        {
            // Insert the "Others" slice at the end			
            let othersSlice = { label: '{{i18n PROGRAM_DATAVIZ_OTHERS_SLICE_LABEL}}', value: remainPercentage };
            data.push(othersSlice);
        }
        
        // If data values are not expressed as a percentage, convert them
		if (percentage !== true)
		{
			// Replace number by the percentage
	        data.forEach(item => {
		 	 let p = (item.value / totalNb) * 100; // value in %
	         item.value = Math.round(p * 100) / 100; // Round to 2 decimals
	        });
		}
		
		return data;
	},
	
	/**
     * Get specific options for pie/doughnut chart
     * @param {Object} opts The user options
     * @return {Object} The pie/doughnut specific options
     */
    _getPieOptions: function(opts)
	{
		return {
			plugins: {
                legend: {
                    position: opts.legend.position,
                    labels: $j.extend({
                        generateLabels(chart) {
                            // Get the legend items
                            let legendItems = Chart.controllers.pie.overrides.plugins.legend.labels.generateLabels(chart);
                            
                            // Return truncated legend items depending of the chart size
                            return legendItems.map(item => {
                                return {
                                    ...item,
                                    text: Dataviz._truncateText(chart, item.text)
                                };
                            });
                        }
                    }, opts.legend.labels),
					align: opts.legend.align,
                }
            }
        };
	},
	
	/**
     * Get specific options for horizontal bar chart
     * @param {Object} opts The user options
     * @return {Object} The horizontal bar options
     */
    _getHorizontalBarOptions: function(opts)
	{
		var me = this;
		return {
			indexAxis: 'y', // Make the bar chart horizontal
            maintainAspectRatio: false, // Allow to adapt the height of the chart to fit all labels
            aspectRatio: 1, // Initial aspect ratio (width / height)
			scales: {
				x: {
		        	ticks: opts.scales.ticks
		      	},
			  	y: {
		        	ticks: $j.extend(opts.scales.ticks, {
				  		callback: function(value, index, ticks) {
						  	if (window.innerWidth < Dataviz.SMALL_SCREEN_MAX_WIDTH)
	                      	{
								let labels = this.getLabelForValue(value)
								return $j.isArray(labels) ? labels.join(' / ') : labels; // Show short label on small screen on one line
                          	}
                          	else
                          	{
                             	// TODO truncate long labels if needed
                             	return this.getLabelForValue(value); // Show full label on large screen (on two lines if needed)
                          	}
                  		}
                	})
              	}
            },
            plugins: {
                // Change options for ALL labels of THIS CHART
              	legend: opts.legend,
              	datalabels: $j.extend(opts.datalabels, {
                    formatter: function(value, context) {
                        return value + "%"; // Show percentage value with % sign on bars
                    }
                }),
                tooltip: {
                    callbacks: {
                        label: function(context) {
                            return context.parsed.x + "%"; // Show percentage value with % sign in tooltip
                        }
                    }
                }
            }
		}
	},
    
    /**
     * Truncate the text depending of the size of the chart. Add ... if the text is truncated
     * @param {Object} chart The chart object
     * @param {String} text The text to truncate
     */ 
    _truncateText: function(chart, text)
    {
        let ctx = chart.ctx;
        let maxWidth = chart.legend.width - chart.legend.options.labels.boxWidth - 20; // Remove 20px to be sure that the text is not truncated by the canvas

        let characters = text.split('');
        let line = "";
        for (let character of characters) 
        {
            let testLine = line ? line + character : character;
            let width = ctx.measureText(testLine).width;
            if (width > maxWidth) 
            {
                line = line.substring(0, line.length - 3).trim() + "…";
                break;
            }
            else 
            {
                line = testLine;
            }
        }

        return line;
    }
}
