(function() {
  'use strict';

  function init() {
    initProgrammeCards();
    initProgrammeTabs();
    initBackToCards();
    initProgrammeTreeView();
  }

  AmetysFront.Event.listen('dom:ready', init);

  function initProgrammeCards() {
    const formationTabs = document.querySelectorAll('.ametys-programme-cards[role="tablist"] button[role="tab"]');
    const formationPanels = document.querySelectorAll('.ametys-programme-content[role="tabpanel"]');

    if (!formationTabs.length) return;

    formationTabs.forEach(tab => {
      tab.addEventListener('click', function(e) {
        e.preventDefault();
        switchFormationTab(tab, formationTabs, formationPanels);
      });

      tab.addEventListener('keydown', function(e) {
        handleFormationTabKeyboard(e, tab, formationTabs, formationPanels);
      });
    });
  }

  function switchFormationTab(selectedTab, allTabs, allPanels) {
    const targetPanelId = selectedTab.getAttribute('aria-controls');

    allTabs.forEach(tab => {
      tab.setAttribute('aria-selected', 'false');
      tab.setAttribute('tabindex', '-1');
      tab.classList.remove('active');
    });

    selectedTab.setAttribute('aria-selected', 'true');
    selectedTab.setAttribute('tabindex', '0');
    selectedTab.classList.add('active');

    allPanels.forEach(panel => {
      panel.hidden = true;
    });

    const targetPanel = document.getElementById(targetPanelId);
    if (targetPanel) {
      targetPanel.hidden = false;

      setTimeout(() => {
        targetPanel.scrollIntoView({
          behavior: 'smooth',
          block: 'nearest'
        });

        const firstSemesterTab = targetPanel.querySelector('.ametys-programme-tab[aria-selected="true"]');
        if (firstSemesterTab) {
          firstSemesterTab.focus();
        }
      }, 100);
    }
  }

  function handleFormationTabKeyboard(e, currentTab, allTabs, allPanels) {
    const tabs = Array.from(allTabs);
    const currentIndex = tabs.indexOf(currentTab);
    let targetTab = null;

    switch(e.key) {
      case 'ArrowLeft':
        e.preventDefault();
        targetTab = tabs[currentIndex - 1] || tabs[tabs.length - 1];
        break;
      case 'ArrowRight':
        e.preventDefault();
        targetTab = tabs[currentIndex + 1] || tabs[0];
        break;
      case 'Home':
        e.preventDefault();
        targetTab = tabs[0];
        break;
      case 'End':
        e.preventDefault();
        targetTab = tabs[tabs.length - 1];
        break;
      case 'Enter':
      case ' ':
        e.preventDefault();
        switchFormationTab(currentTab, allTabs, allPanels);
        return;
    }

    if (targetTab) {
      switchFormationTab(targetTab, allTabs, allPanels);
      targetTab.focus();
    }
  }

  function initProgrammeTabs() {
    const semesterTablists = document.querySelectorAll('.ametys-programme-tabs__nav[role="tablist"]');

    if (!semesterTablists.length) return;

    semesterTablists.forEach(tablist => {
      const tabButtons = tablist.querySelectorAll('.ametys-programme-tab[role="tab"]');

      tabButtons.forEach(button => {
        button.addEventListener('click', function() {
          switchSemesterTab(button, tabButtons);
        });

        button.addEventListener('keydown', function(e) {
          handleSemesterTabKeyboard(e, button, tabButtons);
        });
      });
    });
  }

  function switchSemesterTab(selectedButton, allButtons) {
    const panelId = selectedButton.getAttribute('aria-controls');

    allButtons.forEach(btn => {
      btn.classList.remove('active');
      btn.setAttribute('aria-selected', 'false');
      btn.setAttribute('tabindex', '-1');
    });

    selectedButton.classList.add('active');
    selectedButton.setAttribute('aria-selected', 'true');
    selectedButton.setAttribute('tabindex', '0');

    const formationPanel = selectedButton.closest('.ametys-programme-content[role="tabpanel"]');
    if (!formationPanel) return;

    const panels = formationPanel.querySelectorAll('.ametys-programme-panel');
    panels.forEach(panel => {
      panel.classList.remove('active');
      panel.setAttribute('hidden', '');
    });

    const selectedPanel = document.getElementById(panelId);
    if (selectedPanel) {
      selectedPanel.classList.add('active');
      selectedPanel.removeAttribute('hidden');
    }
  }


  function initBackToCards() {
    const backLink = document.getElementById('back-to-cards');
    if (!backLink) return;

    backLink.addEventListener('click', function(e) {
      e.preventDefault();

      const cardsSection = document.getElementById('programme-cards');
      if (cardsSection) {
        cardsSection.scrollIntoView({
          behavior: 'smooth',
          block: 'start'
        });

        setTimeout(() => {
          const activeCard = document.querySelector('.ametys-programme-card.active');
          if (activeCard) {
            activeCard.focus();
          }
        }, 100);
      }
    });
  }


  function handleSemesterTabKeyboard(e, currentButton, allButtons) {
    const buttons = Array.from(allButtons);
    const currentIndex = buttons.indexOf(currentButton);
    let targetButton = null;

    switch(e.key) {
      case 'ArrowLeft':
        e.preventDefault();
        targetButton = buttons[currentIndex - 1] || buttons[buttons.length - 1];
        break;
      case 'ArrowRight':
        e.preventDefault();
        targetButton = buttons[currentIndex + 1] || buttons[0];
        break;
      case 'Home':
        e.preventDefault();
        targetButton = buttons[0];
        break;
      case 'End':
        e.preventDefault();
        targetButton = buttons[buttons.length - 1];
        break;
      case 'Enter':
      case ' ':
        e.preventDefault();
        switchSemesterTab(currentButton, allButtons);
        return;
    }

    if (targetButton) {
      switchSemesterTab(targetButton, allButtons);
      targetButton.focus();
    }
  }


  function initProgrammeTreeView() {
    const items = document.querySelectorAll('.ametys-programme-item');

    if (!items.length) return;

    items.forEach(item => {
      const toggleBtn = item.querySelector('.item-toggle');
      const children = item.querySelector('.ametys-programme-item__children');

      if (!toggleBtn) return;

      const hasChildren = children && children.children.length > 0;

      if (!hasChildren) {
        item.classList.add('no-children');
        toggleBtn.setAttribute('disabled', 'true');
        toggleBtn.setAttribute('aria-hidden', 'true');
        return;
      }

      toggleBtn.addEventListener('click', function(e) {
        e.preventDefault();
        e.stopPropagation();
        toggleProgrammeItem(item, toggleBtn);
      });

      toggleBtn.addEventListener('keydown', function(e) {
        if (e.key === 'Enter' || e.key === ' ') {
          e.preventDefault();
          e.stopPropagation();
          toggleProgrammeItem(item, toggleBtn);
        }
      });

      toggleBtn.setAttribute('aria-expanded', 'false');
    });
  }


  function toggleProgrammeItem(item, toggleBtn) {
    const isOpen = item.classList.contains('is-open');

    if (isOpen) {
      item.classList.remove('is-open');
      if (toggleBtn) {
        toggleBtn.setAttribute('aria-expanded', 'false');
      }
    } else {
      item.classList.add('is-open');
      if (toggleBtn) {
        toggleBtn.setAttribute('aria-expanded', 'true');
      }
    }
  }

})();
