/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
 
;(function ($) {
  'use strict';

  var AmetysSearchAutocomplete = window.AmetysSearchAutocomplete || {};

  AmetysSearchAutocomplete = (function () {

    function AmetysSearchAutocomplete(element, options) {

      var $self = this;

      /**
       * Current element.
       *
       * @public
       */
      this.$element = $(element);

      /**
       * Current options set by the caller including defaults.
       *
       * @public
       */
      $self.options = $.extend(true, {}, AmetysSearchAutocomplete.defaults, options);

      this.initialize();

    }

    return AmetysSearchAutocomplete;

  }());

  AmetysSearchAutocomplete.defaults = {
    containerSelector: null,
    autocompleteTargetSelector: null,
    autocompletePagesSelector: null,
    autocompleteSkillsSelector: null,
    autocompleteTagsSelector: null,
    autocompleteServerUrl: null,
    autocompleteSiteName: null,
    autocompleteLang: null,
    autocompleteZoneItemId: null,
    searchUrl: null,
    openPageResultsInNewTab: false,
	showMoreResultsLink: false,
    autocompleteMoreSelector: null,
    autocompleteQuerySelector: null
  };

  /**
   * Initialization of the plugin.
   *
   * @protected
   */
  AmetysSearchAutocomplete.prototype.initialize = function () {
    // vars
    var $self = this,
        search = $self.$element[0].id,
        target = null,
        searchTimeout = null;

    $self.options.containerSelector = '#' + $self.$element[0].id;

    $self.options.autocompleteTargetSelector = $self.options.containerSelector + ' [data-autocomplete-target]';
    $self.options.autocompletePagesSelector = $self.options.containerSelector + ' [data-autocomplete-data="pages"]';
    $self.options.autocompleteSkillsSelector = $self.options.containerSelector + ' [data-autocomplete-data="skills"]';
    $self.options.autocompleteTagsSelector = $self.options.containerSelector + ' [data-autocomplete-data="tags"]';
	$self.options.autocompleteMoreSelector = $self.options.containerSelector + ' [data-autocomplete-more]';
	$self.options.autocompleteQuerySelector = $self.options.containerSelector + ' [data-autocomplete-query]';

    $($self.options.autocompleteTargetSelector).on('keyup', function () {
            
            if (searchTimeout != null) {
			    clearTimeout(searchTimeout);
			}

            var $this = $(this);
            var me = this;
			  
            searchTimeout = setTimeout(function() {
                searchTimeout = null;  
    
	            target = $this.data('autocomplete-target');
	            
	            var $target = $(target),
	                
	                $pages = $($self.options.autocompletePagesSelector),
	                $skills = $($self.options.autocompleteSkillsSelector),
	                $tags = $($self.options.autocompleteTagsSelector),
	                
	                pages = [],
	                skills = [],
	                tags = [],
                    
                    openPageResultsInNewTab = $self.options.openPageResultsInNewTab;
					showMoreResultsLink = $self.options.showMoreResultsLink;
	            
	            // Show / Hide
	            if ($this.val().length > 2) {
	                
	                var params = {
	                    q: $this.val(),
	                    siteName: $self.options.autocompleteSiteName,
	                    lang: $self.options.autocompleteLang,
	                    zoneItemId: $self.options.autocompleteZoneItemId
	                }
	                // Fetch data
	                $.getJSON($self.options.autocompleteServerUrl, params).done(function (data) {
	                    
						let nbPages = data.pages ? data.pages.length : 0;
                        let nbSkills = data.skills ? data.skills.length : 0;
                        let nbTags = data.tags ? data.tags.length : 0;
                        
                        if (nbPages + nbSkills + nbTags === 0) {
							// No result
                            $target.fadeOut(200);
                            return;
                        }
						
						let nbOfSlotsWithResults = 0;
                        if (nbPages > 0) nbOfSlotsWithResults++;
                        if (nbSkills > 0) nbOfSlotsWithResults++;
                        if (nbTags > 0) nbOfSlotsWithResults++;
						
						// Adjust maxNbItemsPerType depending on number of types with results
						let maxNbItemsPerType = Math.ceil(10 / nbOfSlotsWithResults);;
                        
						// Distribute available slots if one type has less than maxNbItemsPerType results
                        if (nbPages < maxNbItemsPerType) {
                            let remainingSlots = maxNbItemsPerType - nbPages;
                            maxNbItemsPerType += Math.min(remainingSlots, maxNbItemsPerType - nbSkills);
                        }
                        if (nbSkills < maxNbItemsPerType) {
                            let remainingSlots = maxNbItemsPerType - nbSkills;
                            maxNbItemsPerType += Math.min(remainingSlots, maxNbItemsPerType - nbTags);
                        }
                        
                        // Limit number of displayed results
                        if (data.pages) {
                            data.pages = data.pages.slice(0, maxNbItemsPerType);
                        }
                        if (data.skills) {
                            data.skills = data.skills.slice(0, maxNbItemsPerType);
                        }
                        if (data.tags) {
                            data.tags = data.tags.slice(0, maxNbItemsPerType);
                        }
						
	                    if (data.pages && $pages) {
	                        
	                        $.each(data.pages, function (i) {
	                            
								// Hightlight search term in title
                                var title = data.pages[i].title;
                                var regex = new RegExp('(' + $this.val() + ')', 'gi');
                                title = title.replace(regex, '<strong>$1</strong>');
								
	                            pages.push('<li class="autocomplete__item" role="option">' +
	                                '<a class="autocomplete__item__link" href="' + data.pages[i].url + (!openPageResultsInNewTab || data.pages[i].url.indexOf("javascript:") == 0 ? '">' : '" target="_blank">') +
	                                title +
	                                '</a>' +
	                                '</li>');
	                        });
	                        
	                        $pages.html(pages.join(''));
							$pages.parent().show();
	                    }
						else
						{
							$pages.parent().hide();
						}
	                    
	                    if (data.skills && $skills) {
	                        
	                        $.each(data.skills, function (i) {
	                            
								// Hightlight search term in title
                                var title = data.skills[i].title;
                                var regex = new RegExp('(' + $this.val() + ')', 'gi');
                                title = title.replace(regex, '<strong>$1</strong>');
																
	                            skills.push('<li class="autocomplete__item">' +
	                                '<a class="autocomplete__item__link" href="' + $self.options.searchUrl + '?submit-form=&skillId=' + data.skills[i].id + '">' + title + '</a>' +
	                                '</li>');
	                        });
	                        
	                        $skills.html(skills.join(''));
							$skills.parent().show();
                        }
                        else
                        {
                            $skills.parent().hide();
	                    }
	                    
	                    if (data.tags && $tags) {
	                        
	                        $.each(data.tags, function (i) {
	                            
								// Hightlight search term in title
                                var title = data.tags[i].title;
                                var regex = new RegExp('(' + $this.val() + ')', 'gi');
                                title = title.replace(regex, '<strong>$1</strong>');
																
	                            tags.push('<li class="autocomplete__item">' +
	                                '<a class="autocomplete__item__link" href="' + $self.options.searchUrl + '?submit-form=&tagId=' + data.tags[i].id + '">' + title + '</a>' +
	                                '</li>');
	                            
	                        });
	                        $tags.html(tags.join(''));
							$tags.parent().show();
                        }
                        else
                        {
                            $tags.parent().hide();
	                    }
	                });
	                
	                // End Fetch data
	                
	                $target.fadeIn(200);
					                    
                    // Update "more results" link and query display
					if (showMoreResultsLink)
                    {
						$($self.options.autocompleteQuerySelector).html($this.val());
						$($self.options.autocompleteMoreSelector).attr('href', $self.options.searchUrl + '?submit-form=&' + $this[0].name + "=" + encodeURIComponent($this.val()));
                        $($self.options.autocompleteMoreSelector).show();
                    }
                    else
                    {
                        $($self.options.autocompleteMoreSelector).hide();
                    }
	                
	            } else {
	                
	                pages = [];
	                skills = [];
	                tags = [];
	                
	                $target.fadeOut(200, function () {
	                    
	                    if ($pages) $pages.html('');
	                    
	                    if ($skills) $skills.html('');
	                    
	                    if ($tags) $tags.html('');
	                    
	                });
	            }
	            
	            // End Show / Hide
            }, 500);
        });
        
    // Hide
    $(document).on('click', 'body', function (e) {
    
        var $target = $(e.target).parents(target),
            $search = $(e.target).parents(search);
    
            if (($target.length === 1) || ($search.length === 1)) {
                return 0;
            } else {
                $(target).fadeOut(200);
            }
    });
  };

  $.fn.HSAmetysSearchAutocomplete = function () {

    var _ = this,
      opt = arguments[0],
      args = Array.prototype.slice.call(arguments, 1),
      l = _.length,
      i,
      ret;

    for (i = 0; i < l; i++) {
      if (typeof opt == 'object' || typeof opt == 'undefined') {

        _[i].AmetysSearchAutocomplete = new AmetysSearchAutocomplete(_[i], opt);

      } else {

        ret = _[i].AmetysSearchAutocomplete[opt].apply(_[i].AmetysSearchAutocomplete, args);

      }

      if (typeof ret != 'undefined') {

        return ret;

      }
    }

    return _;

  };

})(jQuery);