/**
 * Gallery Photos with Lightbox
 * @description Gestion de la galerie photos avec système de lightbox
 */

(function() {
    'use strict';

    class GalleryLightbox {
        constructor() {
            this.lightbox = document.getElementById('lightbox');
            if (!this.lightbox) return;

            this.lightboxImage = this.lightbox.querySelector('.lightbox__image');
            this.lightboxClose = this.lightbox.querySelector('.lightbox__close');
            this.lightboxPrev = this.lightbox.querySelector('.lightbox__prev');
            this.lightboxNext = this.lightbox.querySelector('.lightbox__next');
            this.lightboxCurrent = this.lightbox.querySelector('.lightbox__current');
            this.lightboxTotal = this.lightbox.querySelector('.lightbox__total');

            this.galleryItems = document.querySelectorAll('[data-gallery-item]');
            this.currentIndex = 0;
            this.images = [];

            this.init();
        }

        init() {
            // Collecter toutes les images de la galerie
            this.galleryItems.forEach((item, index) => {
                const href = item.getAttribute('href');
                const img = item.querySelector('img');
                const alt = img ? img.getAttribute('alt') : '';

                this.images.push({
                    src: href,
                    alt: alt
                });

                // Ajouter un événement click sur chaque item
                item.addEventListener('click', (e) => {
                    e.preventDefault();
                    this.openLightbox(index);
                });
            });

            // Mettre à jour le compteur total
            if (this.lightboxTotal) {
                this.lightboxTotal.textContent = this.images.length;
            }

            // Événements des boutons
            if (this.lightboxClose) {
                this.lightboxClose.addEventListener('click', () => this.closeLightbox());
            }

            if (this.lightboxPrev) {
                this.lightboxPrev.addEventListener('click', () => this.prevImage());
            }

            if (this.lightboxNext) {
                this.lightboxNext.addEventListener('click', () => this.nextImage());
            }

            // Fermer avec Escape
            document.addEventListener('keydown', (e) => {
                if (e.key === 'Escape' && this.lightbox.classList.contains('active')) {
                    this.closeLightbox();
                }
            });

            // Navigation avec les flèches du clavier
            document.addEventListener('keydown', (e) => {
                if (!this.lightbox.classList.contains('active')) return;

                if (e.key === 'ArrowLeft') {
                    this.prevImage();
                } else if (e.key === 'ArrowRight') {
                    this.nextImage();
                }
            });

            // Fermer en cliquant sur le fond
            this.lightbox.addEventListener('click', (e) => {
                if (e.target === this.lightbox) {
                    this.closeLightbox();
                }
            });
        }

        openLightbox(index) {
            this.currentIndex = index;
            this.updateImage();
            this.lightbox.classList.add('active');
            document.body.style.overflow = 'hidden';
        }

        closeLightbox() {
            this.lightbox.classList.remove('active');
            document.body.style.overflow = '';
        }

        updateImage() {
            const image = this.images[this.currentIndex];
            if (this.lightboxImage) {
                this.lightboxImage.src = image.src;
                this.lightboxImage.alt = image.alt;
            }

            if (this.lightboxCurrent) {
                this.lightboxCurrent.textContent = this.currentIndex + 1;
            }
        }

        prevImage() {
            this.currentIndex = (this.currentIndex - 1 + this.images.length) % this.images.length;
            this.updateImage();
        }

        nextImage() {
            this.currentIndex = (this.currentIndex + 1) % this.images.length;
            this.updateImage();
        }
    }

    // Initialiser la lightbox quand le DOM est prêt avec un délai de 500ms
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', () => {
            setTimeout(() => {
                new GalleryLightbox();
            }, 500);
        });
    } else {
        setTimeout(() => {
            new GalleryLightbox();
        }, 500);
    }

})();
