function waitForElm(selector, { timeout = 10000 } = {}) {
  return new Promise((resolve, reject) => {
    const found = document.querySelector(selector);
    if (found) return resolve(found);

    const obs = new MutationObserver(() => {
      const el = document.querySelector(selector);
      if (el) {
        obs.disconnect();
        resolve(el);
      }
    });

    obs.observe(document.documentElement, { childList: true, subtree: true });

    if (timeout) {
      setTimeout(() => {
        obs.disconnect();
        reject(new Error(`Timeout: ${selector} introuvable`));
      }, timeout);
    }
  });
}

(function () {
  Promise.all([
    waitForElm('#video-lightbox'),
    waitForElm('.play-btn[data-video-id]')
  ])
  .then(([lightbox]) => {
    const playerWrap = lightbox.querySelector('[data-player]');
    const closeEls   = lightbox.querySelectorAll('[data-close]');
    let lastFocused  = null;

    function openLightbox(videoId) {
      lastFocused = document.activeElement;
      const src = `https://www.youtube.com/embed/${videoId}?autoplay=1&rel=0&modestbranding=1`;
      playerWrap.innerHTML = `<iframe
        width="100%" height="100%" frameborder="0" allow="autoplay; encrypted-media"
        allowfullscreen src="${src}" title="Lecture vidéo YouTube"></iframe>`;

      lightbox.setAttribute('aria-hidden', 'false');
      document.body.classList.add('ylightbox-open');
      lightbox.querySelector('.ylightbox__close')?.focus();
      document.addEventListener('keydown', onKeydown);
	  AmetysFront.Accessibility.trapFocus(lightbox);
    }

    function closeLightbox() {
      lightbox.setAttribute('aria-hidden', 'true');
      document.body.classList.remove('ylightbox-open');
      playerWrap.innerHTML = ''; 
      document.removeEventListener('keydown', onKeydown);
	  AmetysFront.Accessibility.untrapFocus(lightbox);
      if (lastFocused && typeof lastFocused.focus === 'function') lastFocused.focus();
    }

    function onKeydown(e) {
      if (e.key === 'Escape' || e.key === 'Esc') closeLightbox();
    }

  
    document.querySelectorAll('.play-btn[data-video-id]').forEach(btn => {
      btn.addEventListener('click', (e) => {
        e.preventDefault();
        const id = btn.getAttribute('data-video-id');
        if (id) openLightbox(id);
      });
    });

  
    closeEls.forEach(el => el.addEventListener('click', closeLightbox));
    lightbox.addEventListener('click', (e) => {
      const withinDialog = e.target.closest('.ylightbox__dialog');
      if (!withinDialog) closeLightbox();
    });
  })
  .catch(err => {
    console.error(err);
  });
})();