<?xml version="1.0" encoding="UTF-8"?>
<!--
   Copyright 2025 Anyware Services

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
   -->
<xsl:stylesheet version="1.0"
        xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
        xmlns:i18n="http://apache.org/cocoon/i18n/2.1"
        xmlns:sitemap="http://www.ametys.org/inputdata/sitemap/3.0"
        xmlns:xalan="http://xml.apache.org/xalan"
        xmlns:ametys="org.ametys.web.transformation.xslt.AmetysXSLTHelper"
        xmlns:resolver="org.ametys.cms.transformation.xslt.ResolveURIComponent"
        xmlns:odf="org.ametys.odf.OdfXSLTHelper"
        xmlns:escaper="org.apache.commons.lang.StringEscapeUtils"
        xmlns:docbook="http://docbook.org/ns/docbook" 
        xmlns:exslt="http://exslt.org/common"
        exclude-result-prefixes="sitemap ametys resolver odf escaper docbook exslt">

    <!-- Ordered semesters -->
    <xsl:variable name="periodItems" select="odf:getTableRefItems('odf-enumeration.Period', /view/content/@language, true())"/>
    
    <!-- +
         | Get the available semesters (period) among the child program items.
         + -->
    <xsl:template name="get-available-semesters">
        <xsl:param name="parentProgramItems"/>
        
        <semesters>
            <xsl:for-each select="$periodItems/item">
                <xsl:if test="$parentProgramItems//container[attributes/nature/@code='semestre' and attributes/period/@code= current()/@code]">
                    <semester>
                        <xsl:copy-of select="@*"/>
                    </semester>
                </xsl:if>
            </xsl:for-each>
        </semesters>
    </xsl:template>
    
    <!-- + 
         | Main template to expose the program structure. Parameters are:
         | - title: the title of this section (optional)
         | - currentProgramItem: the current program item that contains subprograms and/or years 
         + -->
    <xsl:template name="program-structure">
        <xsl:param name="title"/>
        <xsl:param name="currentProgramItem"/>
        
        <xsl:variable name="subprogramsAndYears" select="$currentProgramItem/subprogram|$currentProgramItem/container[attributes/nature/@code='annee']"/>
        
        <xsl:choose>
            <xsl:when test="$subprogramsAndYears">
                <!-- Program structure with subprograms and/or years -->
                <xsl:if test="$title">
                    <h3 class="ametys-formation-title-2" id="programme-cards"><xsl:copy-of select="$title"/></h3>
                </xsl:if>
                
                <!-- Navigation on child program items -->
                <xsl:call-template name="program-structure-nav">
                    <xsl:with-param name="parentProgramItems" select="$subprogramsAndYears"/>
                </xsl:call-template>
                
                <xsl:call-template name="program-structure-content">
                    <xsl:with-param name="parentProgramItems" select="$subprogramsAndYears"/>
                </xsl:call-template>
            </xsl:when>
            <xsl:when test="$currentProgramItem/container[attributes/nature/@code='semestre']">
                <!-- There is no subprograms or years, but semesters under the root program item -->
                <xsl:variable name="directSemesters" select="$currentProgramItem/container[attributes/nature/@code='semestre']"/>
                
                <xsl:variable name="availableSemesters">
                    <xsl:call-template name="get-available-semesters">
                        <xsl:with-param name="parentProgramItems" select="$currentProgramItem"/>
                    </xsl:call-template>
                </xsl:variable>
        
                <xsl:call-template name="semesters-tab-nav">
                    <xsl:with-param name="availableSemesters" select="exslt:node-set($availableSemesters)"/>
                    <xsl:with-param name="currentSemesters" select="$directSemesters"/>
                </xsl:call-template>
        
                <xsl:call-template name="semesters-tab-content">
                    <xsl:with-param name="currentSemesters" select="$directSemesters"/>
                    <xsl:with-param name="parentProgramItem" select="$currentProgramItem"/>
                </xsl:call-template>
            </xsl:when>
            <xsl:when test="$currentProgramItem/courselist[course]">
                <!-- There is no subprograms nor years nor semesters but courses under the root program item -->
                <ul id="accordion-{@code}" class="ametys-accordion accordion mb-0">
                    <xsl:apply-templates select="$currentProgramItem/courselist" mode="course-row"/>
                </ul>
            </xsl:when>
        </xsl:choose>
    </xsl:template>
    
    <!-- + 
         | Template to navigate on subprograms and years. Parameters are:
         | - parentProgramItems: the root program items to be display (by default, the subprograms or year containers of first level)
         + -->
    <xsl:template name="program-structure-nav">
        <xsl:param name="parentProgramItems"/>
        
        <div class="ametys-programme-cards ametys-programme-cards--flexible" role="tablist" aria-label="Programmes et années">
	        <xsl:for-each select="$parentProgramItems">
                <xsl:variable name="activeCls"><xsl:if test="position() = 1"> active</xsl:if></xsl:variable>
                
	            <button class="ametys-programme-card ametys-programme-card--w-33" 
                     id="tab-program-{@code}-{generate-id()}"
                     aria-controls="tabpanel-program-{@code}-{generate-id()}"
                     data-programme="{@code}"
                     role="tab"
                     tabindex="0"
                     aria-selected="{position() = 1}">
                    
                    <div class="ametys-programme-card__header">
                        <!-- <xsl:call-template name="program-nav-item-date"/> -->
                        <h4 class="ametys-programme-card__title"><xsl:value-of select="@title"/></h4>
                    </div>
                    
                    <xsl:call-template name="program-nav-item-badge"/>
	            </button>
	        </xsl:for-each>                                   
        </div>
    </xsl:template>  
    
    <xsl:template name="program-nav-item-date">
        <div class="date-icon">
            <svg width="24" height="20" viewBox="0 0 24 20" fill="none" xmlns="http://www.w3.org/2000/svg" focusable="false" aria-hidden="true">
                <path
                    d="M7.88285 3.33366H7.09475C5.99104 3.33366 5.43878 3.33366 5.01722 3.51531C4.6464 3.6751 4.34514 3.92989 4.1562 4.24349C3.94141 4.60001 3.94141 5.06707 3.94141 6.00049V6.66699M7.88285 3.33366H15.7657M7.88285 3.33366V1.66699M15.7657 3.33366H16.5542C17.6579 3.33366 18.209 3.33366 18.6306 3.51531C19.0014 3.6751 19.3036 3.92989 19.4926 4.24349C19.7072 4.59966 19.7072 5.06615 19.7072 5.99775V6.66699M15.7657 3.33366V1.66699M3.94141 6.66699V14.0005C3.94141 14.9339 3.94141 15.4004 4.1562 15.7569C4.34514 16.0705 4.6464 16.3257 5.01722 16.4855C5.43837 16.667 5.98996 16.667 7.09151 16.667H16.5571C17.6586 16.667 18.2094 16.667 18.6306 16.4855C19.0014 16.3257 19.3036 16.0705 19.4926 15.7569C19.7072 15.4007 19.7072 14.9349 19.7072 14.0033V6.66699M3.94141 6.66699H19.7072M15.7657 13.3337H15.7677L15.7677 13.3353L15.7657 13.3353V13.3337ZM11.8243 13.3337H11.8263L11.8262 13.3353L11.8243 13.3353V13.3337ZM7.88285 13.3337H7.88482L7.88477 13.3353L7.88285 13.3353V13.3337ZM15.7677 10.0003V10.002L15.7657 10.002V10.0003H15.7677ZM11.8243 10.0003H11.8263L11.8262 10.002L11.8243 10.002V10.0003ZM7.88285 10.0003H7.88482L7.88477 10.002L7.88285 10.002V10.0003Z"
                    stroke="#2E1E3D" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"></path>
                <path
                    d="M7.88285 3.33366H7.09475C5.99104 3.33366 5.43878 3.33366 5.01722 3.51531C4.6464 3.6751 4.34514 3.92989 4.1562 4.24349C3.94141 4.60001 3.94141 5.06707 3.94141 6.00049V6.66699M7.88285 3.33366H15.7657M7.88285 3.33366V1.66699M15.7657 3.33366H16.5542C17.6579 3.33366 18.209 3.33366 18.6306 3.51531C19.0014 3.6751 19.3036 3.92989 19.4926 4.24349C19.7072 4.59966 19.7072 5.06615 19.7072 5.99775V6.66699M15.7657 3.33366V1.66699M3.94141 6.66699V14.0005C3.94141 14.9339 3.94141 15.4004 4.1562 15.7569C4.34514 16.0705 4.6464 16.3257 5.01722 16.4855C5.43837 16.667 5.98996 16.667 7.09151 16.667H16.5571C17.6586 16.667 18.2094 16.667 18.6306 16.4855C19.0014 16.3257 19.3036 16.0705 19.4926 15.7569C19.7072 15.4007 19.7072 14.9349 19.7072 14.0033V6.66699M3.94141 6.66699H19.7072M15.7657 13.3337H15.7677L15.7677 13.3353L15.7657 13.3353V13.3337ZM11.8243 13.3337H11.8263L11.8262 13.3353L11.8243 13.3353V13.3337ZM7.88285 13.3337H7.88482L7.88477 13.3353L7.88285 13.3353V13.3337ZM15.7677 10.0003V10.002L15.7657 10.002V10.0003H15.7677ZM11.8243 10.0003H11.8263L11.8262 10.002L11.8243 10.002V10.0003ZM7.88285 10.0003H7.88482L7.88477 10.002L7.88285 10.002V10.0003Z"
                    stroke="url(#paint0_linear_4633_7395)" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"></path>
                <defs>
                    <linearGradient id="paint0_linear_4633_7395" x1="11.8243" y1="1.66699" x2="11.8243" y2="16.667" gradientUnits="userSpaceOnUse">
                        <stop stop-color="#A2A2A2" stop-opacity="0.6"></stop>
                        <stop offset="1" stop-color="#A2A2A2" stop-opacity="0.6"></stop>
                    </linearGradient>
                </defs>
            </svg>
            <span>1ère année</span>
      </div>
    </xsl:template>
    
    <xsl:template name="program-nav-item-badge">
        <xsl:choose>
            <xsl:when test="local-name() = 'subprogram'">
                <svg xmlns="http://www.w3.org/2000/svg" width="97" height="38" viewBox="0 0 97 38" fill="none">
                    <path class="bg" d="M54.4248 0.571289C69.8412 -1.07865 84.5301 0.92416 96.248 5.55469V34C96.248 36.2091 94.4572 38 92.248 38H0C1.5904 19.9087 24.2542 3.80041 54.4248 0.571289Z" fill="#521C7E"/>
                    <path class="border" d="M54.4248 0.571289C69.8412 -1.07865 84.5301 0.92416 96.248 5.55469V34C96.248 36.2091 94.4572 38 92.248 38H0C1.5904 19.9087 24.2542 3.80041 54.4248 0.571289Z" fill="url(#paint0_linear_7313_10085)"/>
                    <defs>
                        <linearGradient id="paint0_linear_7313_10085" x1="0" y1="19" x2="96.248" y2="19" gradientUnits="userSpaceOnUse">
                        <stop stop-color="white" stop-opacity="0.85"/><stop offset="1" stop-color="white" stop-opacity="0.85"/></linearGradient>
                    </defs>
                </svg>
                <span class="ametys-programme-card__badge">
                    <span class="ametys-programme-card__badge-text">
                        <i18n:text i18n:key="PROGRAM_STRUCTURE_NAV_SUBPROGRAM" i18n:catalogue="skin.{$skin}" />
                    </span>
                </span>
            </xsl:when>
            <xsl:otherwise>
                <!-- <span class="ametys-programme-card__badge">
                    <span class="ametys-programme-card__badge-text">
                        <i18n:text i18n:key="PROGRAM_STRUCTURE_NAV_COMMON" i18n:catalogue="skin.{$skin}" />
                    </span>
                </span> -->
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>
    
    <!-- +
         | Template to display the content of subprograms and years.
         | - parentProgramItems: the root program items to be display (by default, the subprograms or year containers of first level)
         + -->
    <xsl:template name="program-structure-content">
        <xsl:param name="parentProgramItems"/>
        
        <!-- All available semesters (period) among the child program items -->
        <xsl:variable name="availableSemesters">
            <xsl:call-template name="get-available-semesters">
                <xsl:with-param name="parentProgramItems" select="$parentProgramItems"/>
            </xsl:call-template>
        </xsl:variable>
        
        <xsl:variable name="currentId" select="/view/content/@id"/>
        
        <xsl:for-each select="$parentProgramItems">
            <div id="tabpanel-program-{@code}-{generate-id()}" class="ametys-programme-content" role="tabpanel" aria-labelledby="tabpanel-program-title-{@code}-{generate-id()}">
                <xsl:if test="position() != 1"><xsl:attribute name="hidden">true</xsl:attribute></xsl:if>  
                
                <a href="#programme-cards" class="sr-only sr-only-focusable" id="back-to-cards">Retour à la liste des programmes</a>
                                    
                <!-- Program item's presentation -->
                <xsl:apply-templates select="." mode="program-item-presentation"/>
                
                <div class="ametys-programme-tabs">
                    <xsl:if test="not(.//subprogram) and exslt:node-set($availableSemesters)/semesters/semester">
                        <!-- Navigation in semesters (table header) -->
                        <xsl:call-template name="semesters-tab-nav">
                            <xsl:with-param name="availableSemesters" select="exslt:node-set($availableSemesters)"/>
                            <xsl:with-param name="currentSemesters" select=".//container[attributes/nature/@code='semestre']"/>
                        </xsl:call-template>
                    </xsl:if>
                    
                    <xsl:if test="not(.//subprogram)">
                        <xsl:choose>
                            <xsl:when test="exslt:node-set($availableSemesters)/semesters/semester">
                                <!-- Table content -->
                                <xsl:call-template name="semesters-tab-content">
                                    <xsl:with-param name="currentSemesters" select=".//container[attributes/nature/@code='semestre']"/>
                                    <xsl:with-param name="parentProgramItem" select="."/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test=".//courselist[course]">
                                <xsl:variable name="parentCourseId">
                                    <xsl:choose>
                                        <xsl:when test="local-name() = 'subprogram'"><xsl:value-of select="concat(';', @id, ';', /view/content/@id)"/></xsl:when>
                                        <xsl:otherwise><xsl:value-of select="concat(';', /view/content/@id)"/></xsl:otherwise>
                                    </xsl:choose>
                                </xsl:variable>
                                <!-- If there is no semester, display the courses -->
                                <ul id="accordion-{@code}" class="ametys-accordion accordion mb-0">
                                    <xsl:apply-templates select=".//courselist[not(ancestor::courselist)]" mode="course-row">
                                        <xsl:with-param name="parentCourseId" select="$parentCourseId"/>
                                    </xsl:apply-templates>
                                </ul>
                            </xsl:when>
                        </xsl:choose>
                    </xsl:if>
                </div>
            </div>                               
        </xsl:for-each>                    
    </xsl:template>
    
    <xsl:template name="semesters-tab-bg-url">
        <xsl:param name="programItem"/>
        
        <xsl:variable name="programItemImgUri">
            <xsl:if test="local-name($programItem) = 'subprogram'">
                <xsl:call-template name="get-image-uri-from-attribute">
                    <xsl:with-param name="id" select="$programItem/@id"/>
                    <xsl:with-param name="height" select="1000"/>
                    <xsl:with-param name="width" select="1000"/>
                </xsl:call-template>
            </xsl:if>
        </xsl:variable>
        
        <xsl:choose>
            <xsl:when test="$programItemImgUri != ''"><xsl:value-of select="$programItemImgUri"/></xsl:when>
            <xsl:otherwise><xsl:call-template name="semesters-tab-bg-default-url"/></xsl:otherwise>
        </xsl:choose>
    </xsl:template>
    
    <xsl:template name="semesters-tab-bg-default-url"/>
    
    <!-- +
         | Template to display of short overview of the subprogram or year
         + -->
    <xsl:template match="subprogram|container" mode="program-item-presentation">
    
        <xsl:apply-templates select="." mode="program-item-title"/>
        
        <div class="ametys-richtext">
            <xsl:apply-templates select="." mode="program-item-text"/>
                
            <xsl:apply-templates select="." mode="program-item-see-more"/>
        </div>
    </xsl:template>
    
    <xsl:template match="subprogram|container" mode="program-item-title">
        <h3 id="tabpanel-program-title-{@code}-{generate-id()}" class="ametys-programme-content__title"><xsl:value-of select="@title"/></h3>
    </xsl:template>
    
    <xsl:template match="container" mode="program-item-text">
        <xsl:if test="ametys:hasValue(@id, 'description')">
            <xsl:variable name="description" select="ametys:contentAttribute(@id, 'description')"/>
            <xsl:apply-templates select="$description/docbook:article">
                <xsl:with-param name="level" select="2"/>
            </xsl:apply-templates>
        </xsl:if>
    </xsl:template>
    
    <xsl:template match="subprogram" mode="program-item-text">
        <xsl:if test="ametys:hasValue(@id, 'presentation')">
            <xsl:variable name="presentation" select="ametys:contentAttribute(@id, 'presentation')"/>
            <xsl:apply-templates select="$presentation/docbook:article">
                <xsl:with-param name="level" select="2"/>
            </xsl:apply-templates>
        </xsl:if>
    </xsl:template>
    
    <xsl:template match="container" mode="program-item-see-more"/>
    
    <xsl:template match="subprogram" mode="program-item-see-more">
        <p>
            <a class="internal" href="{resolver:resolve('odf', concat(@id, ';', /view/content/@id))}"><i18n:text i18n:key="PROGRAM_SUBPROGRAM_SEE_FULL_PAGE" i18n:catalogue="skin.{$skin}"/></a>
        </p>
    </xsl:template>

    <!-- + 
         | Template to navigate on semesters. Parameters are:
         | - availableSemesters : the ordered and available semesters to use for table header navigation
         | - currentSemesters : the semesters belonging to the current year or subprogram
         + -->
    <xsl:template name="semesters-tab-nav">
        <xsl:param name="availableSemesters"/>
        <xsl:param name="currentSemesters"/>
        
        <div class="ametys-programme-tabs__nav" role="tablist">
            <xsl:for-each select="$availableSemesters/semesters/semester">
                <button class="ametys-programme-tab"
                    role="tab"
                    id="tab-semester-{@code}-{generate-id()}"
                    aria-controls="semester-{@code}-{generate-id()}"
                    aria-selected="false"
                    tabindex="0">
                    
                    <xsl:if test="$currentSemesters[attributes/period/@code = current()/@code]/@code">
                        <xsl:attribute name="aria-controls">semester-<xsl:value-of select="$currentSemesters[attributes/period/@code = current()/@code]/@code"/>-<xsl:value-of select="generate-id($currentSemesters[attributes/period/@code = current()/@code])"/></xsl:attribute>
                    </xsl:if>
                    
                    <xsl:choose>
                        <xsl:when test="$currentSemesters[1][attributes/period/@code = current()/@code]/@code">
                            <xsl:attribute name="class">ametys-programme-tab active</xsl:attribute>
                            <xsl:attribute name="aria-selected">true</xsl:attribute>
                        </xsl:when>
                        <xsl:when test="$currentSemesters[attributes/period/@code = current()/@code]/@code">
                            <xsl:attribute name="class">ametys-programme-tab</xsl:attribute>
                        </xsl:when>
                        <xsl:otherwise>
                            <xsl:attribute name="disabled">true</xsl:attribute>
                            <xsl:attribute name="tabindex">-1</xsl:attribute>
                        </xsl:otherwise>
                    </xsl:choose>
                    
                    <xsl:value-of select="@title"/>
                </button>
            </xsl:for-each>
        </div>
    </xsl:template>
    
    <!-- +
         | Semester table content. Parameters are:
         | - currentSemesters : the semesters belonging to the current year or subprogram
         + -->
    <xsl:template name="semesters-tab-content">
        <xsl:param name="currentSemesters"/>
        <xsl:param name="parentProgramItem"/>
        
        <xsl:if test="$currentSemesters">
	        <div class="ametys-programme-tabs__content">
	            <xsl:for-each select="$currentSemesters">
		            <div id="semester-{@code}-{generate-id()}" class="ametys-programme-panel" role="tabpanel">
                        <!-- Active the first semester -->
                        <xsl:choose>
                            <xsl:when test="position() = 1"><xsl:attribute name="class">ametys-programme-panel active</xsl:attribute></xsl:when>
                            <xsl:otherwise><xsl:attribute name="hidden">true</xsl:attribute></xsl:otherwise>
                        </xsl:choose>
                        
                        <!-- Child courses -->
                        <xsl:if test=".//courselist[course]">
                            <div class="ametys-programme-table">
                                <!-- Table header -->
                                <xsl:call-template name="course-table-header"/>
                                
                                <!-- Table body -->
                                <ul class="ametys-programme-table__body">
                                    <xsl:variable name="parentCourseId">
                                        <xsl:choose>
                                            <xsl:when test="local-name($parentProgramItem) = 'subprogram'"><xsl:value-of select="concat(';', $parentProgramItem/@id, ';', /view/content/@id)"/></xsl:when>
                                            <xsl:otherwise><xsl:value-of select="concat(';', /view/content/@id)"/></xsl:otherwise>
                                        </xsl:choose>
                                    </xsl:variable>
                                    <!-- Iterate on first level courses -->
                                    <xsl:apply-templates select=".//courselist[not(ancestor::courselist)]" mode="first-level">
                                        <xsl:with-param name="parentCourseId" select="$parentCourseId"/>
                                    </xsl:apply-templates>
                                </ul>
                            </div>
                        </xsl:if>
		            </div>
	            </xsl:for-each>
	        </div>
        </xsl:if>
    </xsl:template>
    
    <xsl:template name="course-table-header">
        <div class="ametys-programme-table__header">
          <div class="header-col header-ue">UE</div>
          <div class="header-col header-title"></div>
          <div class="header-col header-credits">Crédits</div>
          <div class="header-col header-cm">CM</div>
          <div class="header-col header-td">TD</div>
          <div class="header-col header-tp">TP</div>
        </div>
    </xsl:template>
    
    <xsl:template match="courselist[attributes/choiceType/@value='OPTIONAL' or attributes/choiceType/@value='CHOICE']" mode="first-level" priority="100">
        <xsl:param name="parentCourseId"/>
        
        <li class="ametys-programme-section">
            <xsl:apply-templates select="." mode="courselist-row-title"/>
                
            <ul class="ametys-programme-item__children">
                <xsl:apply-templates select="course" mode="course-row">
                    <xsl:with-param name="parentCourseId" select="$parentCourseId"/>
                </xsl:apply-templates>
            </ul>
        </li>
    </xsl:template>
    
    <xsl:template match="courselist" mode="first-level">
        <xsl:param name="parentCourseId"/>
        
        <xsl:apply-templates select="course" mode="course-row">
            <xsl:with-param name="parentCourseId" select="$parentCourseId"/>
        </xsl:apply-templates>
    </xsl:template>
    
    <xsl:template match="courselist[attributes/choiceType/@value='OPTIONAL' or attributes/choiceType/@value='CHOICE']" mode="course-row" priority="100">
        <xsl:param name="parentCourseId"/>
        
        <ul class="ametys-programme-item__children" id="courselist-{@code}-{generate-id()}">
            <li class="ametys-programme-section">
                <xsl:apply-templates select="." mode="courselist-row-title"/>
                    
                <ul class="ametys-programme-item__children">
                    <xsl:apply-templates select="course" mode="course-row">
                        <xsl:with-param name="parentCourseId" select="$parentCourseId"/>
                    </xsl:apply-templates>
                </ul>
            </li>
        </ul>
    </xsl:template>
    
    <xsl:template match="courselist" mode="course-row">
        <xsl:param name="parentCourseId"/>
        
         <xsl:if test="course">
            <ul class="ametys-programme-item__children" id="courselist-{@code}-{generate-id()}">
                <xsl:apply-templates select="course" mode="course-row">
                    <xsl:with-param name="parentCourseId" select="$parentCourseId"/>
                </xsl:apply-templates>
            </ul>
        </xsl:if>
    </xsl:template>
    
    <xsl:template match="courselist[attributes/choiceType/@value='CHOICE']" mode="courselist-row-title">
        <p class="section-label">
            <xsl:variable name="i18nKey">COURSE_LIST_CHOICE<xsl:if test="attributes/max > 0 and attributes/min != attributes/max">_WITH_MAX</xsl:if></xsl:variable>
            
            <i18n:translate>
                <i18n:text i18n:key="{$i18nKey}" i18n:catalogue="skin.{$skin}"/>
                <i18n:param name="min"><xsl:value-of select="attributes/min"/></i18n:param>
                <i18n:param name="max"><xsl:value-of select="attributes/max"/></i18n:param>
                <i18n:param name="total"><xsl:value-of select="count(course)"/></i18n:param>
            </i18n:translate>
        </p>
    </xsl:template>
    
    <xsl:template match="courselist[attributes/choiceType/@value='OPTIONAL']" mode="courselist-row-title">
        <p class="section-label">
            <i18n:text i18n:key="COURSE_LIST_OPTIONAL" i18n:catalogue="skin.{$skin}"/>
        </p>
    </xsl:template>
    
    <!-- +
         | Display a course with its titles, ects and hours.
         + -->
    <xsl:template match="course" mode="course-row">
        <xsl:param name="parentCourseId" select="concat(';', /view/content/@id)"/>
        
        <li class="ametys-programme-item">
            <xsl:if test="not(courselist/course)">
                <xsl:attribute name="class">ametys-programme-item no-children</xsl:attribute>
            </xsl:if>
                
            <xsl:apply-templates select="." mode="course-row-header">
                <xsl:with-param name="parentCourseId" select="$parentCourseId"/>
            </xsl:apply-templates>
            
            <!-- Child courses -->
            <xsl:apply-templates select="courselist[course]" mode="course-row">
                <xsl:with-param name="parentCourseId" select="concat (';', @id, $parentCourseId)"/>
            </xsl:apply-templates>
        </li>
    </xsl:template>
    
    <xsl:template match="course" mode="course-row-header">
        <xsl:param name="parentCourseId"/>
        
        <div class="ametys-programme-item__header">
                
            <xsl:if test="not(ancestor::course)">
                <!-- For courses of first level only -->
            </xsl:if>
            
            <button class="item-toggle" aria-label="{@title}" aria-expanded="false">
                <xsl:choose>
                    <xsl:when test="not(courselist/course)">
                        <xsl:attribute name="disabled">true</xsl:attribute>
                        <xsl:attribute name="aria-hidden">true</xsl:attribute>
                    </xsl:when>
                    <xsl:otherwise>
                        <xsl:attribute name="aria-controls">courselist-<xsl:value-of select="courselist/@code"/>-<xsl:value-of select="generate-id(courselist)"/></xsl:attribute>
                    </xsl:otherwise>
                </xsl:choose>
            </button>
            
            <!-- Course's title -->
            <xsl:apply-templates select="." mode="course-row-title">
                <xsl:with-param name="parentCourseId" select="$parentCourseId"/>
            </xsl:apply-templates>
            <xsl:apply-templates select="." mode="course-row-ects"/><!-- Credits --> 
            
            <xsl:apply-templates select="." mode="coursepart-hours">
                <xsl:with-param name="natureCode" select="'CM'"></xsl:with-param>
            </xsl:apply-templates>
            <xsl:apply-templates select="." mode="coursepart-hours">
                <xsl:with-param name="natureCode" select="'TD'"></xsl:with-param>
            </xsl:apply-templates>
            <xsl:apply-templates select="." mode="coursepart-hours">
                <xsl:with-param name="natureCode" select="'TP'"></xsl:with-param>
            </xsl:apply-templates>
        </div>
    </xsl:template>
    
    <xsl:template match="course" mode="course-row-title">
        <xsl:param name="parentCourseId"/>
        
        <a class="item-title" href="{resolver:resolve('odf', concat(@id, $parentCourseId))}">
            <xsl:value-of select="@title"/>
        </a>
    </xsl:template>
    
    <xsl:template match="course" mode="course-row-ects">
        <span class="item-credits">
            <xsl:variable name="computedEcts" select="odf:getEcts(@id, @path)"/>
            <xsl:choose>
                <xsl:when test="$computedEcts">
                    <xsl:variable name="ects"><xsl:call-template name="transform-double"><xsl:with-param name="value" select="$computedEcts"/></xsl:call-template></xsl:variable>
                    <xsl:value-of select="$ects"/> <i18n:text i18n:key="PROGRAM_ECTS_UNITS" i18n:catalogue="skin.{$skin}"/>
                </xsl:when>
                <xsl:otherwise><xsl:call-template name="program-tab-empty-value"/></xsl:otherwise>
            </xsl:choose>
        </span>
    </xsl:template>
    
    <xsl:template match="course" mode="course-row-hours">
        <span class="item-cm">
		    <xsl:if test="attributes/nbHours">
		        <xsl:variable name="hours"><xsl:call-template name="transform-double"><xsl:with-param name="value" select="attributes/nbHours"/></xsl:call-template></xsl:variable>
		        <xsl:value-of select="$hours"/>h
		    </xsl:if>
        </span>
    </xsl:template>
    
    <xsl:template match="course" mode="coursepart-hours">
        <xsl:param name="natureCode"/>
        
        <xsl:variable name="nbHours"><xsl:value-of select="sum(courseparts/coursepart[attributes/nature/@code = $natureCode]/attributes/nbHours)"/></xsl:variable>
        <span class="item-cm">
            <xsl:choose>
                <xsl:when test="$nbHours != '' and $nbHours != '0' and $nbHours != '0.0'">
                    <xsl:variable name="hours"><xsl:call-template name="transform-double"><xsl:with-param name="value" select="$nbHours"/></xsl:call-template></xsl:variable>
                    <xsl:value-of select="$hours"/>h
                </xsl:when>
                <xsl:otherwise><xsl:call-template name="program-tab-empty-value"/></xsl:otherwise>
            </xsl:choose>
        </span>
    </xsl:template>
    
    <xsl:template name="program-tab-empty-value">-</xsl:template>
    
</xsl:stylesheet>