<?xml version="1.0" encoding="UTF-8"?>
<!--
   Copyright 2019 Anyware Services

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
   -->
<xsl:stylesheet version="1.0"
        xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
        xmlns:i18n="http://apache.org/cocoon/i18n/2.1"
        xmlns:sitemap="http://www.ametys.org/inputdata/sitemap/3.0"
        xmlns:xalan="http://xml.apache.org/xalan"
        xmlns:ametys="org.ametys.web.transformation.xslt.AmetysXSLTHelper"
        xmlns:resolver="org.ametys.cms.transformation.xslt.ResolveURIComponent"
        xmlns:odf="org.ametys.odf.OdfXSLTHelper"
        xmlns:escaper="org.apache.commons.lang.StringEscapeUtils"
        xmlns:docbook="http://docbook.org/ns/docbook" 
        xmlns:exslt="http://exslt.org/common"
        exclude-result-prefixes="sitemap ametys resolver odf escaper docbook exslt">

    <xsl:import href="skin://stylesheets/content/common/course-popup.xsl"/>
    
    <xsl:variable name="usePopupForCourses" select="true()"/>
    
    <!-- Ordered semesters -->
    <xsl:variable name="periodItems" select="odf:getTableRefItems('odf-enumeration.Period', /view/content/@language, true())"/>
    
    <!-- +
         | Get the available semesters (period) among the child program items.
         + -->
    <xsl:template name="get-available-semesters">
        <xsl:param name="parentProgramItems"/>
        
        <semesters>
            <xsl:for-each select="$periodItems/item">
                <xsl:if test="$parentProgramItems//container[attributes/nature/@code='semestre' and attributes/period/@code= current()/@code]">
                    <semester>
                        <xsl:copy-of select="@*"/>
                    </semester>
                </xsl:if>
            </xsl:for-each>
        </semesters>
    </xsl:template>
    
    <!-- + 
         | Main template to expose the program structure. Parameters are:
         | - title: the title of this section (optional)
         | - currentProgramItem: the current program item that contains subprograms and/or years 
         + -->
    <xsl:template name="program-structure">
        <xsl:param name="title"/>
        <xsl:param name="currentProgramItem"/>
        
        <xsl:variable name="subprogramsAndYears" select="$currentProgramItem/subprogram|$currentProgramItem/container[attributes/nature/@code='annee']"/>
        
        <xsl:choose>
            <xsl:when test="$subprogramsAndYears">
                <!-- Program structure with subprograms and/or years -->
                <xsl:if test="$title">
                    <h3 class="ametys-richtext-title-5"><xsl:copy-of select="$title"/></h3>
                </xsl:if>
                
                <!-- Navigation on child program items -->
                <xsl:call-template name="program-structure-nav">
                    <xsl:with-param name="parentProgramItems" select="$subprogramsAndYears"/>
                </xsl:call-template>
                
                <xsl:call-template name="program-structure-content">
                    <xsl:with-param name="parentProgramItems" select="$subprogramsAndYears"/>
                </xsl:call-template>
            </xsl:when>
            <xsl:when test="$currentProgramItem/container[attributes/nature/@code='semestre']">
                <!-- There is no subprograms or years, but semesters under the root program item -->
                <xsl:variable name="directSemesters" select="$currentProgramItem/container[attributes/nature/@code='semestre']"/>
                
                <xsl:variable name="availableSemesters">
                    <xsl:call-template name="get-available-semesters">
                        <xsl:with-param name="parentProgramItems" select="$currentProgramItem"/>
                    </xsl:call-template>
                </xsl:variable>
        
                <xsl:call-template name="semesters-tab-nav">
                    <xsl:with-param name="availableSemesters" select="exslt:node-set($availableSemesters)"/>
                    <xsl:with-param name="currentSemesters" select="$directSemesters"/>
                </xsl:call-template>
        
                <xsl:call-template name="semesters-tab-content">
                    <xsl:with-param name="currentSemesters" select="$directSemesters"/>
                    <xsl:with-param name="parentProgramItem" select="$currentProgramItem"/>
                </xsl:call-template>
            </xsl:when>
            <xsl:when test="$currentProgramItem/courselist[course]">
                <!-- There is no subprograms nor years nor semesters but courses under the root program item -->
                <ul id="accordion-{@code}" class="ametys-accordion accordion mb-0">
                    <xsl:apply-templates select="$currentProgramItem/courselist" mode="course-row"/>
                </ul>
            </xsl:when>
        </xsl:choose>
        
        <!-- Popup for courses -->
        <xsl:if test="$usePopupForCourses">
            <xsl:call-template name="course-popups">
                <xsl:with-param name="courses" select="$currentProgramItem//course"></xsl:with-param>
            </xsl:call-template>
        </xsl:if>
                
    </xsl:template>
    
    <!-- + 
         | Template to navigate on subprograms and years. Parameters are:
         | - parentProgramItems: the root program items to be display (by default, the subprograms or year containers of first level)
         + -->
    <xsl:template name="program-structure-nav">
        <xsl:param name="parentProgramItems"/>
        
        <div id="programmeBadges" class="ametys-checkbox-badged-wrap ametys-checkbox-badged-wrap--lg-v2 nav g-mb-30" role="tablist">
            
	        <xsl:for-each select="$parentProgramItems">
                <xsl:variable name="activeCls"><xsl:if test="position() = 1"> active</xsl:if></xsl:variable>
                
	            <div class="ametys-checkbox-badge ametys-checkbox-badge--alt ametys-checkbox-badge--rounded ametys-checkbox-badge--lg-v2{$activeCls}" 
                     role="tab" 
	                 data-target="#tab-content-{@code}-{generate-id()}"
	                 data-toggle="tab">
                     <xsl:if test="position() = 1"><xsl:attribute name="aria-selected">true</xsl:attribute></xsl:if>
                    
	                <input id="programFilterRadio{@code}-{generate-id()}" class="ametys-checkbox-badge__control" name="programFilterRadioGroup" type="radio" value="{@id}">
                        <xsl:if test="position() = 1"><xsl:attribute name="checked">checked</xsl:attribute></xsl:if>
                    </input>
                    
	                <label class="ametys-checkbox-badge__label" for="programFilterRadio{@code}-{generate-id()}">
	                    <span class="ametys-checkbox-badge__label-text"><xsl:value-of select="@title"/></span>
                        <xsl:if test="local-name() = 'subprogram'">
	                       <span class="ametys-checkbox-badge__label-indicator">
                                <span class="ametys-checkbox-badge__label-indicator-text"><i18n:text i18n:key="PROGRAM_STRUCTURE_NAV_SUBPROGRAM" i18n:catalogue="skin.{$skin}"/></span>
                           </span>
                        </xsl:if>
	                </label>
	            </div>
	        </xsl:for-each>                                   
        </div>
        
        <script>
            $j('#programmeBadges [data-toggle="tab"]').on('show.bs.tab', function (e) {
                
                var $this = $j(this),
                    $radio = $this.find('.ametys-checkbox-badge__control');
                
                $radio.prop('checked', true);
            });
            
            $j.HSCore.components.HSScrollTo.init('#programmeBadges .ametys-checkbox-badge', {
                header: '#pageNav',
                target: '#programmeContent',
                mobileOnly: true
            });
        </script>
    </xsl:template>  
    
    <!-- +
         | Template to display the content of subprograms and years.
         | - parentProgramItems: the root program items to be display (by default, the subprograms or year containers of first level)
         + -->
    <xsl:template name="program-structure-content">
        <xsl:param name="parentProgramItems"/>
        
        <!-- All available semesters (period) among the child program items -->
        <xsl:variable name="availableSemesters">
            <xsl:call-template name="get-available-semesters">
                <xsl:with-param name="parentProgramItems" select="$parentProgramItems"/>
            </xsl:call-template>
        </xsl:variable>
        
        <div id="programmeContent" class="tab-content">
            <xsl:variable name="currentId" select="/view/content/@id"/>
            
            <xsl:for-each select="$parentProgramItems">
                <div id="tab-content-{@code}-{generate-id()}" class="tab-pane" role="tabpanel">
                    <xsl:if test="position() = 1"><xsl:attribute name="class">tab-pane active</xsl:attribute></xsl:if>     
                    
                    <xsl:variable name="bgImgUrl">
                        <xsl:call-template name="semesters-tab-bg-url">
                            <xsl:with-param name="programItem" select="."/>
                        </xsl:call-template>
                    </xsl:variable>
                    
                    <div>
                        <!-- Program item's presentation -->
                        <xsl:choose>
                            <xsl:when test="$bgImgUrl != ''">
                                <xsl:attribute name="class">ametys-section-with-bg-image ametys-section-with-blur</xsl:attribute>
                                
                                <div class="ametys-section-with-bg-image__helper ametys-section-with-bg-image__helper--transparent" style="background-image: url({ametys:escapeURLforCSS($bgImgUrl)});"></div>
                                <div class="ametys-section-with-blur__helper ametys-section-with-blur__helper--gray"></div>
                                
                                <xsl:apply-templates select="." mode="program-item-presentation-with-bg-img"/>
                            </xsl:when>
                            <xsl:otherwise>
                                <xsl:apply-templates select="." mode="program-item-presentation"/>
                            </xsl:otherwise>
                        </xsl:choose>
                        
                        <xsl:if test="not(.//subprogram) and exslt:node-set($availableSemesters)/semesters/semester">
                            <!-- Navigation in semesters (table header) -->
                            <xsl:call-template name="semesters-tab-nav">
                                <xsl:with-param name="availableSemesters" select="exslt:node-set($availableSemesters)"/>
                                <xsl:with-param name="currentSemesters" select=".//container[attributes/nature/@code='semestre']"/>
                                <xsl:with-param name="withBgImg" select="$bgImgUrl != ''"/>
                            </xsl:call-template>
                        </xsl:if>
                    </div>
                    
                    <xsl:if test="not(.//subprogram)">
                        <xsl:choose>
                            <xsl:when test="exslt:node-set($availableSemesters)/semesters/semester">
                                <!-- Table content -->
                                <xsl:call-template name="semesters-tab-content">
                                    <xsl:with-param name="currentSemesters" select=".//container[attributes/nature/@code='semestre']"/>
                                    <xsl:with-param name="parentProgramItem" select="."/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test=".//courselist[course]">
                                <xsl:variable name="parentCourseId">
                                    <xsl:choose>
                                        <xsl:when test="local-name() = 'subprogram'"><xsl:value-of select="concat(';', @id, ';', /view/content/@id)"/></xsl:when>
                                        <xsl:otherwise><xsl:value-of select="concat(';', /view/content/@id)"/></xsl:otherwise>
                                    </xsl:choose>
                                </xsl:variable>
                                <!-- If there is no semester, display the courses -->
                                <ul id="accordion-{@code}" class="ametys-accordion accordion mb-0">
                                    <xsl:apply-templates select=".//courselist[not(ancestor::courselist)]" mode="course-row">
                                        <xsl:with-param name="parentCourseId" select="$parentCourseId"/>
                                    </xsl:apply-templates>
                                </ul>
                            </xsl:when>
                        </xsl:choose>
                    </xsl:if>
                </div>                               
            </xsl:for-each>                    
        </div>
    </xsl:template>
    
    <xsl:template name="semesters-tab-bg-url">
        <xsl:param name="programItem"/>
        
        <xsl:variable name="programItemImgUri">
            <xsl:if test="local-name($programItem) = 'subprogram'">
                <xsl:call-template name="get-image-uri-from-attribute">
                    <xsl:with-param name="id" select="$programItem/@id"/>
                    <xsl:with-param name="height" select="1000"/>
                    <xsl:with-param name="width" select="1000"/>
                </xsl:call-template>
            </xsl:if>
        </xsl:variable>
        
        <xsl:choose>
            <xsl:when test="$programItemImgUri != ''"><xsl:value-of select="$programItemImgUri"/></xsl:when>
            <xsl:otherwise><xsl:call-template name="semesters-tab-bg-default-url"/></xsl:otherwise>
        </xsl:choose>
    </xsl:template>
    
    <xsl:template name="semesters-tab-bg-default-url"/>
    
    <!-- +
         | Template to display of short overview of the subprogram or year
         + -->
    <xsl:template match="subprogram|container" mode="program-item-presentation">
        <div class="ametys-custom-typo-v1 g-mb-30">
            <div class="ametys-richtext">
                <xsl:apply-templates select="." mode="program-item-title"/>
                
                <xsl:apply-templates select="." mode="program-item-text"/>
                
                <xsl:apply-templates select="." mode="program-item-see-more"/>
            </div>
        </div>
    </xsl:template>
    
    <xsl:template match="subprogram|container" mode="program-item-presentation-with-bg-img">
        <div class="ametys-custom-typo-v1 position-relative p-4">
            <div class="ametys-richtext">
                <xsl:apply-templates select="." mode="program-item-title"/>
                
                <xsl:apply-templates select="." mode="program-item-text"/>
                
                <xsl:apply-templates select="." mode="program-item-see-more"/>
            </div>
        </div>
    </xsl:template>
    
    <xsl:template match="subprogram|container" mode="program-item-title">
        <h3 class="ametys-richtext-title-4"><xsl:value-of select="@title"/></h3>
    </xsl:template>
    
    <xsl:template match="container" mode="program-item-text">
        <xsl:if test="ametys:hasValue(@id, 'description')">
            <xsl:variable name="description" select="ametys:contentAttribute(@id, 'description')"/>
            <xsl:apply-templates select="$description/docbook:article">
                <xsl:with-param name="level" select="2"/>
            </xsl:apply-templates>
        </xsl:if>
    </xsl:template>
    
    <xsl:template match="subprogram" mode="program-item-text">
        <xsl:if test="ametys:hasValue(@id, 'presentation')">
            <xsl:variable name="presentation" select="ametys:contentAttribute(@id, 'presentation')"/>
            <xsl:apply-templates select="$presentation/docbook:article">
                <xsl:with-param name="level" select="2"/>
            </xsl:apply-templates>
        </xsl:if>
    </xsl:template>
    
    <xsl:template match="container" mode="program-item-see-more"/>
    
    <xsl:template match="subprogram" mode="program-item-see-more">
        <p>
            <a class="know-more" href="{resolver:resolve('odf', concat(@id, ';', /view/content/@id))}"><i18n:text i18n:key="PROGRAM_SUBPROGRAM_SEE_FULL_PAGE" i18n:catalogue="skin.{$skin}"/></a>
        </p>
    </xsl:template>

    <!-- + 
         | Template to navigate on semesters. Parameters are:
         | - availableSemesters : the ordered and available semesters to use for table header navigation
         | - currentSemesters : the semesters belonging to the current year or subprogram
         + -->
    <xsl:template name="semesters-tab-nav">
        <xsl:param name="availableSemesters"/>
        <xsl:param name="currentSemesters"/>
        <xsl:param name="withBgImg" select="false()"/>
        
        <nav>
            <xsl:if test="$withBgImg"><xsl:attribute name="class">position-relative px-2</xsl:attribute></xsl:if>
            
            <ul class="ametys-tabs-controls ametys-tabs-controls--justified nav" role="tablist">
                <xsl:for-each select="$availableSemesters/semesters/semester">
                    <li class="ametys-tabs-controls__item">
	                    <a href="#" role="tab" aria-controls="tab1" aria-selected="true" data-toggle="tab">
                            <xsl:if test="$currentSemesters[attributes/period/@code = current()/@code]/@code">
                                <xsl:attribute name="href">#semester-<xsl:value-of select="$currentSemesters[attributes/period/@code = current()/@code]/@code"/>-<xsl:value-of select="generate-id($currentSemesters[attributes/period/@code = current()/@code])"/></xsl:attribute>
                            </xsl:if>
                            <xsl:attribute name="aria-selected">
                                <xsl:choose>
                                    <xsl:when test="$currentSemesters[1][attributes/period/@code = current()/@code]">true</xsl:when>
                                    <xsl:when test="$currentSemesters[attributes/period/@code = current()/@code]">false</xsl:when>
                                    <xsl:otherwise>false</xsl:otherwise>
                                </xsl:choose>
                            </xsl:attribute>
                            <xsl:attribute name="class">
	                            <xsl:choose>
                                    <xsl:when test="$currentSemesters[1][attributes/period/@code = current()/@code]">ametys-tabs-controls__link active</xsl:when>
                                    <xsl:when test="$currentSemesters[attributes/period/@code = current()/@code]">ametys-tabs-controls__link</xsl:when>
                                    <xsl:otherwise>ametys-tabs-controls__link disabled</xsl:otherwise>
                                </xsl:choose>
                            </xsl:attribute>
                            <xsl:value-of select="@title"/>
                        </a>
	                </li>
                </xsl:for-each>
            </ul>
        </nav>
    </xsl:template>
    
    <!-- +
         | Semester table content. Parameters are:
         | - currentSemesters : the semesters belonging to the current year or subprogram
         + -->
    <xsl:template name="semesters-tab-content">
        <xsl:param name="currentSemesters"/>
        <xsl:param name="parentProgramItem"/>
        
        <xsl:if test="$currentSemesters">
	        <div class="tab-content">
	            <xsl:for-each select="$currentSemesters">
		            <div id="semester-{@code}-{generate-id()}" class="tab-pane" role="tabpanel">
                        <!-- Active the first semester -->
	                    <xsl:if test="position() = 1"><xsl:attribute name="class">tab-pane active</xsl:attribute></xsl:if>
                        
                        <!-- Child courses -->
                        <xsl:if test=".//courselist[course]">
                            <xsl:variable name="parentCourseId">
                                <xsl:choose>
                                    <xsl:when test="$parentProgramItem/@id != /view/content/@id and local-name($parentProgramItem) = 'subprogram'"><xsl:value-of select="concat(';', $parentProgramItem/@id, ';', /view/content/@id)"/></xsl:when>
                                    <xsl:otherwise><xsl:value-of select="concat(';', /view/content/@id)"/></xsl:otherwise>
                                </xsl:choose>
                            </xsl:variable>
                            <ul id="accordion-{@code}" class="ametys-accordion accordion mb-0">
                                <!-- Iterate on first level courses -->
                                <xsl:apply-templates select=".//courselist[not(ancestor::courselist)]" mode="course-row">
                                    <xsl:with-param name="parentCourseId" select="$parentCourseId"/>
                                </xsl:apply-templates>
                            </ul>
                        </xsl:if>
		            </div>
	            </xsl:for-each>
	        </div>
        </xsl:if>
    </xsl:template>
    
    <xsl:template name="course-popups">
        <xsl:param name="courses"/>
        
        <xsl:for-each select="$courses">
            <xsl:apply-templates select="." mode="course-popup"/>
        </xsl:for-each>
    </xsl:template>
    
    <xsl:template match="courselist[attributes/choiceType/@value='OPTIONAL' or attributes/choiceType/@value='CHOICE']" mode="course-row" priority="100">
        <xsl:param name="parentCourseId"/>
        
        <li class="ametys-accordion__item ametys-accordion__item--container">
            <header class="ametys-accordion__item-header">
                <xsl:apply-templates select="." mode="courselist-row-title"/>
                <span class="ametys-accordion__item-text"></span>
                <span class="ametys-accordion__item-badge"></span>
            </header>
                
            <ul class="ametys-accordion">
                <xsl:apply-templates select="course" mode="course-row">
                    <xsl:with-param name="parentCourseId" select="$parentCourseId"/>
                </xsl:apply-templates>
            </ul>
        </li>
    </xsl:template>
    
    <xsl:template match="courselist" mode="course-row">
        <xsl:param name="parentCourseId"/>
        
        <xsl:apply-templates select="course" mode="course-row">
            <xsl:with-param name="parentCourseId" select="$parentCourseId"/>
        </xsl:apply-templates>
    </xsl:template>
    
    <xsl:template match="courselist[attributes/choiceType/@value='CHOICE']" mode="courselist-row-title">
        <h2 class="ametys-accordion__item-title">
            <span class="ametys-accordion__item-instructions">
                <xsl:variable name="i18nKey">COURSE_LIST_CHOICE<xsl:if test="attributes/max > 0 and attributes/min != attributes/max">_WITH_MAX</xsl:if></xsl:variable>
                
                <i18n:translate>
                    <i18n:text i18n:key="{$i18nKey}" i18n:catalogue="skin.{$skin}"/>
                    <i18n:param name="min"><xsl:value-of select="attributes/min"/></i18n:param>
                    <i18n:param name="max"><xsl:value-of select="attributes/max"/></i18n:param>
                    <i18n:param name="total"><xsl:value-of select="count(course)"/></i18n:param>
                </i18n:translate>
            </span>
        </h2>
    </xsl:template>
    
    <xsl:template match="courselist[attributes/choiceType/@value='OPTIONAL']" mode="courselist-row-title">
        <h2 class="ametys-accordion__item-title">
            <span class="ametys-accordion__item-instructions">
                <i18n:text i18n:key="COURSE_LIST_OPTIONAL" i18n:catalogue="skin.{$skin}"/>
            </span>
        </h2>
    </xsl:template>
    
    <!-- +
         | Display a course with its titles, ects and hours.
         + -->
    <xsl:template match="course" mode="course-row">
        <xsl:param name="parentCourseId" select="concat(';', /view/content/@id)"/>
        
        <li class="ametys-accordion__item">
	        
            <xsl:apply-templates select="." mode="course-row-header">
                <xsl:with-param name="parentCourseId" select="$parentCourseId"/>
            </xsl:apply-templates>
            
            <!-- Child courses -->
            <xsl:if test="courselist[course]">
                <div id="course-{@code}" class="collapse">
                    <div class="ametys-accordion__item-body">
                        <ul class="ametys-accordion ametys-accordion--inner accordion">
                            <xsl:apply-templates select="courselist[course]" mode="course-row">
                                <xsl:with-param name="parentCourseId" select="concat (';', @id, $parentCourseId)"/>
                            </xsl:apply-templates>
                        </ul>
                    </div>
                </div>
            </xsl:if>
        </li>
    </xsl:template>
    
    <xsl:template match="course" mode="course-row-header">
        <xsl:param name="parentCourseId"/>
        
        <header class="ametys-accordion__item-header">
                
            <xsl:if test="not(ancestor::course)">
                <!-- For courses of first level only -->
                <xsl:attribute name="class">ametys-accordion__item-header ametys-accordion__item-header--top</xsl:attribute>
            </xsl:if>
            
            <xsl:if test="courselist/course">
                <a class="ametys-accordion__item-invoker collapsed" href="#" aria-expanded="false"
                   data-toggle="collapse"
                   data-target="#course-{@code}">
                    <span class="fas fa-angle-down"></span>
                </a>
            </xsl:if>
            
            <!-- Course's title -->
            <xsl:apply-templates select="." mode="course-row-title">
                <xsl:with-param name="parentCourseId" select="$parentCourseId"/>
            </xsl:apply-templates>
            <xsl:apply-templates select="." mode="course-row-ects"/><!-- Credits --> 
            <xsl:apply-templates select="." mode="course-row-hours"/><!-- Hours -->
        </header>
    </xsl:template>
    
    <xsl:template match="course" mode="course-row-title">
        <xsl:param name="parentCourseId"/>
        
        <h2 class="ametys-accordion__item-title">
            <xsl:choose>
                <xsl:when test="$usePopupForCourses">
                    <a id="popup-{@code}-invoker-{generate-id()}" href="javascript:;" data-src="#popup-{@code}-{generate-id()}" data-speed="350">
                        <xsl:value-of select="@title"/>
                    </a>
                    <script type="text/javascript">
                        $j.HSCore.components.HSPopup.init('#popup-<xsl:value-of select="@code"/>-invoker-<xsl:value-of select="generate-id()"/>', {
                            touch: false,
                            autoFocus: false,
                            afterShow: function(instance, current) {
                                <xsl:variable name="itemId"><xsl:call-template name="compute-cartitem-id"/></xsl:variable>
                                
                                function _updateLink(answer)
                                {
                                    $j.HSCore.components.HSToggleFavorite.init('#course-<xsl:value-of select="@code"/>-add-to-wishlist-<xsl:value-of select="generate-id()"/>', {
                                        isInWishList: answer,
                                        itemId: "<xsl:value-of select="$itemId"/>",
                                        itemTitle: "<xsl:value-of select="escaper:escapeJavaScript(@title)"/>",
                                        callback: function() {}
                                    });
                                }
                                
                                ODFCart.hasItem('<xsl:value-of select="$itemId"/>', _updateLink);
                            }
                        });
                    </script>
                </xsl:when>
                <xsl:otherwise>
                    <a href="{resolver:resolve('odf', concat(@id, $parentCourseId))}">
                        <xsl:value-of select="@title"/>
                    </a>
                </xsl:otherwise>
            </xsl:choose>
        </h2>
    </xsl:template>
    
    <xsl:template match="course" mode="course-row-ects">
        <span class="ametys-accordion__item-text">
            <xsl:variable name="computedEcts" select="odf:getEcts(@id, @path)"/>
            <xsl:if test="$computedEcts">
                <xsl:variable name="ects"><xsl:call-template name="transform-double"><xsl:with-param name="value" select="$computedEcts"/></xsl:call-template></xsl:variable>
                <xsl:value-of select="$ects"/> <i18n:text i18n:key="PROGRAM_ECTS_UNITS" i18n:catalogue="skin.{$skin}"/>
            </xsl:if>
        </span>
    </xsl:template>
    
    <xsl:template match="course" mode="course-row-hours">
        <span class="ametys-accordion__item-badge">
            <xsl:if test="local-name(../..) != 'course' and attributes/nbHours">
                <xsl:attribute name="class">ametys-accordion__item-badge ametys-accordion__item-badge--total</xsl:attribute>
            </xsl:if>
		    <xsl:if test="attributes/nbHours">
		        <xsl:variable name="hours"><xsl:call-template name="transform-double"><xsl:with-param name="value" select="attributes/nbHours"/></xsl:call-template></xsl:variable>
		        <xsl:value-of select="$hours"/>h
		    </xsl:if>
        </span>
    </xsl:template>
    
</xsl:stylesheet>