/*
 *  Copyright 2013 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * This class is the abstract class for all explorer application.
 * Create your own Application class by inheriting this one and define at least the following methods: #getToolId, #getToolParams.
 * 
 * 		Ext.define("My.Application", {
 * 			extend: "Ametys.explorer.applications.Application",
 * 
 * 			getToolId: function() {
 * 				return "tool-id";
 * 			},
 * 
 * 			getToolParams: function(node) {
 * 				return {
					id: node.getId()
				};
 * 			}
 * 		});
 * 
 * @private
 */
Ext.define('Ametys.explorer.applications.ExplorerApplication', {
	
	config: {
		/**
		 * @cfg {String} id (required) The unique id of the application. Cannot be null.
		 */
		id: null,
		
		/**
		 * @cfg {String} pluginName (required) The name of the plugin that declared the application. Cannot be null.
		 */
		pluginName: null
	},
	
	constructor: function(config)
	{
		this.initConfig(config);
		this.callParent(arguments);
	},
	
	/**
	 * Retrieves the value of an application parameter
	 * @param {String} name the name of the parameter
	 * @return {String} The value of the application parameter
	 */
	getApplicationParameter: function(name)
	{
		return this.getInitialConfig()[name];
	},
	
	/**
	 * Inherit this method to return the tool id for this application
	 * @return {String} The tool id
	 */
	getToolId: function()
	{
		throw new Error("Method #getToolId is not implemented on your Ametys.explorer.applications.ExplorerApplication");
	},
	

	/**
	 * Inherit this method to return the tool params for this application
	 * @param {Ametys.explorer.tree.ExplorerTree.NodeEntry} node The node from which the tool params must be extracted 
	 * @return {Object} node The tool params object (can be empty but no null)
	 */
	getToolParams: function(node) 
	{
		throw new Error("Method #getToolParams is not implemented on your Ametys.explorer.applications.ExplorerApplication");
	},
	
	/**
	 * Inherit this method to return the message target id for this application
	 * @return {Object} node The tool params object (can be empty but no null)
	 */
	getMessageTargetId: function()
	{
		throw new Error("Method #getMessageTargetId is not implemented on your Ametys.explorer.applications.ExplorerApplication");
	},
	
	/**
	 * Inherit this method to return the name of the class for the explorer node bound to this application
	 * @return {String} The name of the class
	 */
	getExplorerNodeClassName: function()
	{
		throw new Error("Method #getExplorerNodeClassName is not implemented on your Ametys.explorer.applications.ExplorerApplication");
	},
	
	/**
	 * Inherit this method to return the right identifier for renaming a node of this application
	 * @return {String} The right id
	 */
	getRightIdOnRename: function()
	{
		throw new Error("Method #getRightIdOnRename is not implemented on your Ametys.explorer.applications.ExplorerApplication");
	},
	
	/**
	 * Inherit this method to return the right identifier for dropping a node of this application
	 * @return {String} The right id
	 */
	getRightIdOnDrop: function()
	{
		throw new Error("Method #getRightIdOnDrop is not implemented on your Ametys.explorer.applications.ExplorerApplication");
		
	},
	
	/**
	 * Inherit this method to return the right identifier for dragging a node of this application
	 * @return {String} The right id
	 */
	getRightIdOnDrag: function()
	{
		throw new Error("Method #getRightIdOnDrag is not implemented on your Ametys.explorer.applications.ExplorerApplication");
	},
	
	/**
	 * @protected
	 * @template
	 * Determines if the node application can be renamed. Returns false by default.
	 * Override this method to allow the renaming.
	 * @param {Ext.data.Model} node The node
	 * @return {Boolean} true if the node can be renamed
	 */
	canRenameNode: function(node) 
	{
		return false;
	},
	
	/**
	 * @protected
	 * @template
	 * Determines if the node application can be moved. Returns false by default.
	 * Override this method to allow the move.
	 * @param {Ext.data.Model} node The node
	 * @param {Ext.data.Model} targetNode The target node to move into.
	 * @return {Boolean} true if the node can be renamed
	 */
	canMoveNode: function(node, targetNode) 
	{
		return false;
	},
	
	/**
	 * Determines if a file can be renamed.
	 * @param {String} path The resource path in the explorer
	 * @return {Boolean} true if the file can be renamed.
	 */
	canRenameFile: function(path)
	{
		return false;
	},
	
	/**
	 * Must provide the object which hold the rename action.
	 * Rename action is mandatory as long as canRenameNode return value is not always false.
	 * The rename action parameters are :
	 * @return {Function} The object holding the rename action. Expected parameters for the rename function are :
	 * - {String} id The explorer node id to rename
	 * - {String} name The new name
	 * - {Function} callback The callback function called when the explorer node has been renamed
	 * - {String} callback.id The id of the renamed explorer node.
	 * - {String} callback.name The new name
	 * - {Object} scope the callback scope
	 */
	getRenameActionHolder: function()
	{
		throw new Error("Method #getRenameActionHolder is not implemented on your Ametys.explorer.applications.ExplorerApplication");
	},
	
	/**
	 * Template method to allow specific processing when updating a node of the tree
	 * @param {Ext.data.Model} explorerNode The explorer node of the application (of the class returned by #method-getExplorerNodeClassName)
	 * @param {Ext.data.Model} treeNode The node of the explorer tree to update
	 */
	updateNodeUISpecific: function(explorerNode, treeNode)
	{
		// Nothing by default
	}
});
