/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * This class controls a ribbon button representing a node of resource explorer
 * @private
 */
Ext.define('Ametys.explorer.controllers.ExplorerNodeController', {
	extend: 'Ametys.ribbon.element.ui.ButtonController',
	
	inheritableStatics: {
		/**
		 * @private
		 * @readonly
		 * @property {RegExp} _targetFilter The regexp that match explorer related target type.
		 */
		_targetFilter : /^(explorer-collection|resource)$/
	},

	/**
	 * @cfg {Boolean} [enabled-on-modifiable-only=false] If true the controller will be disabled when the selection is not modifiable
	 */
	/**
	 * @cfg {String} no-modifiable-description The description text used for tooltip when the selection is not modifiable
	 */
	
	/**
	 * @cfg {Boolean} [enabled-on-traversable-only=false] If true the controller will be disabled when the selection is not traversable (ie. cannot have children)
	 */
	/**
	 * @cfg {String} no-traversable-description The description text used for tooltip when the selection is not traversable
	 */
	
	/**
	 * @cfg {Boolean} [disabled-on-root=false] If true the controller will be disabled if the selection is one of the root node
	 */
	
	/**
	 * @cfg {String} [application-id] The id of the application the selection should match. If the selection does not match the application id, the controller will be disabled.
	 * Assume that the parameters of the message target contains the id of the application it belongs to.
	 */
	
	additionalErrorDescriptionOnSelectionChanged: function(targets)
	{
		// Search for the first explorer target (explorer node or resource).
		var target = Ametys.message.MessageTargetHelper.findTarget(targets, this.self._targetFilter);
		
		if (!target)
		{
			// Should not happens, but prevents a possible null error.
			return "";
		}
		
		if (String(this.getInitialConfig("enabled-on-modifiable-only")) == 'true' && !target.getParameters().isModifiable)
		{
			return this.getInitialConfig('no-modifiable-description') || '';
		}
		else if (String(this.getInitialConfig("enabled-on-traversable-only")) == 'true' && !target.getParameters().canCreateChild)
		{
			return this.getInitialConfig('no-traversable-description') || '';
		}
		else if (String(this.getInitialConfig("disabled-on-root")) == 'true' && this._isRoot(Ametys.message.MessageBus.getCurrentSelectionMessage(), target))
		{
			return "";
		}
		else if (this.getInitialConfig('application-id') && target.getParameters().applicationId != this.getInitialConfig('application-id'))
		{
			return "";
		}
	},
	
	/**
	 * @private
	 * Determines if the target represents a root explorer node.
	 * @param {Ametys.message.Message} message The selection message. 
	 * @param {Ametys.message.MessageTarget} target The target
	 */
	_isRoot: function(message, target)
	{
		return target.getParameters().rootId == target.getParameters().id;
	},
	
	/**
	 * @inheritdoc
	 */
	areSameTargets: function (target1, target2)
	{
		// This function filters targets of type explorer and then compares each identifiers.
		
		var explorerTargets1 = Ametys.message.MessageTargetHelper.findTargets([target1], this.self._targetFilter);
		var explorerTargets2 = Ametys.message.MessageTargetHelper.findTargets([target2], this.self._targetFilter);
		
		if (explorerTargets1.length != explorerTargets2.length)
		{
			return false;
		}
			
		var targetToIdConverter = function(target) {return target.getParameters().id;}
		explorerTargets1 = Ext.Array.map(explorerTargets1, targetToIdConverter);
		explorerTargets2 = Ext.Array.map(explorerTargets2, targetToIdConverter);
		
		var diff = Ext.Array.difference(explorerTargets1, explorerTargets2);
		return diff.length === 0;
	}
});
